#ifndef __DALI_LAYER_H__
#define __DALI_LAYER_H__

/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

// INTERNAL INCLUDES
#include <dali/public-api/object/ref-object.h>
#include <dali/public-api/actors/actor.h>
#include <dali/public-api/math/rect.h>
#include <dali/public-api/math/vector3.h>
#include <dali/public-api/images/frame-buffer-image.h>

namespace Dali
{
/**
 * @addtogroup dali_core_actors
 * @{
 */

namespace Internal DALI_INTERNAL
{
class Layer;
}

/**
 * @brief Rectangle describing area on screen that a layer can draw to
 *
 * @since_tizen 2.4
 * @see Dali::Layer::SetClippingBox()
 */
typedef Rect<int> ClippingBox;

/**
 * @brief Layers provide a mechanism for overlaying groups of actors on top of each other.
 *
 * When added to the stage, a layer can be ordered relative to other layers. The bottom
 * layer is at depth zero. The stage provides a default layer for it's children (see Stage::GetRootLayer()).
 *
 * Layered actors inherit position etc. as normal, but are drawn in an order determined
 * by the layers. In case of LAYER_3D, the depth buffer is cleared before each layer is rendered unless depth
 * test is disabled or there's no need for it based on the layers contents;
 * actors in lower layers cannot obscure actors in higher layers.
 *
 * A layer has either LAYER_2D or LAYER_3D mode. LAYER_2D has better performance,
 * the depth test is disabled, and a child actor hides its parent actor.
 * LAYER_3D uses the depth test, thus a close actor hides a farther one.
 * LAYER_2D is the default mode and recommended for general cases.
 * See Layer::Behavior and SetBehavior() for more information.
 *
 * Layer is a type of Actor, thus can have parent or children actors.
 * A layer influences rendering of its all descendant actors,
 * until another layer appears in the actor tree and manages its own subtree.
 *
 * If depth test is disabled, there is no performance overhead from clearing the depth buffer.
 *
 * @since_tizen 2.4
 */
class DALI_IMPORT_API Layer : public Actor
{
public:

  /**
   * @brief An enumeration of properties belonging to the Layer class.
   *
   * Properties additional to Actor.
   *
   * @since_tizen 2.4
   */
  struct Property
  {
    enum
    {
      CLIPPING_ENABLE = DEFAULT_DERIVED_ACTOR_PROPERTY_START_INDEX, ///< type bool @since_tizen 2.4
      CLIPPING_BOX,                                                 ///< type Rect<int> @since_tizen 2.4
      BEHAVIOR,                                                     ///< type String @since_tizen 2.4
    };
  };

  /**
   * @brief Enumeration for the behavior of the layer.
   *
   * Check each value to see how it affects the layer.
   * @since_tizen 2.4
   */
  enum Behavior
  {
    /**
     * @brief Layer doesn't make use of the depth test (default mode).
     *
     * This mode is expected to have better performance than the 3D mode.
     * When using this mode any ordering would be with respect to tree level of each Actor.
     *
     * For the following actor tree of the Layer1 object, D and E hide B, B and C hides A,
     * and F hides C, regardless of their Z positions.
     * Rendering order between siblings, such as D & E or B & C, is not determined.
     * If you have two overlapped actors, just make them parent-child, not siblings.
     *
     * @code
     *
     *     Layer1 (parent)
     *       |
     *       A
     *      / \
     *     B   C
     *    / \   \
     *   D   E   F
     *
     * @endcode
     *
     * @since_tizen 2.4
     */
    LAYER_2D,

    /**
     * @brief Layer will use depth test and do several clears.
     *
     * When using this mode depth test will be used. A depth clear will happen for each layer,
     * which means actors in a layer "above" other layers will be rendered in front of actors in
     * those layers regardless of their Z positions (see Layer::Raise() and Layer::Lower()).
     * Opaque renderers are drawn first and write to the depth buffer.
     * Then transparent renderers are drawn with depth test enabled but depth write switched off.
     * Unlike LAYER_2D, parent-child relationship does not affect rendering order at all.
     *
     * @since_tizen 2.4
     * @remarks This is an experimental feature. Using 2D UI components of DALi Toolkit
     * in LAYER_3D mode has not been enoughly tested yet
     * because they are orginally designed for 2D use cases.
     * Simple controls such as Toolkit::Control or Toolkit::ImageView might not have any problem with LAYER_3D,
     * but more complex one like Toolkit::PushButton, you might get unexpected rendered order in LAYER_3D.
     * Although we'll support 2D controls in LAYER_3D soon, we recommend to use 2D controls with LAYER_2D only at this moment.
     * Of course, controls rendered in 3D space, such as SpiralLayout of Toolkit::ItemView
     * (see Toolkit::DefaultItemLayout::New), should be used with LAYER_3D.
     */
    LAYER_3D,
  };

  /**
   * @brief TREE_DEPTH_MULTIPLIER is used by the rendering sorting algorithm to decide which actors to render first.
   * @since_tizen 2.4
   */
  enum TreeDepthMultiplier
  {
    TREE_DEPTH_MULTIPLIER = 10000,
  };
  /**
   * @brief The sort function type
   *
   * @since_tizen 2.4
   * @param[in] position This is the actor translation from camera.
   */
  typedef float (*SortFunctionType)( const Vector3& position );

  /**
   * @brief Create an empty Layer handle.
   *
   * This can be initialised with Layer::New(...).
   * @since_tizen 2.4
   */
  Layer();

  /**
   * @brief Create a Layer object.
   *
   * @since_tizen 2.4
   * @return A handle to a newly allocated Layer
   */
  static Layer New();

  /**
   * @brief Downcast a handle to Layer handle.
   *
   * If handle points to a Layer the downcast produces valid
   * handle. If not the returned handle is left uninitialized.
   * @since_tizen 2.4
   * @param[in] handle Handle to An object
   * @return Handle to a Layer or an uninitialized handle
   */
  static Layer DownCast( BaseHandle handle );

  /**
   * @brief Destructor
   *
   * This is non-virtual since derived Handle types must not contain data or virtual methods.
   * @since_tizen 2.4
   */
  ~Layer();

  /**
   * @brief Copy constructor
   *
   * @since_tizen 2.4
   * @param [in] copy The actor to copy
   */
  Layer(const Layer& copy);

  /**
   * @brief Assignment operator
   *
   * @since_tizen 2.4
   * @param [in] rhs The actor to copy
   * @return A reference to this
   */
  Layer& operator=(const Layer& rhs);

  /**
   * @brief Query the depth of the layer
   *
   * 0 is bottom most layer, higher number is on top.
   * @since_tizen 2.4
   * @return The current depth of the layer
   * @pre Layer is on the stage.
   * If layer is not added to the stage, returns 0.
   */
  unsigned int GetDepth() const;

  /**
   * @brief Increment the depth of the layer.
   *
   * @since_tizen 2.4
   * @pre Layer is on the stage.
   */
  void Raise();

  /**
   * @brief Decrement the depth of the layer.
   *
   * @since_tizen 2.4
   * @pre Layer is on the stage.
   */
  void Lower();

  /**
   * @brief Ensures the layers depth is greater than the target layer.
   *
   * If the layer already is above the target layer its depth is not changed.
   * If the layer was below target, its new depth will be immediately above target.
   * @since_tizen 2.4
   * @param target Layer to get above of
   * @pre Layer is on the stage.
   * @pre Target layer is on the stage.
   * @note All layers between this layer and target get new depth values.
   */
  void RaiseAbove( Layer target );

  /**
   * @brief Ensures the layers depth is less than the target layer.
   *
   * If the layer already is below the target layer its depth is not changed.
   * If the layer was above target, its new depth will be immediately below target.
   * @since_tizen 2.4
   * @param target Layer to get below of
   * @pre Layer is on the stage.
   * @pre Target layer is on the stage.
   * @note All layers between this layer and target get new depth values.
   */
  void LowerBelow( Layer target );

  /**
   * @brief Raises the layer to the top.
   * @since_tizen 2.4
   * @pre Layer is on the stage.
   */
  void RaiseToTop();

  /**
   * @brief Lowers the layer to the bottom.
   * @since_tizen 2.4
   * @pre layer is on the stage.
   */
  void LowerToBottom();

  /**
   * @brief Moves the layer directly above the given layer.
   *
   * After the call this layers depth will be immediately above target.
   * @since_tizen 2.4
   * @param target Layer to get on top of
   * @pre Layer is on the stage.
   * @pre Target layer is on the stage.
   * @note All layers between this layer and target get new depth values.
   */
  void MoveAbove( Layer target );

  /**
   * @brief Moves the layer directly below the given layer.
   *
   * After the call this layers depth will be immediately below target.
   * @since_tizen 2.4
   * @param target Layer to get below of
   * @pre Layer is on the stage.
   * @pre Target layer is on the stage.
   * @note All layers between this layer and target get new depth values.
   */
  void MoveBelow( Layer target );

  /**
   * @brief Set the behavior of the layer.
   *
   * @since_tizen 2.4
   * @param[in] behavior The desired behavior
   */
  void SetBehavior( Behavior behavior );

  /**
   * @brief Get the behavior of the layer.
   *
   * @since_tizen 2.4
   * @return The behavior of the layer
   */
  Behavior GetBehavior() const;

  /**
   * @brief Sets whether clipping is enabled for a layer.
   *
   * Clipping is initially disabled; see also SetClippingBox().
   * @since_tizen 2.4
   * @param [in] enabled True if clipping is enabled.
   *
   * @note When clipping is enabled, the default clipping box is empty (0,0,0,0) which means everything is clipped.
   */
  void SetClipping(bool enabled);

  /**
   * @brief Query whether clipping is enabled for a layer.
   * @since_tizen 2.4
   * @return True if clipping is enabled.
   */
  bool IsClipping() const;

  /**
   * @brief Sets the clipping box of a layer, in window coordinates.
   *
   * The contents of the layer will not be visible outside this box, when clipping is
   * enabled. The default clipping box is empty (0,0,0,0) which means everything is clipped.
   * You can only do rectangular clipping using this API in window coordinates.
   * For other kinds of clipping, see Dali::Actor::SetDrawMode().
   * @since_tizen 2.4
   * @param [in] x The X-coordinate of the top-left corner of the box
   * @param [in] y The Y-coordinate of the top-left corner of the box
   * @param [in] width  The width of the box
   * @param [in] height The height of the box
   */
  void SetClippingBox(int x, int y, int width, int height);

  /**
   * @brief Sets the clipping box of a layer, in window coordinates.
   *
   * The contents of the layer will not be visible outside this box, when clipping is
   * enabled. The default clipping box is empty (0,0,0,0).
   * @since_tizen 2.4
   * @param [in] box The clipping box
   */
  void SetClippingBox(ClippingBox box);

  /**
   * @brief Retrieves the clipping box of a layer, in window coordinates.
   *
   * @since_tizen 2.4
   * @return The clipping box
   */
  ClippingBox GetClippingBox() const;

  // Depth test

  /**
   * @brief Whether to disable the depth test.
   *
   * By default a layer enables depth test if there is more than one opaque actor or if there is one opaque actor and one, or more, transparent actors in LAYER_3D mode.
   * However, it's possible to disable the depth test by calling this method.
   *
   * @since_tizen 2.4
   * @param[in] disable \e True disables depth test. \e false sets the default behavior.
   */
  void SetDepthTestDisabled( bool disable );

  /**
   * @brief Retrieves whether depth test is disabled.
   *
   * @since_tizen 2.4
   * @return \e True if depth test is disabled.
   */
  bool IsDepthTestDisabled() const;

  // Sorting

  /**
   * @brief This allows the user to specify the sort function that the layer should use.
   *
   * The sort function is used to determine the order in which the actors are drawn
   * and input is processed on the actors in the layer.
   *
   * A function of the following type should be used:
   * @code
   *  float YourSortFunction(const Vector3& position);
   * @endcode
   *
   * @since_tizen 2.4
   * @param[in]  function  The sort function pointer
   * @note If the sort function returns a low number, the actor with the data will be
   * drawn in front of an actor whose data yields a high value from the sort function.
   *
   * @note All child layers use the same sort function.  If a child layer is added to this
   * layer then the sort function used by the child layer will also be the same.
   *
  */
  void SetSortFunction( SortFunctionType function );

  /**
   * @brief This allows the user to specify whether this layer should consume touch (including gestures).
   *
   * If set, any layers behind this layer will not be hit-test.
   *
   * @since_tizen 2.4
   * @param[in]  consume  Whether the layer should consume touch (including gestures).
   */
  void SetTouchConsumed( bool consume );

  /**
   * @brief Retrieves whether the layer consumes touch (including gestures).
   *
   * @since_tizen 2.4
   * @return True if consuming touch, false otherwise.
   */
  bool IsTouchConsumed() const;

  /**
   * @brief This allows the user to specify whether this layer should consume hover.
   *
   * If set, any layers behind this layer will not be hit-test.
   *
   * @since_tizen 2.4
   * @param[in]  consume  Whether the layer should consume hover.
   */
  void SetHoverConsumed( bool consume );

  /**
   * @brief Retrieves whether the layer consumes hover.
   *
   * @since_tizen 2.4
   * @return True if consuming hover, false otherwise.
   */
  bool IsHoverConsumed() const;

public: // Not intended for application developers

  /**
   * @brief This constructor is used by Layer::New() methods.
   *
   * @since_tizen 2.4
   * @param [in] Layer A pointer to a newly allocated Dali resource
   */
  explicit DALI_INTERNAL Layer(Internal::Layer* Layer);
};

/**
 * @}
 */
} // namespace Dali

#endif //__DALI_LAYER_H__
