/*
* Copyright (c) 2011 Samsung Electronics Co., Ltd All Rights Reserved
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
* http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/


#ifndef __TIZEN_CONTENT_MEDIA_CONTENT_TYPE_H__
#define __TIZEN_CONTENT_MEDIA_CONTENT_TYPE_H__

#include <time.h>
#include <tizen.h>

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/**
 * @file media_content_type.h
 * @brief This file contains API related to media-content enumerations for media data types, groups, orientations, \n
 *        classes of errors and definitions of media-data. \n
 *        Listed APIs are called when iterating over lists of album, group, bookmark and other media,  \n
 *        when media items and burst shot are inserted completely and when notification of media DB change is subscribed.
 */

/**
* @addtogroup CAPI_MEDIA_CONTENT_MODULE
 * @{
 */

/**
 * @ingroup CAPI_MEDIA_CONTENT_MODULE
 * @brief Enumeration for the media file format.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 * @remarks Since 4.0, #MEDIA_CONTENT_TYPE_OTHERS is related to the following feature:\n
 *              %http://tizen.org/feature/content.scanning.others\n
 *              If this feature is not suppported on the device, MEDIA_CONTENT_TYPE_OTHERS type file is not scanned.
 */
typedef enum {
	MEDIA_CONTENT_TYPE_IMAGE		= 0,	/**<The type of an image */
	MEDIA_CONTENT_TYPE_VIDEO		= 1,	/**<The type of a video */
	MEDIA_CONTENT_TYPE_SOUND	= 2,	/**<The type of sound */
	MEDIA_CONTENT_TYPE_MUSIC		= 3,	/**<The type of music */
	MEDIA_CONTENT_TYPE_OTHERS	= 4,	/**<The type of other */
} media_content_type_e;

/**
 * @ingroup CAPI_CONTENT_MEDIA_CONTENT_MODULE
 * @brief Enumeration for the storage type.
 * @details This information is used to establish where the folder is.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 */
typedef enum {
	MEDIA_CONTENT_STORAGE_INTERNAL	= 0,	/**< The device's internal storage */
	MEDIA_CONTENT_STORAGE_EXTERNAL	= 1,	/**< The device's external storage like sd card*/
	MEDIA_CONTENT_STORAGE_EXTERNAL_USB = 2, /**< The external USB storage (Since @if MOBILE 2.4 @elseif WEARABLE 3.0 @endif) */
	MEDIA_CONTENT_STORAGE_CLOUD	= 100,	/**< The Cloud storage (Since @if MOBILE 2.4 @elseif WEARABLE 3.0 @endif, Deprecated since 4.0) */
} media_content_storage_e;

/**
 * @ingroup CAPI_MEDIA_CONTENT_MODULE
 * @brief Enumeration for media content DB update items.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 */
typedef enum {
	MEDIA_ITEM_FILE		= 0,		/**< File type, an item updated to DB */
	MEDIA_ITEM_DIRECTORY	= 1,		/**< Directory type, an item updated to DB */
} media_content_db_update_item_type_e;

/**
 * @ingroup CAPI_MEDIA_CONTENT_MODULE
 * @brief Enumeration for media content DB update types.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 */
typedef enum {
	MEDIA_CONTENT_INSERT		= 0,		/**< Insert, the type of DB update */
	MEDIA_CONTENT_DELETE		= 1,		/**< Delete, The type of DB update */
	MEDIA_CONTENT_UPDATE	= 2,		/**< Update, The type of DB update */
} media_content_db_update_type_e;

/**
 * @ingroup CAPI_CONTENT_MEDIA_INFO_MODULE
 * @brief Enumeration for orientation types.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 */
typedef enum {
	MEDIA_CONTENT_ORIENTATION_NOT_AVAILABLE  = 0,       /**< Not available*/
	MEDIA_CONTENT_ORIENTATION_NORMAL         = 1,       /**< Normal*/
	MEDIA_CONTENT_ORIENTATION_HFLIP          = 2,       /**< Flip horizontal*/
	MEDIA_CONTENT_ORIENTATION_ROT_180        = 3,       /**< Rotate 180 degrees*/
	MEDIA_CONTENT_ORIENTATION_VFLIP          = 4,       /**< Flip vertical*/
	MEDIA_CONTENT_ORIENTATION_TRANSPOSE      = 5,       /**< Transpose*/
	MEDIA_CONTENT_ORIENTATION_ROT_90         = 6,       /**< Rotate 90 degrees*/
	MEDIA_CONTENT_ORIENTATION_TRANSVERSE     = 7,       /**< Transverse*/
	MEDIA_CONTENT_ORIENTATION_ROT_270        = 8,       /**< Rotate 270 degrees*/
} media_content_orientation_e;

/**
 * @ingroup CAPI_MEDIA_CONTENT_MODULE
 * @brief Enumeration for ordering.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 */
typedef enum {
	MEDIA_CONTENT_ORDER_ASC   = 0,      /**< Ascending order*/
	MEDIA_CONTENT_ORDER_DESC  = 1,      /**< Descending order*/
	MEDIA_CONTENT_ORDER_OTHER = 2,		/**< order by order key**/
} media_content_order_e;

/**
 * @ingroup CAPI_MEDIA_CONTENT_MODULE
 * @brief Enumeration for collations.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 */
typedef enum {
	MEDIA_CONTENT_COLLATE_DEFAULT   = 0,        /**< Default collation BINARY */
	MEDIA_CONTENT_COLLATE_NOCASE    = 1,        /**< Collation NOCASE, not case sensitive */
	MEDIA_CONTENT_COLLATE_RTRIM     = 2,        /**< Collation RTRIM, trailing space characters are ignored */
	MEDIA_CONTENT_COLLATE_LOCALIZED = 3,        /**< Collation LOCALIZATION, NOCASE also applied */
} media_content_collation_e;

/**
* @brief Error class.
* @details Class for Media Content error
* @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
*
*/
#define MEDIA_CONTENT_ERROR_CLASS				TIZEN_ERROR_MEDIA_CONTENT

/**
 * @ingroup CAPI_MEDIA_CONTENT_MODULE
 * @brief Enumeration for a media content error.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 */
typedef enum {
	MEDIA_CONTENT_ERROR_NONE                    = TIZEN_ERROR_NONE,                    /**< Successful */
	MEDIA_CONTENT_ERROR_INVALID_PARAMETER       = TIZEN_ERROR_INVALID_PARAMETER,       /**< Invalid parameter */
	MEDIA_CONTENT_ERROR_OUT_OF_MEMORY           = TIZEN_ERROR_OUT_OF_MEMORY,           /**< Out of memory */
	MEDIA_CONTENT_ERROR_INVALID_OPERATION       = TIZEN_ERROR_INVALID_OPERATION,       /**< Invalid Operation */
	MEDIA_CONTENT_FILE_NO_SPACE_ON_DEVICE       = TIZEN_ERROR_FILE_NO_SPACE_ON_DEVICE, /**< No space left on device */
	MEDIA_CONTENT_ERROR_PERMISSION_DENIED		  = TIZEN_ERROR_PERMISSION_DENIED,		  /**< Permission denied */
	MEDIA_CONTENT_ERROR_DB_FAILED               = MEDIA_CONTENT_ERROR_CLASS | 0x01,    /**< DB operation failed */
	MEDIA_CONTENT_ERROR_DB_BUSY                 = MEDIA_CONTENT_ERROR_CLASS | 0x02,    /**< DB operation BUSY */
	MEDIA_CONTENT_ERROR_NETWORK                 = MEDIA_CONTENT_ERROR_CLASS | 0x03,    /**< Network Fail */
	MEDIA_CONTENT_ERROR_UNSUPPORTED_CONTENT     = MEDIA_CONTENT_ERROR_CLASS | 0x04,    /**< Unsupported Content */
	MEDIA_CONTENT_ERROR_NOT_SUPPORTED           = TIZEN_ERROR_NOT_SUPPORTED,           /**< Not supported */
} media_content_error_e;

/**
 * @ingroup CAPI_MEDIA_CONTENT_MODULE
 * @brief Enumeration for a media group.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 */
typedef enum {
	MEDIA_CONTENT_GROUP_DISPLAY_NAME = 0,    /**< Media group ID for display name */
	MEDIA_CONTENT_GROUP_TYPE,                /**< Media group ID for a media type */
	MEDIA_CONTENT_GROUP_MIME_TYPE,           /**< Media group ID for a mime type */
	MEDIA_CONTENT_GROUP_SIZE,                /**< Media group ID for content size */
	MEDIA_CONTENT_GROUP_ADDED_TIME,          /**< Media group ID for the added time */
	MEDIA_CONTENT_GROUP_MODIFIED_TIME,       /**< Media group ID for the modified time */
	MEDIA_CONTENT_GROUP_TITLE,               /**< Media group ID for a content title */
	MEDIA_CONTENT_GROUP_ARTIST,              /**< Media group ID for an artist*/
	MEDIA_CONTENT_GROUP_ALBUM_ARTIST,        /**< Media group ID for an album artist */
	MEDIA_CONTENT_GROUP_GENRE,               /**< Media group ID for a genre*/
	MEDIA_CONTENT_GROUP_COMPOSER,            /**< Media group ID for a composer*/
	MEDIA_CONTENT_GROUP_YEAR,                /**< Media group ID for a year*/
	MEDIA_CONTENT_GROUP_RECORDED_DATE,       /**< Media group ID for the recorded date*/
	MEDIA_CONTENT_GROUP_COPYRIGHT,           /**< Media group ID for the copyright*/
	MEDIA_CONTENT_GROUP_TRACK_NUM,           /**< Media group ID for a track number*/
	MEDIA_CONTENT_GROUP_DESCRIPTION,         /**< Media group ID for a description */
	MEDIA_CONTENT_GROUP_LONGITUDE,           /**< Media group ID for the longitude*/
	MEDIA_CONTENT_GROUP_LATITUDE,            /**< Media group ID for the latitude*/
	MEDIA_CONTENT_GROUP_ALTITUDE,            /**< Media group ID for the altitude*/
	MEDIA_CONTENT_GROUP_BURST_IMAGE,         /**< Media group ID for the burst shot (Deprecated since 4.0)*/
	MEDIA_CONTENT_GROUP_RATING,              /**< Media group ID for a rating*/
	MEDIA_CONTENT_GROUP_AUTHOR,              /**< Media group ID for an author (Deprecated since 4.0)*/
	MEDIA_CONTENT_GROUP_PROVIDER,            /**< Media group ID for a provider (Deprecated since 4.0)*/
	MEDIA_CONTENT_GROUP_CONTENT_NAME,        /**< Media group ID for the content name (Deprecated since 4.0)*/
	MEDIA_CONTENT_GROUP_CATEGORY,            /**< Media group ID for a category (Deprecated since 4.0)*/
	MEDIA_CONTENT_GROUP_LOCATION_TAG,        /**< Media group ID for a location tag (Deprecated since 4.0)*/
	MEDIA_CONTENT_GROUP_AGE_RATING,          /**< Media group ID for an age rating (Deprecated since 4.0)*/
	MEDIA_CONTENT_GROUP_KEYWORD,             /**< Media group ID for a keyword (Deprecated since 4.0)*/
	MEDIA_CONTENT_GROUP_WEATHER,             /**< Media group ID for the weather (Deprecated since 4.0)*/
	MEDIA_CONTENT_GROUP_MAX                  /**< Invalid media group ID*/
} media_group_e;

/**
 * @ingroup CAPI_CONTENT_MEDIA_INFO_MODULE
 * @brief The structure type for the Media info handle.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 */
typedef struct media_info_s *media_info_h;

/**
 * @ingroup CAPI_CONTENT_MEDIA_FOLDER_MODULE
 * @brief The structure type for the Media folder handle.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 */
typedef struct media_folder_s *media_folder_h;

/**
 * @ingroup CAPI_CONTENT_MEDIA_PLAYLIST_MODULE
 * @brief The structure type for the Media playlist handle.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 */
typedef struct media_playlist_s *media_playlist_h;

/**
 * @ingroup CAPI_CONTENT_MEDIA_TAG_MODULE
 * @brief The structure type for the Media tag handle.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 */
typedef struct media_tag_s *media_tag_h;

/**
 * @ingroup CAPI_CONTENT_MEDIA_BOOKMARK_MODULE
 * @brief The structure type for the Media bookmark handle.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 */
typedef struct media_bookmark_s *media_bookmark_h;

/**
 * @ingroup CAPI_CONTENT_MEDIA_ALBUM_MODULE
 * @brief The structure type for the Media album handle.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 */
typedef struct media_album_s *media_album_h;

/**
 * @ingroup CAPI_CONTENT_MEDIA_IMAGE_MODULE
 * @brief The structure type for the Image metadata handle.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 */
typedef struct image_meta_s *image_meta_h;

/**
 * @ingroup CAPI_CONTENT_MEDIA_VIDEO_META_MODULE
 * @brief The structure type for the Video metadata handle.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 */
typedef struct video_meta_s *video_meta_h;

/**
 * @ingroup CAPI_CONTENT_MEDIA_AUDIO_META_MODULE
 * @brief The structure type for the Audio metadata handle.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 */
typedef struct audio_meta_s *audio_meta_h;

/**
 * @ingroup CAPI_CONTENT_MEDIA_FILTER_MODULE
 * @brief The structure type for the Media filter handle.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 */
typedef struct filter_s *filter_h;

/**
 * @ingroup CAPI_CONTENT_MEDIA_STORAGE_MODULE
 * @brief The structure type for the Media storage handle.
 * @since_tizen @if MOBILE 2.4 @elseif WEARABLE 3.0 @endif
 */
typedef void *media_storage_h;

/**
 * @ingroup CAPI_CONTENT_MEDIA_FACE_MODULE
 * @brief The structure type for the Media face handle.
 * @since_tizen 3.0
 */
typedef void *media_face_h;

/**
 * @ingroup CAPI_CONTENT_MEDIA_CONTENT_MODULE
 * @brief The structure type for the Media content noti handle.
 * @since_tizen 3.0
 */
typedef void *media_content_noti_h;

/**
 * @ingroup CAPI_MEDIA_CONTENT_MODULE
 * @brief Called when the media scanning is finished.
 * @details The following error codes can be received: \n
 *             #MEDIA_CONTENT_ERROR_NONE : Success \n
 *             #MEDIA_CONTENT_ERROR_INVALID_PARAMETER : Invalid parameter \n
 *             #MEDIA_CONTENT_ERROR_INVALID_OPERATION : Invalid operation \n
 *             #MEDIA_CONTENT_ERROR_PERMISSION_DENIED : Permission denied \n
 *             #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY : Out of memory \n
 *             #MEDIA_CONTENT_ERROR_DB_FAILED : DB Operation failed \n
 *             #MEDIA_CONTENT_ERROR_DB_BUSY : DB Operation busy \n
 *             #MEDIA_CONTENT_ERROR_NETWORK : Network fail \n
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 * @remarks The callback is called in a separate thread(not in the main loop).
 *
 * @param[in] error The error code
 * @param[in] user_data The user data passed from the foreach function
 *
 * @pre media_content_scan_folder().
 * @see media_content_scan_folder()
 */
typedef void (*media_scan_completed_cb)(media_content_error_e error, void * user_data);

/**
 * @ingroup CAPI_MEDIA_CONTENT_MODULE
 * @brief Called when the notification of the media DB change is subscribed.
 * @details The following error codes can be received: \n
 *             #MEDIA_CONTENT_ERROR_NONE : Success \n
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @remarks The callback is called in a separate thread(not in the main loop).
 *
 * @param[in] error The error code
 * @param[in] pid The PID which publishes notification
 * @param[in] update_item The update item of notification
 * @param[in] update_type The update type of notification
 * @param[in] media_type The type of the media content (#media_content_type_e)
 * @param[in] uuid The UUID of media or directory, which is updated
 * @param[in] path The path of the media or directory
 * @param[in] mime_type The MIME of the media info
 * @param[in] user_data The user data passed from the foreach function
 *
 * @pre media_content_db_update_subscribe().
 * @see media_content_db_update_subscribe()
 */
typedef void (*media_content_db_update_cb)(
				media_content_error_e error,
				int pid,
				media_content_db_update_item_type_e update_item,
				media_content_db_update_type_e update_type,
				media_content_type_e media_type,
				char *uuid,
				char *path,
				char *mime_type,
				void *user_data);


/**
 * @ingroup CAPI_CONTENT_MEDIA_INFO_MODULE
 * @brief Called for every available media info.
 * @details Iterates over a list of media info.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @remarks To use the @a media outside this function, copy the handle with media_info_clone() function. \n
 *                  The callback is called in the main loop.
 *
 * @param[in] media The handle to the media info
 * @param[in] user_data The user data passed from the foreach function
 *
 * @return @c true to continue with the next iteration of the loop,
 *         otherwise @c false to break out of the loop
 *
 * @pre media_tag_foreach_media_from_db(), media_playlist_foreach_media_from_db(), media_genre_foreach_media_from_db(),
 *      media_info_foreach_media_from_db(), media_folder_foreach_media_from_db() will invoke this function.
 *
 * @see media_info_clone()
 * @see media_album_foreach_media_from_db()
 * @see media_playlist_foreach_media_from_db()
 * @see media_tag_foreach_media_from_db()
 * @see media_info_foreach_media_from_db()
 * @see media_folder_foreach_media_from_db()
 */
typedef bool (*media_info_cb)(media_info_h media, void *user_data);


/**
 * @ingroup CAPI_CONTENT_MEDIA_INFO_MODULE
 * @brief Called when media items are inserted completely.
 * @details The following error codes can be received: \n
 *             #MEDIA_CONTENT_ERROR_NONE : Success \n
 *             #MEDIA_CONTENT_ERROR_INVALID_PARAMETER : Invalid parameter \n
 *             #MEDIA_CONTENT_ERROR_INVALID_OPERATION : Invalid operation \n
 *             #MEDIA_CONTENT_ERROR_PERMISSION_DENIED : Permission denied \n
 *             #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY : Out of memory \n
 *             #MEDIA_CONTENT_ERROR_DB_FAILED : DB Operation failed \n
 *             #MEDIA_CONTENT_ERROR_DB_BUSY : DB Operation busy \n
 *             #MEDIA_CONTENT_ERROR_NETWORK : Network fail \n
 *             #MEDIA_CONTENT_ERROR_NOT_SUPPORTED : Not supported \n
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @remarks The callback is called in a separate thread(not in the main loop).
 *
 * @param[in] error The error code
 * @param[in] user_data The user data passed from the foreach function
 *
 * @pre media_info_insert_batch_to_db()
 * @see media_info_insert_batch_to_db()
 */
typedef void (*media_insert_completed_cb)(media_content_error_e error, void * user_data);


/**
 * @deprecated Deprecated since 4.0.
 * @ingroup CAPI_CONTENT_MEDIA_INFO_MODULE
 * @brief Called when the burst shot is inserted completely.
 * @details The following error codes can be received: \n
 *             #MEDIA_CONTENT_ERROR_NONE : Success \n
 *             #MEDIA_CONTENT_ERROR_INVALID_PARAMETER : Invalid parameter \n
 *             #MEDIA_CONTENT_ERROR_INVALID_OPERATION : Invalid operation \n
 *             #MEDIA_CONTENT_ERROR_PERMISSION_DENIED : Permission denied \n
 *             #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY : Out of memory \n
 *             #MEDIA_CONTENT_ERROR_DB_FAILED : DB Operation failed \n
 *             #MEDIA_CONTENT_ERROR_DB_BUSY : DB Operation busy \n
 *             #MEDIA_CONTENT_ERROR_NETWORK : Network fail \n
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @remarks The callback is called in a separate thread(not in the main loop).
 *
 * @param[in] error The error code
 * @param[in] user_data The user data passed from the foreach function
 *
 * @pre media_info_insert_burst_shot_to_db()
 * @see media_info_insert_burst_shot_to_db()
 *
 */
typedef void (*media_insert_burst_shot_completed_cb)(media_content_error_e error, void * user_data);


/**
 * @ingroup CAPI_CONTENT_MEDIA_INFO_MODULE
 * @brief Called when creating a thumbnail image.
 * @details This callback is called for completion of generating the thumbnail image.
 *         The following error codes can be delivered. \n
 *         #MEDIA_CONTENT_ERROR_NONE, \n
 *         #MEDIA_CONTENT_ERROR_INVALID_PARAMETER, \n
 *         #MEDIA_CONTENT_ERROR_INVALID_OPERATION, \n
 *         #MEDIA_CONTENT_ERROR_PERMISSION_DENIED, \n
 *         #MEDIA_CONTENT_ERROR_DB_FAILED, \n
 *         #MEDIA_CONTENT_ERROR_UNSUPPORTED_CONTENT
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @remarks The callback is called in a separate thread(not in the main loop).
 *
 * @param[in] error The error code
 * @param[in] path The path of the thumbnail which is generated
 * @param[in] user_data The user data passed from the foreach function
 *
 * @pre media_info_create_thumbnail()
 * @see media_info_create_thumbnail()
 */
typedef void (*media_thumbnail_completed_cb)(media_content_error_e error, const char *path, void *user_data);

/**
 * @ingroup CAPI_CONTENT_MEDIA_FACE_DETECTION_MODULE
 * @brief Called when face detection on the image is completed.
 *
 * @details The following error codes can be delivered. \n
 *         #MEDIA_CONTENT_ERROR_NONE, \n
 *         #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY, \n
 *         #MEDIA_CONTENT_ERROR_INVALID_OPERATION, \n
 *         #MEDIA_CONTENT_ERROR_DB_FAILED, \n
 *         #MEDIA_CONTENT_ERROR_DB_BUSY, \n
 *         #MEDIA_CONTENT_ERROR_UNSUPPORTED_CONTENT
 *
 * @since_tizen 3.0
 *
 * @remarks The callback is called in a separate thread(not in the main loop).
 *
 * @param[in] error The error code
 * @param[in] face_count The number of all detected faces
 * @param[in] user_data The user data passed from the foreach function
 *
 * @pre media_info_start_face_detection()
 * @see media_info_start_face_detection()
 */
typedef void (*media_face_detection_completed_cb)(media_content_error_e error, const int face_count, void *user_data);


/**
 * @ingroup CAPI_CONTENT_MEDIA_FOLDER_MODULE
 * @brief Called for every available media folder.
 * @details Iterates over a list of folders.
 *
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @remarks To use the @a folder outside this function, copy the handle with the media_folder_clone() function. \n
 *                  The callback is called in the main loop.
 *
 * @param[in] folder The handle to the media folder
 * @param[in] user_data The user data passed from the foreach function
 *
 * @return @c true to continue with the next iteration of the loop,
 *         otherwise @c false to break out of the loop
 *
 * @pre media_folder_foreach_folder_from_db() will invoke this function.
 * @see media_folder_clone()
 * @see media_folder_foreach_folder_from_db()
 */
typedef bool (*media_folder_cb)(media_folder_h folder, void *user_data);

/**
 * @ingroup CAPI_CONTENT_MEDIA_PLAYLIST_MODULE
 * @brief Called for every playlist in the obtained list of playlists.
 * @details Iterates over a playlist list.
 *
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @remarks To use the @a playlist outside this function, copy the handle with the media_playlist_clone() function. \n
 *                  The callback is called in the main loop.
 *
 * @param[in] playlist The handle to the media playlist
 * @param[in] user_data The user data passed from the foreach function
 *
 * @return @c true to continue with the next iteration of the loop,
 *         otherwise @c false to break out of the loop
 *
 * @pre media_playlist_foreach_playlist_from_db() will invoke this function.
 *
 * @see media_playlist_clone()
 * @see media_playlist_foreach_playlist_from_db()
 */
typedef bool (*media_playlist_cb)(media_playlist_h playlist, void *user_data);

/**
 * @ingroup CAPI_CONTENT_MEDIA_PLAYLIST_MODULE
 * @brief Called for every media info with playlist member ID in the obtained list of media info.
 * @details Iterates over playlist members.
 *
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @remarks To use the @a media outside this function, copy the handle with the media_info_clone() function. \n
 *                  The callback is called in the main loop.
 *
 * @param[in] playlist_member_id The ID of the playlist member
 * @param[in] media The handle to the media info
 * @param[in] user_data The user data passed from the foreach function
 *
 * @return @c true to continue with the next iteration of the loop,
 *         otherwise @c false to break out of the loop
 *
 * @pre media_playlist_foreach_media_from_db() will invoke this function.
 *
 * @see media_info_clone()
 * @see media_playlist_foreach_media_from_db()
 */
typedef bool (*playlist_member_cb)(int playlist_member_id, media_info_h media, void *user_data);

/**
 * @ingroup CAPI_CONTENT_MEDIA_TAG_MODULE
 * @brief Called for every tag in the obtained list of tags.
 * @details Iterates over a list of tags.
 *
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @remarks To use the @a tag outside this function, copy the handle with the media_tag_clone() function. \n
 *                  The callback is called in the main loop.
 *
 * @param[in] tag The handle to the media tag
 * @param[in] user_data The user data passed from the foreach function
 *
 * @return @c true to continue with the next iteration of the loop,
 *         otherwise @c false to break out of the loop
 *
 * @pre media_tag_foreach_tag_from_db(), media_info_foreach_tag_from_db() will invoke this function.
 *
 * @see media_tag_clone()
 * @see media_tag_foreach_tag_from_db()
 * @see media_info_foreach_tag_from_db()
 */
typedef bool (*media_tag_cb)(media_tag_h tag, void *user_data);

/**
 * @ingroup CAPI_CONTENT_MEDIA_BOOKMARK_MODULE
 * @brief Called for every bookmark in the obtained list of bookmarks.
 * @details Iterates over a bookmark list.
 *
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @remarks To use the @a bookmark outside this function, copy the handle with the media_bookmark_clone() function. \n
 *                  The callback is called in the main loop.
 *
 * @param[in] bookmark The handle to the media bookmark
 * @param[in] user_data The user data passed from the foreach function
 *
 * @return @c true to continue with the next iteration of the loop,
 *         otherwise @c false to break out of the loop
 *
 * @pre media_info_foreach_bookmark_from_db() will invoke this function.
 * @see media_info_foreach_bookmark_from_db()
 */
typedef bool (*media_bookmark_cb)(media_bookmark_h bookmark, void *user_data);

/**
 * @ingroup CAPI_CONTENT_MEDIA_ALBUM_MODULE
 * @brief Called for every album in the obtained list of groups.
 * @details Iterates over an album list.
 *
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @remarks To use the @a album outside this function, copy the handle with the media_album_clone() function. \n
 *                  The callback is called in the main loop.
 *
 * @param[in] album The handle to the media album
 * @param[in] user_data The user data passed from the foreach function
 *
 * @return @c true to continue with the next iteration of the loop,
 *         otherwise @c false to break out of the loop
 *
 * @pre media_album_foreach_album_from_db() will invoke this function.
 *
 * @see media_album_clone()
 * @see media_album_foreach_album_from_db()
 */
typedef bool (*media_album_cb)(media_album_h album, void *user_data);

/**
 * @ingroup CAPI_CONTENT_MEDIA_GROUP_MODULE
 * @brief Called for every group in the obtained list of groups.
 * @details Iterates over a media group list.
 *
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @remarks You should not free @a group_name returned by this function. \n
 *                  The callback is called in the main loop.
 *
 * @param[in] group_name The name of the media group
 * @param[in] user_data The user data passed from the foreach function
 *
 * @return @c true to continue with the next iteration of the loop,
 *         otherwise @c false to break out of the loop
 *
 * @pre media_group_foreach_group_from_db() will invoke this function.
 * @see media_group_foreach_group_from_db()
 */
typedef bool (*media_group_cb)(const char *group_name, void *user_data);

/**
 * @ingroup CAPI_CONTENT_MEDIA_STORAGE_MODULE
 * @brief Called for every storage in the obtained list of storages.
 * @details Iterates over a media storage list.
 *
 * @since_tizen @if MOBILE 2.4 @elseif WEARABLE 3.0 @endif
 *
 * @remarks You should not destroy @a storage returned by this function. \n
 *                  The callback is called in the main loop.
 *
 * @param[in] storage The handle of the media storage
 * @param[in] user_data The user data passed from the foreach function
 *
 * @return @c true to continue with the next iteration of the loop,
 *         otherwise @c false to break out of the loop
 *
 * @pre media_storage_foreach_storage_from_db() will invoke this function.
 * @see media_storage_foreach_storage_from_db()
 */
typedef bool (*media_storage_cb)(media_storage_h storage, void *user_data);

/**
 * @ingroup CAPI_CONTENT_MEDIA_FACE_MODULE
 * @brief Called for every face in the obtained list of face.
 * @details Iterates over a media face list.
 *
 * @since_tizen 3.0
 *
 * @remarks You should not destroy @a face returned by this function. \n
 *                  The callback is called in the main loop.
 *
 * @param[in] face The handle of the media face
 * @param[in] user_data The user data passed from the foreach function
 *
 * @return @c true to continue with the next iteration of the loop,
 *         otherwise @c false to break out of the loop
 *
 * @pre media_info_foreach_face_from_db() will invoke this function.
 * @see media_info_foreach_face_from_db()
 */
typedef bool (*media_face_cb)(media_face_h face, void *user_data);

/**
 * @}
 */

/**
 * @addtogroup CAPI_CONTENT_MEDIA_FILTER_MODULE
 * @{
 */

 /**
 * @brief Media ID.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_ID "MEDIA_ID"

/**
 * @brief Media full path.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_PATH "MEDIA_PATH"

/**
 * @brief The file name including the extension.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_DISPLAY_NAME "MEDIA_DISPLAY_NAME"

/**
 * @brief Media type.
 * @details There are 5 media types:\n
 *               0-image, 1-video, 2-sound, 3-music, 4-other\n
 *               You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_TYPE "MEDIA_TYPE"

/**
 * @brief Media MIME type.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_MIME_TYPE "MEDIA_MIME_TYPE"


/**
 * @brief Media file size.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_SIZE "MEDIA_SIZE"

/**
 * @brief The time that content file was first added to media database.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_ADDED_TIME "MEDIA_ADDED_TIME"

/**
 * @brief The last modification time provided by the file system.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_MODIFIED_TIME "MEDIA_MODIFIED_TIME"

/**
 * @brief  Media created time.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_TIMELINE "MEDIA_TIMELINE"

/**
 * @brief Media thumbnail path.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_THUMBNAIL_PATH "MEDIA_THUMBNAIL_PATH"

/**
 * @brief Media title get from tag or file name.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_TITLE "MEDIA_TITLE"

/**
 * @brief Media album name.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_ALBUM "MEDIA_ALBUM"

/**
 * @brief Media artist.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_ARTIST "MEDIA_ARTIST"

/**
 * @brief Media album artist.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_ALBUM_ARTIST "MEDIA_ALBUM_ARTIST"

/**
 * @brief Media genre.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_GENRE "MEDIA_GENRE"

/**
 * @brief Media composer.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_COMPOSER "MEDIA_COMPOSER"

/**
 * @brief Media year.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_YEAR "MEDIA_YEAR"

/**
 * @brief Media recorded date.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_RECORDED_DATE "MEDIA_RECORDED_DATE"

/**
 * @brief Media copyright.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_COPYRIGHT "MEDIA_COPYRIGHT"

/**
 * @brief Media track number.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_TRACK_NUM "MEDIA_TRACK_NUM"

/**
 * @brief Media description.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_DESCRIPTION "MEDIA_DESCRIPTION"

/**
 * @brief Media bitrate.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_BITRATE "MEDIA_BITRATE"

/**
 * @brief Media bit per sample.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_BITPERSAMPLE "MEDIA_BITPERSAMPLE"

/**
 * @brief Media sample rate.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_SAMPLERATE "MEDIA_SAMPLERATE"

/**
 * @brief Media channel.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_CHANNEL "MEDIA_CHANNEL"

/**
 * @brief Media duration.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_DURATION "MEDIA_DURATION"

/**
 * @brief Media longitude.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_LONGITUDE "MEDIA_LONGITUDE"

/**
 * @brief Media latitude.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_LATITUDE "MEDIA_LATITUDE"

/**
 * @brief Media altitude.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_ALTITUDE "MEDIA_ALTITUDE"

/**
 * @brief Media width.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_WIDTH "MEDIA_WIDTH"

/**
 * @brief Media height.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_HEIGHT "MEDIA_HEIGHT"

/**
 * @brief Media datetaken.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_DATETAKEN "MEDIA_DATETAKEN"

/**
 * @brief Media orientation.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_ORIENTATION "MEDIA_ORIENTATION"

/**
 * @deprecated Deprecated since 4.0.
 * @brief Media burst ID.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_BURST_ID "BURST_ID"

/**
 * @deprecated Deprecated since 4.0.
 * @brief Media playedcount.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_PLAYED_COUNT "MEDIA_PLAYED_COUNT"

/**
 * @deprecated Deprecated since 4.0.
 * @brief Media last played time.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_LAST_PLAYED_TIME "MEDIA_LAST_PLAYED_TIME"

/**
 * @deprecated Deprecated since 4.0.
 * @brief Media last played position of file.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_LAST_PLAYED_POSITION "MEDIA_LAST_PLAYED_POSITION"

/**
 * @brief Media rating.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_RATING "MEDIA_RATING"

/**
 * @brief The time to register favourite.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_FAVOURITE "MEDIA_FAVOURITE"

/**
 * @deprecated Deprecated since 4.0.
 * @brief Media author.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_AUTHOR "MEDIA_AUTHOR"

/**
 * @deprecated Deprecated since 4.0.
 * @brief Media provider.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_PROVIDER "MEDIA_PROVIDER"

/**
 * @deprecated Deprecated since 4.0.
 * @brief Media content name.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_CONTENT_NAME "MEDIA_CONTENT_NAME"

/**
 * @deprecated Deprecated since 4.0.
 * @brief Media category.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_CATEGORY "MEDIA_CATEGORY"

/**
 * @deprecated Deprecated since 4.0.
 * @brief Media location tag.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_LOCATION_TAG "MEDIA_LOCATION_TAG"

/**
 * @deprecated Deprecated since 4.0.
 * @brief Media age rating.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_AGE_RATING "MEDIA_AGE_RATING"

/**
 * @deprecated Deprecated since 4.0.
 * @brief Media keyword.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_KEYWORD "MEDIA_KEYWORD"

/**
 * @deprecated Deprecated since 4.0.
 * @brief Media weather.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_WEATHER "MEDIA_WEATHER"

/**
 * @brief Is DRM.
 * @details There are 2 types:\n
 *               0-not drm, 1-drm\n
 *               You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_IS_DRM "MEDIA_IS_DRM"

/**
 * @brief Media storage.
 * @details There are 2 types:\n
 *               0-internal storage, 1-external storage\n
 *               You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 */
#define MEDIA_STORAGE_TYPE "MEDIA_STORAGE_TYPE"

/**
 * @brief Media exposure time.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.4 @elseif WEARABLE 3.0 @endif
 *
 */
#define MEDIA_EXPOSURE_TIME "MEDIA_EXPOSURE_TIME"

/**
 * @brief Media fnumber.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.4 @elseif WEARABLE 3.0 @endif
 *
 */
#define MEDIA_FNUMBER "MEDIA_FNUMBER"

/**
 * @brief Media ISO.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.4 @elseif WEARABLE 3.0 @endif
 *
 */
#define MEDIA_ISO "MEDIA_ISO"

/**
 * @brief Media model.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.4 @elseif WEARABLE 3.0 @endif
 *
 */
#define MEDIA_MODEL "MEDIA_MODEL"

/**
 * @brief Media file name pinyin.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 * @remarks Since 4.0, this keyword is related to the following feature:\n
 *          - %http://tizen.org/feature/content.filter.pinyin\n
 *             If this feature is not supported on the device, PINYIN search does not work.
 *
 */
#define MEDIA_FILE_NAME_PINYIN "MEDIA_FILE_NAME_PINYIN"

/**
 * @brief Media title pinyin.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 * @remarks Since 4.0, this keyword is related to the following feature:\n
 *          - %http://tizen.org/feature/content.filter.pinyin\n
 *             If this feature is not supported on the device, PINYIN search does not work.
 *
 */
#define MEDIA_TITLE_PINYIN "MEDIA_TITLE_PINYIN"

/**
 * @brief Media album pinyin.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 * @remarks Since 4.0, this keyword is related to the following feature:\n
 *          - %http://tizen.org/feature/content.filter.pinyin\n
 *            If this feature is not supported on the device, PINYIN search does not work.
 *
 */
#define MEDIA_ALBUM_PINYIN "MEDIA_ALBUM_PINYIN"

/**
 * @brief Media artist pinyin.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 * @remarks Since 4.0, this keyword is related to the following feature:\n
 *          - %http://tizen.org/feature/content.filter.pinyin\n
 *             If this feature is not supported on the device, PINYIN search does not work.
 *
 */
#define MEDIA_ARTIST_PINYIN "MEDIA_ARTIST_PINYIN"

/**
 * @brief Media album artist pinyin.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 * @remarks Since 4.0, this keyword is related to the following feature:\n
 *          - %http://tizen.org/feature/content.filter.pinyin\n
 *            If this feature is not supported on the device, PINYIN search does not work.
 *
 */
#define MEDIA_ALBUM_ARTIST_PINYIN "MEDIA_ALBUM_ARTIST_PINYIN"

/**
 * @brief Media genre pinyin.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 * @remarks Since 4.0, this keyword is related to the following feature:\n
 *          - %http://tizen.org/feature/content.filter.pinyin\n
 *            If this feature is not supported on the device, PINYIN search does not work.
 *
 */
#define MEDIA_GENRE_PINYIN "MEDIA_GENRE_PINYIN"

/**
 * @brief Media composer pinyin.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 * @remarks Since 4.0, this keyword is related to the following feature:\n
 *          - %http://tizen.org/feature/content.filter.pinyin\n
 *            If this feature is not supported on the device, PINYIN search does not work.
 *
 */
#define MEDIA_COMPOSER_PINYIN "MEDIA_COMPOSER_PINYIN"

/**
 * @brief Media copyright pinyin.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 * @remarks Since 4.0, this keyword is related to the following feature:\n
 *          - %http://tizen.org/feature/content.filter.pinyin\n
 *            If this feature is not supported on the device, PINYIN search does not work.
 *
 */
#define MEDIA_COPYRIGHT_PINYIN "MEDIA_COPYRIGHT_PINYIN"

/**
 * @brief Media description pinyin.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 * @remarks Since 4.0, this keyword is related to the following feature:\n
 *          - %http://tizen.org/feature/content.filter.pinyin\n
 *            If this feature is not supported on the device, PINYIN search does not work.
 *
 */
#define MEDIA_DESCRIPTION_PINYIN "MEDIA_DESCRIPTION_PINYIN"

/**
 * @deprecated Deprecated since 4.0.
 * @brief Media author pinyin.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 * @remarks Since 4.0, this keyword is related to the following feature:\n
 *          - %http://tizen.org/feature/content.filter.pinyin\n
 *            If this feature is not supported on the device, PINYIN search does not work.
 *
 */
#define MEDIA_AUTHOR_PINYIN "MEDIA_AUTHOR_PINYIN"

/**
 * @deprecated Deprecated since 4.0.
 * @brief Media provider pinyin.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 * @remarks Since 4.0, this keyword is related to the following feature:\n
 *          - %http://tizen.org/feature/content.filter.pinyin\n
 *            If this feature is not supported on the device, PINYIN search does not work.
 *
 */
#define MEDIA_PROVIDER_PINYIN "MEDIA_PROVIDER_PINYIN"

/**
 * @deprecated Deprecated since 4.0.
 * @brief Media content name pinyin.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 * @remarks Since 4.0, this keyword is related to the following feature:\n
 *          - %http://tizen.org/feature/content.filter.pinyin\n
 *            If this feature is not supported on the device, PINYIN search does not work.
 *
 */
#define MEDIA_CONTENT_NAME_PINYIN "MEDIA_CONTENT_NAME_PINYIN"

/**
 * @deprecated Deprecated since 4.0.
 * @brief Media category pinyin.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 * @remarks Since 4.0, this keyword is related to the following feature:\n
 *          - %http://tizen.org/feature/content.filter.pinyin\n
 *            If this feature is not supported on the device, PINYIN search does not work.
 *
 */
#define MEDIA_CATEGORY_PINYIN "MEDIA_CATEGORY_PINYIN"

/**
 * @deprecated Deprecated since 4.0.
 * @brief Media location tag pinyin.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 * @remarks Since 4.0, this keyword is related to the following feature:\n
 *          - %http://tizen.org/feature/content.filter.pinyin\n
 *            If this feature is not supported on the device, PINYIN search does not work.
 *
 */
#define MEDIA_LOCATION_TAG_PINYIN "MEDIA_LOCATION_TAG_PINYIN"

/**
 * @deprecated Deprecated since 4.0.
 * @brief Media age rating pinyin.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 * @remarks Since 4.0, this keyword is related to the following feature:\n
 *          - %http://tizen.org/feature/content.filter.pinyin\n
 *            If this feature is not supported on the device, PINYIN search does not work.
 *
 */
#define MEDIA_AGE_RATING_PINYIN "MEDIA_AGE_RATING_PINYIN"

/**
 * @deprecated Deprecated since 4.0.
 * @brief Media keyword pinyin.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 * @remarks Since 4.0, this keyword is related to the following feature:\n
 *          - %http://tizen.org/feature/content.filter.pinyin\n
 *            If this feature is not supported on the device, PINYIN search does not work.
 *
 */
#define MEDIA_KEYWORD_PINYIN "MEDIA_KEYWORD_PINYIN"

/**
 * @brief Virtual Reality content.
 * @details You can use above define to set the condition of media filter and order keyword.
 * @since_tizen 3.0
 *
 */
#define MEDIA_360 "MEDIA_360"

 /**
 * @brief Folder ID.
 * @details You can use above define to set the condition of folder filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 */
#define FOLDER_ID "FOLDER_ID"

/**
 * @brief Folder full path.
 * @details You can use above define to set the condition of folder filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 */
#define FOLDER_PATH "FOLDER_PATH"

/**
 * @brief Folder base name.
 * @details You can use above define to set the condition of folder filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 */
#define FOLDER_NAME "FOLDER_NAME"

/**
 * @deprecated Deprecated since 4.0.
 * @brief Folder modified time.
 * @details You can use above define to set the condition of folder filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 */
#define FOLDER_MODIFIED_TIME "FOLDER_MODIFIED_TIME"

/**
 * @brief Folder storage.
 * @details There are 2 types:\n
 *               0-internal storage, 1-external storage\n
 *               You can use above define to set the condition of folder filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 */
#define FOLDER_STORAGE_TYPE "FOLDER_STORAGE_TYPE"

/**
 * @brief Folder base name pinyin.
 * @details You can use above define to set the condition of folder filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 * @remarks Since 4.0, this keyword is related to the following feature:\n
 *          - %http://tizen.org/feature/content.filter.pinyin\n
 *            If this feature is not supported on the device, PINYIN search does not work.
 */
#define FOLDER_NAME_PINYIN "FOLDER_NAME_PINYIN"

/**
 * @deprecated Deprecated since 4.0.
 * @brief Folder order info.
 * @details You can use above define to set the condition of folder filter and order keyword.
 * @since_tizen @if MOBILE 2.4 @elseif WEARABLE 3.0 @endif
 */
#define FOLDER_ORDER "FOLDER_ORDER"

/**
 * @deprecated Deprecated since 4.0.
 * @brief Parent folder id.
 * @details You can use above define to set the condition of folder filter and order keyword.
 * @since_tizen @if MOBILE 2.4 @elseif WEARABLE 3.0 @endif
 */
#define FOLDER_PARENT_FOLDER_ID "FOLDER_PARENT_FOLDER_ID"

/**
 * @brief Playlist name.
 * @details You can use above define to set the condition of playlist filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 */
#define PLAYLIST_NAME "PLAYLIST_NAME"

/**
 * @brief Playlist id.
 * @details You can use above define to set the condition of playlist filter and order keyword.
 * @since_tizen 4.0
 */
#define MEDIA_PLAYLIST_ID "PLAYLIST_ID"

/**
 * @brief Playlist member order.
 * @details You can use above define to set the condition of playlist filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 */
#define PLAYLIST_MEMBER_ORDER "PLAYLIST_MEMBER_ORDER"

/**
 * @brief Media count in playlist.
 * @details You can use above define to set the condition of playlist filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 */
#define PLAYLIST_MEDIA_COUNT "PLAYLIST_MEDIA_COUNT"

/**
 * @brief Tag name.
 * @details You can use above define to set the condition of tag filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 */
#define TAG_NAME "TAG_NAME"

/**
 * @brief Tag id.
 * @details You can use above define to set the condition of tag filter and order keyword.
 * @since_tizen 4.0
 */
#define MEDIA_TAG_ID "TAG_ID"

/**
 * @brief Media count in tag.
 * @details You can use above define to set the condition of tag filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 */
#define TAG_MEDIA_COUNT "TAG_MEDIA_COUNT"

/**
 * @brief Bookmark marked time.
 * @details You can use above define to set the condition of bookmark filter and order keyword.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 */
#define BOOKMARK_MARKED_TIME "BOOKMARK_MARKED_TIME"

/**
 * @brief Bookmark id.
 * @details You can use above define to set the condition of bookmark filter and order keyword.
 * @since_tizen 4.0
 */
#define MEDIA_BOOKMARK_ID "BOOKMARK_ID"

/**
 * @brief Bookmark name.
 * @details You can use above define to set the condition of bookmark filter and order keyword.
 * @since_tizen 4.0
 */
#define BOOKMARK_NAME "BOOKMARK_NAME"

/**
 * @brief Storage ID.
 * @details You can use above define to set the condition of storage filter and order keyword.
 * @since_tizen @if MOBILE 2.4 @elseif WEARABLE 3.0 @endif
 */
#define MEDIA_STORAGE_ID					"STORAGE_ID"

/**
 * @brief Storage path.
 * @details You can use above define to set the condition of storage filter and order keyword.
 * @since_tizen @if MOBILE 2.4 @elseif WEARABLE 3.0 @endif
 */
#define MEDIA_STORAGE_PATH				"STORAGE_PATH"

/**
 * @brief Face tag.
 * @details You can use above define to set the condition of face filter and order keyword.
 * @since_tizen 3.0
 */
#define MEDIA_FACE_TAG	"MEDIA_FACE_TAG"

/**
 * @brief Face id.
 * @details You can use above define to set the condition of face filter and order keyword.
 * @since_tizen 4.0
 */
#define MEDIA_FACE_ID	"MEDIA_FACE_ID"

/**
 * @}
 */


#ifdef __cplusplus
}
#endif /* __cplusplus */


#endif /*__TIZEN_CONTENT_MEDIA_CONTENT_TYPE_H__*/
