/*
* Copyright (c) 2011 Samsung Electronics Co., Ltd All Rights Reserved
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
* http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/


#ifndef __TIZEN_CONTENT_MEDIA_INFO_H__
#define __TIZEN_CONTENT_MEDIA_INFO_H__

#include <media_content_type.h>



#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/**
 * @file media_info.h
 * @brief This file contains the media info function and related functions to proceed with it. \n
 *        You can use the functions to insert, delete, count and get list of content files from media database. \n
 *        You can get properties of content file such as size, mime_type, modified_time etc. And you can set properties such as provider, favorite etc. \n
 *        And you can get bookmark, face, tag info related the content file.
 */


/**
 * @addtogroup CAPI_CONTENT_MEDIA_INFO_MODULE
 * @{
 */

/**
 * @brief  Inserts the content file into the media database.
 * @details In general, you can use this function to insert content files into the media database. \n
 *                You can use media_content_scan_file()/media_content_scan_folder() function instead of this function.
 *
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @privlevel public
 * @privilege %http://tizen.org/privilege/content.write \n
 *                   %http://tizen.org/privilege/mediastorage \n
 *                   %http://tizen.org/privilege/externalstorage
 *
 * @remarks The @a info should be released using media_info_destroy(). \n
 *                   You must add privilege http://tizen.org/privilege/content.write. And You add more privilege depending on your choice of contents path. \n
 *                   If you want to access only internal storage by using this function, you should add privilege http://tizen.org/privilege/mediastorage. \n
 *                   Or if you want to access only external storage by using this function, you should add privilege http://tizen.org/privilege/externalstorage. \n
 *                   If you can access both storage, you must add all privilege.
 * @remarks Since 4.0, this function is related to the following feature:\n
 *              %http://tizen.org/feature/content.scanning.others\n
 *              If this feature is not suppported on the device, MEDIA_CONTENT_TYPE_OTHERS type file is not scanned.
 *
 * @param[in] path The path of the content file to add
 * @param[out] info The handle of the inserted content file
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_INVALID_OPERATION Invalid operation
 * @retval #MEDIA_CONTENT_ERROR_DB_FAILED         DB Operation failed
 * @retval #MEDIA_CONTENT_ERROR_DB_BUSY           DB Operation busy
 * @retval #MEDIA_CONTENT_ERROR_NETWORK           Network fail
 * @retval #MEDIA_CONTENT_ERROR_PERMISSION_DENIED Permission denied
 * @retval #MEDIA_CONTENT_ERROR_NOT_SUPPORTED     Not supported
 *
 * @pre This function requires opened connection to content service by media_content_connect().
 *
 * @see media_content_connect()
 * @see media_content_scan_file()
 * @see media_content_scan_folder()
 */
int media_info_insert_to_db(const char *path, media_info_h *info);

/**
 * @brief Inserts content files into the media database, asynchronously.
 * @details This function can insert multiple content files into the media database.
 *
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @privlevel public
 * @privilege %http://tizen.org/privilege/content.write \n
 *                   %http://tizen.org/privilege/mediastorage \n
 *                   %http://tizen.org/privilege/externalstorage
 *
 * @remarks You must add privilege http://tizen.org/privilege/content.write. And You add more privilege depending on your choice of contents path. \n
 *                   If you want to access only internal storage by using this function, you should add privilege http://tizen.org/privilege/mediastorage. \n
 *                   Or if you want to access only external storage by using this function, you should add privilege http://tizen.org/privilege/externalstorage. \n
 *                   If you can access both storage, you must add all privilege.
 * @remarks Since 4.0, this function is related to the following feature:\n
 *              %http://tizen.org/feature/content.scanning.others\n
 *              If this feature is not suppported on the device, MEDIA_CONTENT_TYPE_OTHERS type file is not scanned.
 *
 * @param[in] path_array The path array of the content files to add
 * @param[in] array_length The length of the array
 * @param[in] callback The callback function to be invoked when media items inserted completely
 * @param[in] user_data The user data to be passed to the callback function
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_INVALID_OPERATION Invalid operation
 * @retval #MEDIA_CONTENT_ERROR_DB_FAILED         DB Operation failed
 * @retval #MEDIA_CONTENT_ERROR_DB_BUSY           DB Operation busy
 * @retval #MEDIA_CONTENT_ERROR_NETWORK           Network fail
 * @retval #MEDIA_CONTENT_ERROR_PERMISSION_DENIED Permission denied
 * @retval #MEDIA_CONTENT_ERROR_NOT_SUPPORTED     Not supported
 *
 * @pre This function requires opened connection to content service by media_content_connect().
 *
 * @see media_content_connect()
 * @see media_insert_completed_cb()
 */
int media_info_insert_batch_to_db(const char **path_array, unsigned int array_length, media_insert_completed_cb callback, void *user_data);

/**
 * @deprecated Deprecated since 4.0.
 * @brief Inserts the burst shot images into the media database, asynchronously.
 * @details This function inserts burst images into the media database.
 *          media_insert_burst_shot_completed_cb() will be called when insertion to media database is finished.
 *
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @privlevel public
 * @privilege %http://tizen.org/privilege/content.write \n
 *                   %http://tizen.org/privilege/mediastorage \n
 *                   %http://tizen.org/privilege/externalstorage
 *
 * @remarks You must add privilege http://tizen.org/privilege/content.write. And You add more privilege depending on your choice of contents path. \n
 *                   If you want to access only internal storage with this function, you should add privilege http://tizen.org/privilege/mediastorage. \n
 *                   Or if you want to access only external storage with this function, you should add privilege http://tizen.org/privilege/externalstorage. \n
 *                   If you can access both storage, you must add all privilege.
 *
 * @param[in] path_array The path array of the burst shot images to add
 * @param[in] array_length The length of the array
 * @param[in] callback The callback function to be invoked when the images are inserted completely
 * @param[in] user_data The user data to be passed to the callback function
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_INVALID_OPERATION Invalid operation
 * @retval #MEDIA_CONTENT_ERROR_DB_FAILED         DB Operation failed
 * @retval #MEDIA_CONTENT_ERROR_DB_BUSY           DB Operation busy
 * @retval #MEDIA_CONTENT_ERROR_NETWORK           Network fail
 * @retval #MEDIA_CONTENT_ERROR_PERMISSION_DENIED Permission denied
 *
 * @pre This function requires opened connection to content service by media_content_connect().
 *
 * @see media_content_connect()
 * @see media_insert_burst_shot_completed_cb()
 */
int media_info_insert_burst_shot_to_db(const char **path_array, unsigned int array_length, media_insert_burst_shot_completed_cb callback, void *user_data) TIZEN_DEPRECATED_API;

/**
 * @brief Deletes the media information from the media database.
 * @details This function only remove media information in the media database. \n
 *               You can use media_content_scan_file()/media_content_scan_folder() function instead of this function if file is removed from the file system.
 *
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @privlevel public
 * @privilege %http://tizen.org/privilege/content.write
 *
 * @param[in] media_id The media ID. It can get from media info handle.
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #MEDIA_CONTENT_ERROR_INVALID_OPERATION Invalid operation
 * @retval #MEDIA_CONTENT_ERROR_DB_FAILED         DB Operation failed
 * @retval #MEDIA_CONTENT_ERROR_DB_BUSY           DB Operation busy
 * @retval #MEDIA_CONTENT_ERROR_NETWORK           Network fail
 * @retval #MEDIA_CONTENT_ERROR_PERMISSION_DENIED Permission denied
 *
 * @pre This function requires opened connection to content service by media_content_connect().
 *
 * @see media_content_connect()
 * @see media_content_scan_file()
 * @see media_content_scan_folder()
 */
int media_info_delete_from_db(const char *media_id);

/**
 * @deprecated Deprecated since 4.0.
 * @brief Deletes media information that meets the conditions of the filter from the media database.
 * @details This function can delete multiple media information from the media database.
 *
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @privlevel public
 * @privilege %http://tizen.org/privilege/content.write
 *
 * @param[in] filter The handle to the media filter
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #MEDIA_CONTENT_ERROR_INVALID_OPERATION Invalid operation
 * @retval #MEDIA_CONTENT_ERROR_DB_FAILED         DB Operation failed
 * @retval #MEDIA_CONTENT_ERROR_DB_BUSY           DB Operation busy
 * @retval #MEDIA_CONTENT_ERROR_NETWORK           Network fail
 * @retval #MEDIA_CONTENT_ERROR_PERMISSION_DENIED Permission denied
 *
 * @pre This function requires opened connection to content service by media_content_connect().
 *
 * @see media_content_connect()
 * @see media_filter_create()
 */
int media_info_delete_batch_from_db(filter_h filter) TIZEN_DEPRECATED_API;


/**
 * @brief Destroys media info.
 * @details The function frees all resources related to the media info handle. This handle
 *          can no longer be used to perform any operation. New media info handle has to
 *          be created before the next usage.
 *
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @param[in] media The handle to the media info
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 *
 * @pre Get copy of media_info handle by calling media_info_clone().
 *
 * @see media_info_clone()
 */
int media_info_destroy(media_info_h media);

/**
 * @brief Clones the media info handle.
 *
 * @details This function copies the media info handle from a source to the destination.
 *          There is no media_info_create() function. The media_info_h is created internally and
 *          available through media info foreach function such as media_info_foreach_media_from_db() or media_info_insert_to_db().
 *          To use this handle outside of these foreach functions, use this function.
 *
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @remarks The @a dst should be released using media_info_destroy().
 *
 * @param[out] dst The destination handle to the media info
 * @param[in] src The source handle to the media info
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 *
 * @see media_info_destroy()
 * @see media_album_foreach_media_from_db()
 * @see media_playlist_foreach_media_from_db()
 * @see media_group_foreach_media_from_db()
 * @see media_tag_foreach_media_from_db()
 * @see media_info_foreach_media_from_db()
 * @see media_info_insert_to_db()
 * @see media_folder_foreach_media_from_db()
 */
int media_info_clone(media_info_h *dst, media_info_h src);

/**
 * @brief Gets the count of media info for the passed @a filter from the media database.
 * @details If @c NULL is passed to the @a filter, then no filtering is applied.
 *
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 *
 * @param[in] filter The handle to the media filter
 * @param[out] media_count The count of media
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #MEDIA_CONTENT_ERROR_DB_FAILED         DB Operation failed
 * @retval #MEDIA_CONTENT_ERROR_DB_BUSY           DB Operation busy
 *
 * @pre This function requires opened connection to content service by media_content_connect().
 *
 * @see media_content_connect()
 * @see media_filter_create()
 */
int media_info_get_media_count_from_db(filter_h filter, int *media_count);

/**
 * @brief Iterates through media info from the media database.
 * @details This function gets all media info handles meeting the given @a filter.
 *          The @a callback function will be invoked for every retrieved media info.
 *          If @c NULL is passed to the @a filter, then no filtering is applied.
 *
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @remarks Do not call updating DB function like media_info_update_to_db(), media_info_refresh_metadata_to_db(), audio_meta_update_to_db(), image_meta_update_to_db() and video_meta_update_to_db()  in your callback function,
 *                   your callback function is invoked as inline function.
 *                   So, your callback function is in read state in SQLite. When you are in read state, sometimes you do not update DB.
 *                   We do not recommend you call updating DB function in callback of foreach function.
 *
 * @param[in] filter The handle to the media filter
 * @param[in] callback The callback function to be invoked
 * @param[in] user_data The user data to be passed to the callback function
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_DB_FAILED         DB Operation failed
 * @retval #MEDIA_CONTENT_ERROR_DB_BUSY           DB Operation busy
 *
 * @pre  This function requires opened connection to content service by media_content_connect().
 * @post This function invokes media_info_cb().
 *
 * @see media_content_connect()
 * @see #media_info_cb
 * @see media_filter_create()
 */
int media_info_foreach_media_from_db(filter_h filter, media_info_cb callback, void *user_data);

/**
 * @brief Gets the count of media tags for the passed @a filter in the given @a media_id from the media database.
 * @details If @c NULL is passed to the @a filter, then no filtering is applied.
 *
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @param[in] media_id The media ID
 * @param[in] filter The handle to the media filter
 * @param[out] tag_count The count of the media tag
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #MEDIA_CONTENT_ERROR_DB_FAILED         DB Operation failed
 * @retval #MEDIA_CONTENT_ERROR_DB_BUSY           DB Operation busy
 *
 * @pre This function requires opened connection to content service by media_content_connect().
 *
 * @see media_content_connect()
 * @see media_filter_create()
 */
int media_info_get_tag_count_from_db(const char *media_id, filter_h filter, int *tag_count);

/**
 * @brief Iterates through the media tag in the given media info from the media database.
 * @details This function gets all the media tags associated with the given @a media_id and calls @a callback for every retrieved media tag. \n
 *               If @c NULL is passed to the @a filter, then no filtering is applied.
 *
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @param[in] media_id The media ID
 * @param[in] filter The handle to the media filter
 * @param[in] callback The callback function to be invoked
 * @param[in] user_data The user data to be passed to the callback function
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_DB_FAILED         DB Operation failed
 * @retval #MEDIA_CONTENT_ERROR_DB_BUSY           DB Operation busy
 *
 * @pre  This function requires opened connection to content service by media_content_connect().
 * @post This function invokes media_tag_cb().
 *
 * @see media_content_connect()
 * @see #media_tag_cb
 * @see media_filter_create()
 */
int media_info_foreach_tag_from_db(const char *media_id, filter_h filter, media_tag_cb callback, void *user_data);

/**
 * @brief Gets the number of bookmarks for the passed @a filter in the given media ID from the media database.
 * @details If @c NULL is passed to the @a filter, then no filtering is applied.
 *
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @param[in] media_id The media ID
 * @param[in] filter The handle to the media filter
 * @param[out] bookmark_count The count of the media tag
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #MEDIA_CONTENT_ERROR_DB_FAILED         DB Operation failed
 * @retval #MEDIA_CONTENT_ERROR_DB_BUSY           DB Operation busy
 *
 * @pre This function requires opened connection to content service by media_content_connect().
 *
 * @see media_content_connect()
 * @see media_filter_create()
 */
int media_info_get_bookmark_count_from_db(const char *media_id, filter_h filter, int *bookmark_count);

/**
 * @brief Iterates through the media bookmark in the given media info from the media database.
 * @details This function gets all media bookmarks associated with the given media and calls @a callback for every retrieved media bookmark.
 *                If @c NULL is passed to the @a filter, then no filtering is applied.
 *
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @param[in] media_id The media ID
 * @param[in] filter The handle to the media filter
 * @param[in] callback The callback function to be invoked
 * @param[in] user_data The user data to be passed to the callback function
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_DB_FAILED         DB Operation failed
 * @retval #MEDIA_CONTENT_ERROR_DB_BUSY           DB Operation busy
 *
 * @pre  This function requires opened connection to content service by media_content_connect().
 * @post This function invokes media_bookmark_cb().
 *
 * @see media_content_connect()
 * @see media_bookmark_cb()
 * @see media_filter_create()
 */
int media_info_foreach_bookmark_from_db(const char *media_id, filter_h filter, media_bookmark_cb callback, void *user_data);

/**
 * @brief Gets the number of face for the passed @a media_id from the media database.
 * @details If @c NULL is passed to the @a filter, then no filtering is applied.
 *
 * @since_tizen 3.0
 *
 * @param[in] media_id The media ID
 * @param[in] filter The handle to the media filter
 * @param[out] face_count The count of media face
 *
 * @return 0 on success, otherwise a negative error value.
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE Successful
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #MEDIA_CONTENT_ERROR_DB_FAILED DB operation failed
 *
 * @pre This function requires opened connection to content service by media_content_connect().
 * @see media_content_connect()
 * @see media_filter_create()
 *
 */
int media_info_get_face_count_from_db(const char *media_id, filter_h filter, int *face_count);

/**
 * @brief Iterates through the media files with optional @a filter in the given @a media_id from the media database.
 * @details This function gets all media face info associated with the given media id and
 * meeting desired filter option and calls @a callback for
 * every retrieved media face info. If NULL is passed to the @a filter, no filtering is applied.
 *
 * @since_tizen 3.0
 *
 * @param[in] media_id The media ID
 * @param[in] filter The handle to the media filter
 * @param[in] callback The callback function to invoke
 * @param[in] user_data The user data to be passed to the callback function
 *
 * @return 0 on success, otherwise a negative error value.
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE Successful
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY Out of memory
 *
 * @pre This function requires opened connection to content service by media_content_connect().
 * @see media_content_connect()
 * @see media_filter_create()
 *
 */
int media_info_foreach_face_from_db(const char *media_id, filter_h filter, media_face_cb callback, void *user_data);

/**
 * @brief Gets the image metadata handle for a given media info.
 * @details This function returns an image metadata handle retrieved from the media info.
 *
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @remarks The @a image should be released using image_meta_destroy().
 *
 * @param[in] media The handle to the media info
 * @param[out] image The handle to the image metadata
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 *
 * @see image_meta_destroy()
 */
int media_info_get_image(media_info_h media, image_meta_h *image);

/**
 * @brief Gets a video metadata handle for a given media info.
 * @details This function returns a video metadata handle retrieved from the media info handle.
 *
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @remarks The @a video should be released using video_meta_destroy().
 *
 * @param[in] media The handle to the media info
 * @param[out] video The handle to the video metadata
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 *
 * @see video_meta_destroy()
 */
int media_info_get_video(media_info_h media, video_meta_h *video);

/**
 * @brief Gets an audio metadata handle for a given media info.
 * @details This function returns an audio metadata handle retrieved from the media info handle.
 *
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @remarks The @a audio should be released using audio_meta_destroy().
 *
 * @param[in] media The handle to the media info
 * @param[out] audio The handle to the audio metadata
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 *
 * @see audio_meta_destroy()
 */
int media_info_get_audio(media_info_h media, audio_meta_h *audio);

/**
 * @brief Gets the media ID.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 * @remarks The @a media_id should be released using free().
 *
 * @param[in] media The handle to the media info
 * @param[out] media_id The media ID
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 */
int media_info_get_media_id(media_info_h media, char **media_id);

/**
 * @brief Gets the full path of the content file.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @remarks The @a path should be released using free().
 *
 * @param[in] media The handle to the media info
 * @param[out] path The full path of the content file
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 *
 */
int media_info_get_file_path(media_info_h media, char **path);

/**
 * @brief Gets the file name including the extension of the content file.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @remarks The @a name should be released using free().
 *
 * @param[in] media The handle to the media info
 * @param[out] name The file name including the extension of the content file
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 */
int media_info_get_display_name(media_info_h media, char **name);

/**
 * @brief Gets the content type of the content file.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @param[in] media The handle to the media info
 * @param[out] type The content type of the content file (#media_content_type_e)
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 *
 * @see #media_content_type_e
 */
int media_info_get_media_type(media_info_h media, media_content_type_e *type);

/**
 * @brief Gets the MIME type of the content file.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @remarks The @a mime_type should be released using free().
 *
 * @param[in] media The handle to the media info
 * @param[out] mime_type The MIME type of the content file
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 */
int media_info_get_mime_type(media_info_h media, char **mime_type);

/**
 * @brief Gets the content file size.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @param[in] media The handle to the media info
 * @param[out] size The content file size
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 */
int media_info_get_size(media_info_h media, unsigned long long *size);

/**
 * @brief Gets the added time of the content file.
 * @details The added time refers to the time that content file was first added to media database.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @param[in] media The handle to the media info
 * @param[out] added_time The added time to the media database
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 */
int media_info_get_added_time(media_info_h media, time_t *added_time);

/**
 * @brief Gets the modified time of the content file.
 * @details The modified time refers to the last modification time provided by the file system.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @param[in] media The handle to the media info
 * @param[out] time The last modification time of the content file
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 */
int media_info_get_modified_time(media_info_h media, time_t *time);

/**
 * @brief Gets the timeline of content file.
 * @details If the content file has the creation time information (like Content recored date or Image creation date), that value is timeline. \n
 *               Otherwise, timeline value is the same as modified time.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @param[in] media The handle to the media info
 * @param[out] time The timeline of content file
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 */
int media_info_get_timeline(media_info_h media, time_t *time);

/**
 * @brief Gets the thumbnail path of content file.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @remarks The @a path should be released using free(). \n
 *                   If the thumbnail extraction for the given media has not been requested yet, this function returns NULL. To create a thumbnail, you should use media_info_create_thumbnail() function. \n
 *                   Since 3.0, this function returns an empty string if media_info_create_thumbnail() has failed to create a thumbnail for the given media.
 *
 * @param[in] media The handle to the media info
 * @param[out] path The thumbnail path
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 */
int media_info_get_thumbnail_path(media_info_h media, char **path);

/**
 * @brief Gets the description of content file.
 * @details If the value is an empty string, the method returns "Unknown". \n
 *                Since 3.0, if the media info has no description, the method returns empty string.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @remarks The @a description should be released using free().
 *
 * @param[in] media The handle to the media info
 * @param[out] description The description of the content file
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 */
int media_info_get_description(media_info_h media, char **description);

/**
 * @brief Gets the longitude of content file.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @param[in] media The handle to the media info
 * @param[out] longitude The longitude of the content file
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 */
int media_info_get_longitude(media_info_h media, double *longitude);

/**
 * @brief Gets the latitude of content file.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @param[in] media The handle to the media info
 * @param[out] latitude The latitude of the content file
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 *
 */
int media_info_get_latitude(media_info_h media, double* latitude);

/**
 * @brief Gets the altitude of content file.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @param[in] media The handle to the media info
 * @param[out] altitude The altitude of the content file
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 */
int media_info_get_altitude(media_info_h media, double* altitude);

/**
 * @deprecated Deprecated since 4.0.
 * @brief Gets the weather of content file.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 * @remarks The @a weather should be released using free().
 *
 * @param[in] media The handle to the media info
 * @param[out] weather The weather of the content file
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 */
int media_info_get_weather(media_info_h media, char **weather) TIZEN_DEPRECATED_API;

/**
 * @brief Gets the rating of content file.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @param[in] media The handle to the media info
 * @param[out] rating The rating of the content file
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 */
int media_info_get_rating(media_info_h media, int *rating);

/**
 * @brief Gets the favorite status of content file which User set.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @param[in] media The handle to the media info
 * @param[out] favorite @c true if content file is set as favorite,
 *                       otherwise @c false if content file is not set as favorite
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 *
 * @see media_info_set_favorite()
 */
int media_info_get_favorite(media_info_h media, bool* favorite);

/**
 * @deprecated Deprecated since 4.0.
 * @brief Gets the author of content file.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @remarks The @a author should be released using free().
 *
 * @param[in] media The handle to the media info
 * @param[out] author The author of the content file
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 */
int media_info_get_author(media_info_h media, char **author) TIZEN_DEPRECATED_API;

/**
 * @deprecated Deprecated since 4.0.
 * @brief Gets the provider of content file which User set.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @remarks The @a provider should be released using free().
 *
 * @param[in] media The handle to the media info
 * @param[out] provider The provider of the content file
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 *
 */
int media_info_get_provider(media_info_h media, char **provider) TIZEN_DEPRECATED_API;

/**
 * @deprecated Deprecated since 4.0. Redundant, use media_info_get_display_name() instead.
 * @brief Gets the content name of content file which User set.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @remarks The @a content_name should be released using free().
 *
 * @param[in] media The handle to the media info
 * @param[out] content_name The content name of the content file
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 *
 */
int media_info_get_content_name(media_info_h media, char **content_name) TIZEN_DEPRECATED_API;

/**
 * @brief Gets the title of content file.
 * @details If the value is an empty string, the method returns "Unknown". \n
 *                Since 3.0, if the content file has no title, the method returns empty string.
 *
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @remarks The @a title should be released using free().
 *
 * @param[in] media The handle to the media info
 * @param[out] title The title of the content file
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 */
int media_info_get_title(media_info_h media, char **title);

/**
 * @deprecated Deprecated since 4.0.
 * @brief Gets the category of content file which User set.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @remarks The @a category should be released using free().
 *
 * @param[in] media The handle to the media info
 * @param[out] category The category of the content file
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 *
 */
int media_info_get_category(media_info_h media, char **category) TIZEN_DEPRECATED_API;

/**
 * @deprecated Deprecated since 4.0.
 * @brief Gets the location tag of content file which User set.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @remarks The @a location_tag should be released using free().
 *
 * @param[in] media The handle to the media info
 * @param[out] location_tag The location of the content file
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 *
 */
int media_info_get_location_tag(media_info_h media, char **location_tag) TIZEN_DEPRECATED_API;

/**
 * @deprecated Deprecated since 4.0.
 * @brief Gets the age_rating of content file which User set.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @remarks The @a age_rating should be released using free().
 *
 * @param[in] media The handle to the media info
 * @param[out] age_rating The age rating of the content file
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 *
 */
int media_info_get_age_rating(media_info_h media, char **age_rating) TIZEN_DEPRECATED_API;

/**
 * @deprecated Deprecated since 4.0. Use media_tag_insert_to_db() instead.
 * @brief Gets the keyword of content file which User set.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @remarks The @a keyword should be released using free().
 *
 * @param[in] media The handle to the media info
 * @param[out] keyword The keyword of the content file
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 *
 */
int media_info_get_keyword(media_info_h media, char **keyword) TIZEN_DEPRECATED_API;

/**
 * @brief Gets the storage id of content file.
 * @since_tizen @if MOBILE 2.4 @elseif WEARABLE 3.0 @endif
 *
 * @remarks The @a storage_id should be released using free().
 *
 * @param[in] media The handle to the media info
 * @param[out] storage_id The ID of the media storage
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 */
int media_info_get_storage_id(media_info_h media, char **storage_id);

/**
 * @brief Checks whether the media is protected via DRM.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @param[in] media The handle to the media info
 * @param[out] is_drm @c true if media is DRM media,
 *                    otherwise @c false if media is not DRM media
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 */
int media_info_is_drm(media_info_h media, bool *is_drm);

/**
 * @brief Checks whether the content file is 360 content.
 * @since_tizen 3.0
 *
 * @param[in] media The handle to the media info
 * @param[out] is_360 @c true if media is 360 content,
 *                    otherwise @c false if media is not 360 content
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 */
int media_info_is_360_content(media_info_h media, bool *is_360);

/**
 * @brief Gets the storage type of content file.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @param[in] media The handle to the media info
 * @param[out] storage_type The storage type of the content file
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 */
int media_info_get_storage_type(media_info_h media, media_content_storage_e *storage_type);

/**
 * @deprecated Deprecated since 4.0. \n
 *        This function does not guarantee independence of the played count value between applications. It is recommended that the value is managed by the application.
 * @brief Gets number which represents how many times given content has been played.
 * @since_tizen @if MOBILE 2.4 @elseif WEARABLE 3.0 @endif
 *
 * @param[in] media The handle to the media info
 * @param[out] played_count The counter of content played
 *
 * @return 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE Successful
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 */
int media_info_get_played_count(media_info_h media, int *played_count) TIZEN_DEPRECATED_API;

/**
 * @deprecated Deprecated since 4.0. \n
 *        This function does not guarantee independence of the played time value between applications. It is recommended that the value is managed by the application.
 * @brief Gets the content's played time parameter.
 * @details Function returns content's elapsed playback time parameter as period
 *          starting from the beginning of the track.
 * @since_tizen @if MOBILE 2.4 @elseif WEARABLE 3.0 @endif
 *
 * @param[in] media The handle to the media info
 * @param[out] played_time The elapsed time of the content
 *
 * @return 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE Successful
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 */
int media_info_get_played_time(media_info_h media, time_t *played_time) TIZEN_DEPRECATED_API;

/**
 * @brief Gets the media info from the media database.
 *
 * @details This function creates a new media handle from the media database by the given @a media_id.
 *          Media info will be created and filled with information.
 *
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @remarks The @a media should be released using media_info_destroy().
 *
 * @param[in] media_id The media ID
 * @param[out] media The handle to the media info
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #MEDIA_CONTENT_ERROR_DB_FAILED DB Operation failed
 * @retval #MEDIA_CONTENT_ERROR_DB_BUSY DB Operation busy
 *
 * @pre This function requires opened connection to content service by media_content_connect().
 *
 * @see media_content_connect()
 * @see media_info_destroy()
 */
int media_info_get_media_from_db(const char *media_id, media_info_h *media);

/**
 * @deprecated Deprecated since 4.0. \n
 *        This function does not guarantee independence of the played count value between applications. It is recommended that the value is managed by the application.
 * @brief Increments the played count to content meta handle.
 * @details You can increase the played(opened) count of the content file.
 * @since_tizen @if MOBILE 2.4 @elseif WEARABLE 3.0 @endif
 *
 * @param[in] media The handle to the media info
 *
 * @return 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE Successful
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 *
 * @post media_info_update_to_db()
 */
int media_info_increase_played_count(media_info_h media) TIZEN_DEPRECATED_API;

/**
 * @deprecated Deprecated since 4.0. \n
 *        This function does not guarantee independence of the played time value between applications. It is recommended that the value is managed by the application.
 * @brief Sets the played time to content meta handle.
 * @details You can set the latest played(opened) time of the content file. the latest played time to be set the current time on the system.
 * @since_tizen @if MOBILE 2.4 @elseif WEARABLE 3.0 @endif
 *
 * @param[in] media The handle to the media info
 *
 * @return 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE Successful
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 *
 * @post media_info_update_to_db()
 */
int media_info_set_played_time(media_info_h media) TIZEN_DEPRECATED_API;

/**
 * @deprecated Deprecated since 4.0. Redundant, use media_info_move_to_db() instead.
 * @brief Sets the file name including the extension of the content file.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @param[in] media The handle to the media info
 * @param[in] display_name The file name including the extension of the content file.
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 *
 * @post media_info_update_to_db().
 *
 */
int media_info_set_display_name(media_info_h media, const char *display_name) TIZEN_DEPRECATED_API;

/**
 * @deprecated Deprecated since 4.0. This data cannot be saved to the file.
 * @brief Sets the description of media info.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @param[in] media The handle to the media info
 * @param[in] description The description of the media info
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 *
 * @post media_info_update_to_db().
 *
 */
int media_info_set_description(media_info_h media, const char *description) TIZEN_DEPRECATED_API;

/**
 * @deprecated Deprecated since 4.0. This data cannot be saved to the file.
 * @brief Sets the longitude of media info.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @param[in] media The handle to the media info
 * @param[in] longitude The longitude of the media info
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 *
 * @post media_info_update_to_db().
 */
int media_info_set_longitude(media_info_h media, double longitude) TIZEN_DEPRECATED_API;

/**
 * @deprecated Deprecated since 4.0. This data cannot be saved to the file.
 * @brief Sets the latitude of media info.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @param[in] media The handle to the media info
 * @param[in] latitude The latitude of the media info
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 *
 * @post media_info_update_to_db().
 */
int media_info_set_latitude(media_info_h media, double latitude) TIZEN_DEPRECATED_API;

/**
 * @deprecated Deprecated since 4.0. This data cannot be saved to the file.
 * @brief Sets the altitude of media info.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @param[in] media The handle to the media info
 * @param[in] altitude The altitude of the media info
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 *
 * @post media_info_update_to_db().
 */
int media_info_set_altitude(media_info_h media, double altitude) TIZEN_DEPRECATED_API;

/**
 * @deprecated Deprecated since 4.0. This data cannot be saved to the file.
 * @brief Sets the weather of media info.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @param[in] media The handle to the media info
 * @param[in] weather The weather of the media info
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 *
 * @post media_info_update_to_db().
 *
 */
int media_info_set_weather(media_info_h media, const char *weather) TIZEN_DEPRECATED_API;

/**
 * @deprecated Deprecated since 4.0. This data cannot be saved to the file.
 * @brief Sets the rating of media info.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @param[in] media The handle to the media info
 * @param[in] rating The rating of the media info
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 *
 * @post media_info_update_to_db().
 */
int media_info_set_rating(media_info_h media, int rating) TIZEN_DEPRECATED_API;

/**
 * @brief Sets the favorite of media info.
 * @details This function can mark favorite of the media. If set to @c true, this fuction record the time of the change moment. \n
 *                So, If you use it in order parameter, you can sort the order of the time was a favorite. \n
 *                Or, if you use it in condition parameter, you can get the result of the favorite media.
 *
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @param[in] media The handle to the media info
 * @param[in] favorite Set @c true to set the media info as favorite,
 *                     otherwise set @c false to not set the media info as favorite
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 */
int media_info_set_favorite(media_info_h media, bool favorite);

/**
 * @deprecated Deprecated since 4.0. This data cannot be saved to the file.
 * @brief Sets the author of media info.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @param[in] media The handle to the media info
 * @param[in] author The author of the media info
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 */
int media_info_set_author(media_info_h media, const char *author) TIZEN_DEPRECATED_API;

/**
 * @deprecated Deprecated since 4.0. This data cannot be saved to the file.
 * @brief Sets the provider of media info.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @param[in] media The handle to the media info
 * @param[in] provider The provider of the media info
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 */
int media_info_set_provider(media_info_h media, const char *provider) TIZEN_DEPRECATED_API;

/**
 * @deprecated Deprecated since 4.0. Redundant, use media_info_move_to_db() instead.
 * @brief Sets the content name of media info.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @param[in] media The handle to the media info
 * @param[in] content_name The content name of the media info
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 *
 * @post media_info_update_to_db()
 */
int media_info_set_content_name(media_info_h media, const char *content_name) TIZEN_DEPRECATED_API;

/**
 * @deprecated Deprecated since 4.0. This data cannot be saved to the file.
 * @brief Sets the category of media info.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @param[in] media The handle to the media info
 * @param[in] category The category of the media info
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 *
 * @post media_info_update_to_db()
 */
int media_info_set_category(media_info_h media, const char *category) TIZEN_DEPRECATED_API;

/**
 * @deprecated Deprecated since 4.0. This data cannot be saved to the file.
 * @brief Sets the location tag of media info.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @param[in] media The handle to the media info
 * @param[in] location_tag The location of the media info
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 *
 * @post media_info_update_to_db()
 */
int media_info_set_location_tag(media_info_h media, const char *location_tag) TIZEN_DEPRECATED_API;

/**
 * @deprecated Deprecated since 4.0. This data cannot be saved to the file.
 * @brief Sets the age rating of media info.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @param[in] media The handle to the media info
 * @param[in] age_rating The age rating of the media info
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 *
 * @post media_info_update_to_db()
 */
int media_info_set_age_rating(media_info_h media, const char *age_rating) TIZEN_DEPRECATED_API;

/**
 * @deprecated Deprecated since 4.0. Use media_tag_insert_to_db() instead.
 * @brief Sets the keyword of media info.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @param[in] media The handle to the media info
 * @param[in] keyword The keyword of the media info
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 *
 * @post media_info_update_to_db()
 */
int media_info_set_keyword(media_info_h media, const char *keyword) TIZEN_DEPRECATED_API;

/**
 * @brief Updates the media info to the media database.
 *
 * @details The function updates the given media info in the media database.
 *
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 * @privlevel public
 * @privilege %http://tizen.org/privilege/content.write
 *
 * @remarks The function should be called after any change in media, to be updated to the media
 *          database. For example, after using media_info_set_favorite()
 *          for setting the name of the media, the media_info_update_to_db() function should be called so as to update
 *          the given media info attributes in the media database.
 *
 * @param[in] media The handle to the media info
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_INVALID_OPERATION Invalid operation
 * @retval #MEDIA_CONTENT_ERROR_DB_FAILED         DB Operation failed
 * @retval #MEDIA_CONTENT_ERROR_DB_BUSY           DB Operation busy
 * @retval #MEDIA_CONTENT_ERROR_NETWORK           Network fail
 * @retval #MEDIA_CONTENT_ERROR_PERMISSION_DENIED Permission denied
 *
 * @pre This function requires opened connection to content service by media_content_connect().
 *
 * @see media_content_connect()
 * @see media_info_set_favorite()
 */
int media_info_update_to_db(media_info_h media);

/**
 * @deprecated Deprecated since 4.0. Use media_content_scan_file() instead.
 * @brief Refreshes the media metadata to the media database.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @privlevel public
 * @privilege %http://tizen.org/privilege/content.write \n
 *                   %http://tizen.org/privilege/mediastorage \n
 *                   %http://tizen.org/privilege/externalstorage
 *
 * @remarks You must add privilege http://tizen.org/privilege/content.write. And You add more privilege depending on your choice of contents path. \n
 *                   If you want to access only internal storage by using this function, you should add privilege http://tizen.org/privilege/mediastorage. \n
 *                   Or if you want to access only external storage by using this function, you should add privilege http://tizen.org/privilege/externalstorage. \n
 *                   If you can access both storage, you should add all privilege.
 *
 * @param[in] media_id The media ID
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_INVALID_OPERATION Invalid operation
 * @retval #MEDIA_CONTENT_ERROR_DB_FAILED         DB Operation failed
 * @retval #MEDIA_CONTENT_ERROR_DB_BUSY           DB Operation busy
 * @retval #MEDIA_CONTENT_ERROR_NETWORK           Network fail
 * @retval #MEDIA_CONTENT_ERROR_PERMISSION_DENIED Permission denied
 *
 * @pre This function requires opened connection to content service by media_content_connect().
 *
 * @see media_content_connect()
 */
int media_info_refresh_metadata_to_db(const char *media_id) TIZEN_DEPRECATED_API;

/**
 * @deprecated Deprecated since 4.0. This data cannot be saved to the file.
 * @brief Sets the added time of media info.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @param[in] media The handle to the media info
 * @param[in] added_time The added time of the media info
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 *
 * @post media_info_update_to_db()
 */
int media_info_set_added_time(media_info_h media, time_t added_time) TIZEN_DEPRECATED_API;

/**
 * @brief Moves the media info to the given destination path in the media database.
 * @details After moving or renaming a file in the filesystem, call this function to update the database. \n
 *               If the source path and destination path are the same, then this function does nothing.
 *               If you want to refresh media information, you should use media_content_scan_file() function.
 *
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @privlevel public
 * @privilege %http://tizen.org/privilege/content.write \n
 *                   %http://tizen.org/privilege/mediastorage \n
 *                   %http://tizen.org/privilege/externalstorage
 *
 * @remarks You must add privilege http://tizen.org/privilege/content.write. And You add more privilege depending on your choice of contents path. \n
 *                   If you want to access only internal storage by using this function, you should add privilege http://tizen.org/privilege/mediastorage. \n
 *                   Or if you want to access only external storage by using this function, you should add privilege http://tizen.org/privilege/externalstorage. \n
 *                   If you can access both storage, you should add all privilege. \n
 *                   This function does not support USB storage.
 *
 * @param[in] media The handle to the media info
 * @param[in] dst_path The path of destination
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter (Especially, if the request is duplicated, this error returns.)
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_INVALID_OPERATION Invalid operation
 * @retval #MEDIA_CONTENT_ERROR_DB_FAILED         DB Operation failed
 * @retval #MEDIA_CONTENT_ERROR_DB_BUSY           DB Operation busy
 * @retval #MEDIA_CONTENT_ERROR_NETWORK           Network fail
 * @retval #MEDIA_CONTENT_ERROR_PERMISSION_DENIED Permission denied
 *
 * @pre This function requires opened connection to content service by media_content_connect().
 *
 * @see media_content_connect()
 * @see media_content_scan_folder()
 */
int media_info_move_to_db(media_info_h media, const char* dst_path);

/**
 * @brief Creates a thumbnail file for the given media, asynchronously.
 * @details This function creates an thumbnail file for given media item and calls @a callback for completion of creating the thumbnail.
 *          If a thumbnail already exists for the given media, then the path of thumbnail will be returned in callback function. \n
 *          Since 3.0, a thumbnail is not automatically extracted during media scanning. \n
 *          Therefore, if there exists no thumbnail for the given media, you MUST call this function to create a thumbnail.
 *
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @privlevel public
 * @privilege %http://tizen.org/privilege/content.write
 *
 * @remarks If you want to destroy media handle before callback invoked, you must cancel thumbnail request by using media_info_cancel_thumbnail() \n
 *          Since 3.0, if creation of a thumbnail is failed, empty string will be passed through media_thumbnail_completed_cb().
 *          Items in external storage except MMC not supported.
 *
 * @param[in] media The handle to the media info
 * @param[in] callback The callback function to be invoked
 * @param[in] user_data The user data to be passed to the callback function
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_INVALID_OPERATION Invalid operation
 * @retval #MEDIA_CONTENT_ERROR_DB_FAILED         DB Operation failed
 * @retval #MEDIA_CONTENT_ERROR_DB_BUSY           DB Operation busy
 * @retval #MEDIA_CONTENT_ERROR_NETWORK           Network fail
 * @retval #MEDIA_CONTENT_ERROR_PERMISSION_DENIED Permission denied
 *
 * @pre This function requires opened connection to content service by media_content_connect().
 *
 * @see media_content_connect()
 */
int media_info_create_thumbnail(media_info_h media, media_thumbnail_completed_cb callback, void *user_data);

/**
 * @brief Cancels the creation of thumbnail file for the given media.
 * @since_tizen @if MOBILE 2.3 @elseif WEARABLE 2.3.1 @endif
 *
 * @privlevel public
 * @privilege %http://tizen.org/privilege/content.write
 *
 * @remarks If you request cancel for the already thumbnail created media, this function return MEDIA_CONTENT_ERROR_INVALID_OPERATION
 *
 * @param[in] media The handle to the media info
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #MEDIA_CONTENT_ERROR_PERMISSION_DENIED Permission denied
 * @retval #MEDIA_CONTENT_ERROR_INVALID_OPERATION Invalid operation
 *
 * @pre This function requires opened connection to content service by media_content_connect().
 * @see media_content_connect()
 */
int media_info_cancel_thumbnail(media_info_h media);

/**
 * @ingroup CAPI_CONTENT_MEDIA_FACE_DETECTION_MODULE
 * @brief Starts face detection for the given image, asynchronously.
 * @details This function detects faces for given image item and calls the given callback function when the detection is completed. \n
 *          The given callback function is called when the detection is completed. \n
 *          To obtain the detected faces, call the media_info_foreach_face_from_db() function.
 *
 * @since_tizen 3.0
 *
 * @privlevel public
 * @privilege %http://tizen.org/privilege/content.write \n
 *                   %http://tizen.org/privilege/mediastorage
 *
 * @remarks If you want to destroy the media handle before callback invoked, you must cancel the face detection request by using media_info_cancel_face_detection(). \n
 *          If face detection fails, the @a face_count argument in media_face_detection_completed_cb() will be set to 0.
 *          Media items in external storage are not supported, with the exception of MMC items.
 *
 * @param[in] media The handle to the media info
 * @param[in] callback The callback function to be invoked when detection is completed
 * @param[in] user_data The user data to be passed to the callback function
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_INVALID_OPERATION Invalid operation
 * @retval #MEDIA_CONTENT_ERROR_DB_FAILED         DB Operation failed
 * @retval #MEDIA_CONTENT_ERROR_DB_BUSY           DB Operation busy
 * @retval #MEDIA_CONTENT_ERROR_NETWORK           Network fail
 * @retval #MEDIA_CONTENT_ERROR_PERMISSION_DENIED Permission denied
 * @retval #MEDIA_CONTENT_ERROR_NOT_SUPPORTED     Not supported
 *
 * @pre This function requires opened connection to content service by media_content_connect().
 *
 * @see media_content_connect()
 * @see media_info_cancel_face_detection()
 */
int media_info_start_face_detection(media_info_h media, media_face_detection_completed_cb callback, void *user_data);

/**
 * @ingroup CAPI_CONTENT_MEDIA_FACE_DETECTION_MODULE
 * @brief Cancels face detection of image for the given media.
 * @details This function cancels face detection for given media item. \n
 *          If you cancel face detection request before callback is invoked, the callback registered by media_info_start_face_detection() function will not be invoked.
 *
 * @since_tizen 3.0
 *
 * @privlevel public
 * @privilege %http://tizen.org/privilege/content.write
 *
 * @remarks If face detection is already done when you request the cancellation, this function return MEDIA_CONTENT_ERROR_INVALID_OPERATION
 *
 * @param[in] media The handle to the media info
 *
 * @return @c 0 on success,
 *         otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE              Successful
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #MEDIA_CONTENT_ERROR_PERMISSION_DENIED Permission denied
 * @retval #MEDIA_CONTENT_ERROR_INVALID_OPERATION Invalid operation
 *
 * @pre This function requires opened connection to content service by media_content_connect().
 *
 * @see media_content_connect()
 * @see media_info_start_face_detection()
 */
int media_info_cancel_face_detection(media_info_h media);

/**
 * @deprecated Deprecated since 4.0. Use media_info_insert_to_db() instead.
 * @brief Creates the media info handle.
 * @details If the information about the file that is already known, you can use this function to generate empty handler.
 * @since_tizen 3.0
 *
 *
 * @privilege %http://tizen.org/privilege/mediastorage \n
 *               %http://tizen.org/privilege/externalstorage
 *
 * @remarks The @a media should be released using media_info_destroy().
 * @remarks If you want to access only internal storage with this function, you should add privilege http://tizen.org/privilege/mediastorage. \n
 *                   Or if you want to access only external storage with this function, you should add privilege http://tizen.org/privilege/externalstorage. \n
 *                   If you can access both storage, you must add all privilege.
 *
 * @param[in] path The path to create the media info handle
 * @param[out] media The handle to the media info
 *
 * @return @c 0 on success,
 *           otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE Successful
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #MEDIA_CONTENT_ERROR_DB_FAILED DB operation failed
 * @retval #MEDIA_CONTENT_ERROR_PERMISSION_DENIED Permission denied
 * @retval #MEDIA_CONTENT_ERROR_INVALID_OPERATION Invalid operation
 *
 * @see media_info_destroy()
 */
int media_info_create(const char *path, media_info_h *media) TIZEN_DEPRECATED_API;

/**
 * @deprecated Deprecated since 4.0. Use media_info_insert_to_db() instead.
 * @brief Inserts media info to database with media info data.
 * @since_tizen 3.0
 *
 * @privlevel public
 * @privilege %http://tizen.org/privilege/content.write
 *
 * @remarks The @a media should be released using media_info_destroy().
 *
 * @param[in] media The handle to the media info
 *
 * @return @c 0 on success,
 *           otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE Successful
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #MEDIA_CONTENT_ERROR_DB_FAILED DB operation failed
 * @retval #MEDIA_CONTENT_ERROR_PERMISSION_DENIED Permission denied
 */
int media_info_insert_to_db_with_data(media_info_h media) TIZEN_DEPRECATED_API;

/**
 * @deprecated Deprecated since 4.0.
 * @brief Sets the title of media info handle.
 * @since_tizen 3.0
 *
 * @param[in] media The handle to the media info
 * @param[in] title The title of media info handle
 *
 * @return @c 0 on success,
 *           otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE Successful
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 */
int media_info_set_title(media_info_h media, const char *title) TIZEN_DEPRECATED_API;

/**
 * @deprecated Deprecated since 4.0.
 * @brief Sets the album of media info handle.
 * @since_tizen 3.0
 *
 * @param[in] media The handle to the media info
 * @param[in] album The album of the media
 *
 * @return @c 0 on success,
 *           otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE Successful
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 */
int media_info_set_album(media_info_h media, const char *album) TIZEN_DEPRECATED_API;

/**
 * @deprecated Deprecated since 4.0.
 * @brief Sets the artist of media info handle.
 * @since_tizen 3.0
 *
 * @param[in] media The handle to the media info
 * @param[in] artist The artist of media info handle
 *
 * @return @c 0 on success,
 *           otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE Successful
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 */
int media_info_set_artist(media_info_h media, const char *artist) TIZEN_DEPRECATED_API;

/**
 * @deprecated Deprecated since 4.0.
 * @brief Sets the genre of media info handle.
 * @since_tizen 3.0
 *
 * @param[in] media The handle to the media info
 * @param[in] genre The genre of media info handle
 *
 * @return @c 0 on success,
 *           otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE Successful
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 */
int media_info_set_genre(media_info_h media, const char *genre) TIZEN_DEPRECATED_API;

/**
 * @deprecated Deprecated since 4.0.
 * @brief Sets the recorded date of media info handle.
 * @since_tizen 3.0
 *
 * @param[in] media The handle to the media info
 * @param[in] recorded_date The recorded date of media info handle
 *
 * @return @c 0 on success,
 *           otherwise a negative error value
 *
 * @retval #MEDIA_CONTENT_ERROR_NONE Successful
 * @retval #MEDIA_CONTENT_ERROR_OUT_OF_MEMORY     Out of memory
 * @retval #MEDIA_CONTENT_ERROR_INVALID_PARAMETER Invalid parameter
 */
int media_info_set_recorded_date(media_info_h media, const char *recorded_date) TIZEN_DEPRECATED_API;


/**
 * @}
 */


#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* __TIZEN_CONTENT_MEDIA_INFO_H__ */
