/*
* Copyright (c) 2011 Samsung Electronics Co., Ltd All Rights Reserved
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
* http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

#ifndef __TIZEN_MULTIMEDIA_IMAGE_UTIL_ENCODE_H__
#define __TIZEN_MULTIMEDIA_IMAGE_UTIL_ENCODE_H__

#include <image_util_type.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
* @addtogroup CAPI_MEDIA_IMAGE_UTIL_ENCODE_DECODE_MODULE
* @{
*/

/**
* @brief Creates a handle of image util encoding.
* @details This function creates a handle of image util encoding.
* @since_tizen 3.0
*
* @remarks The @a handle should be released using image_util_encode_destroy().
*
* @param[in] image_type The type of output image for which to create encode handle.
* @param[out] handle The handle of image util encoding
*
* @return @c 0 on success,
*            otherwise a negative error value
*
* @retval #IMAGE_UTIL_ERROR_NONE Successful
* @retval #IMAGE_UTIL_ERROR_INVALID_PARAMETER Invalid parameter
* @retval #IMAGE_UTIL_ERROR_OUT_OF_MEMORY Out of memory
*
* @post image_util_encode_run_to_file()
* @post image_util_encode_run_to_buffer()
* @post image_util_encode_run_async_to_file()
* @post image_util_encode_run_async_to_buffer()
* @post image_util_encode_destroy()
*
* @see image_util_encode_run_to_file()
* @see image_util_encode_run_to_buffer()
* @see image_util_encode_run_async_to_file()
* @see image_util_encode_run_async_to_buffer()
* @see image_util_encode_destroy()
*
*/
int image_util_encode_create(image_util_type_e image_type, image_util_encode_h *handle);

/**
* @deprecated Deprecated since 5.5. Use image_util_create_image() instead.
* @brief Sets the resolution of the encoded image.
* @since_tizen 3.0
*
* @remarks This should be called before calling image_util_encode_run().\n
*               While encoding animated GIF image, resolution should be set for each frame.
*
* @param[in] handle The handle of image util encoding
* @param[in] width Width of the original image
* @param[in] height Height of the original image
*
* @return @c 0 on success,
*                 otherwise a negative error value
*
* @retval #IMAGE_UTIL_ERROR_NONE Successful
* @retval #IMAGE_UTIL_ERROR_INVALID_PARAMETER Invalid parameter
* @retval #IMAGE_UTIL_ERROR_INVALID_OPERATION Invalid operation
*
* @pre image_util_encode_create()
*
* @post image_util_encode_run() / image_util_encode_run_async()
* @post image_util_encode_destroy()
*
* @see image_util_encode_create()
* @see image_util_encode_set_input_buffer()
* @see image_util_encode_set_output_path()
* @see image_util_encode_set_output_buffer()
* @see image_util_encode_run()
* @see image_util_encode_run_async()
* @see image_util_encode_destroy()
*/
int image_util_encode_set_resolution(image_util_encode_h handle, unsigned long width, unsigned long height) TIZEN_DEPRECATED_API;

/**
* @deprecated Deprecated since 5.5. Use image_util_create_image() instead.
* @brief Sets the colorspace format for image encoding.
* @since_tizen 3.0
*
* @remarks The default colorspace is #IMAGE_UTIL_COLORSPACE_RGBA8888.\n
*             Use image_util_foreach_supported_colorspace() to get supported colorspaces for each image format.\n
*             Errors would be returned if not supported.
*
* @param[in] handle The handle of image util encoding
* @param[in] colorspace The colorspace of the input image to encode
*
* @return @c 0 on success,
*                 otherwise a negative error value
*
* @retval #IMAGE_UTIL_ERROR_NONE Successful
* @retval #IMAGE_UTIL_ERROR_INVALID_PARAMETER Invalid parameter
* @retval #IMAGE_UTIL_ERROR_NOT_SUPPORTED_FORMAT Not supported format
*
* @pre image_util_encode_create()
*
* @post image_util_encode_run() / image_util_encode_run_async()
* @post image_util_encode_destroy()
*
* @see image_util_supported_colorspace_cb()
* @see image_util_foreach_supported_colorspace()
* @see image_util_encode_create()
* @see image_util_encode_set_resolution()
* @see image_util_encode_set_input_buffer()
* @see image_util_encode_set_output_path()
* @see image_util_encode_set_output_buffer()
* @see image_util_encode_run()
* @see image_util_encode_run_async()
* @see image_util_encode_destroy()
*/
int image_util_encode_set_colorspace(image_util_encode_h handle, image_util_colorspace_e colorspace) TIZEN_DEPRECATED_API;

/**
* @brief Sets the quality for image encoding.
* @since_tizen 3.0
*
* @remarks If application does not set this, then by default quality of 75 is set.\n
*              Quality is supported for JPEG format. #IMAGE_UTIL_ERROR_NOT_SUPPORTED_FORMAT will be returned for other formats.
*
* @param[in] handle The handle of image util encoding
* @param[in] quality Encoding quality from 1~100
*
* @return @c 0 on success,
*                 otherwise a negative error value
*
* @retval #IMAGE_UTIL_ERROR_NONE Successful
* @retval #IMAGE_UTIL_ERROR_INVALID_PARAMETER Invalid parameter
* @retval #IMAGE_UTIL_ERROR_NOT_SUPPORTED_FORMAT Not supported format
*
* @pre image_util_encode_create()
*
* @post image_util_encode_run() / image_util_encode_run_async()
* @post image_util_encode_destroy()
*
* @see image_util_encode_create()
* @see image_util_encode_run_to_file()
* @see image_util_encode_run_to_buffer()
* @see image_util_encode_run_async_to_file()
* @see image_util_encode_run_async_to_buffer()
* @see image_util_encode_destroy()
*/
int image_util_encode_set_quality(image_util_encode_h handle, int quality);

/**
* @brief Sets the compression value of PNG image encoding(0~9).
* @since_tizen 3.0
*
* @remarks If application does not set this, then the default value is #IMAGE_UTIL_PNG_COMPRESSION_6.
*
* @param[in] handle The handle of image util encoding
* @param[in] compression The compression value valid from 0~9
*
* @return @c 0 on success,
*                 otherwise a negative error value
*
* @retval #IMAGE_UTIL_ERROR_NONE Successful
* @retval #IMAGE_UTIL_ERROR_INVALID_PARAMETER Invalid parameter
* @retval #IMAGE_UTIL_ERROR_INVALID_OPERATION Invalid operation
* @retval #IMAGE_UTIL_ERROR_NOT_SUPPORTED_FORMAT Not supported format
*
* @pre image_util_encode_create()
*
* @post image_util_encode_run() / image_util_encode_run_async()
* @post image_util_encode_destroy()
*
* @see image_util_encode_create()
* @see image_util_encode_run_to_file()
* @see image_util_encode_run_to_buffer()
* @see image_util_encode_run_async_to_file()
* @see image_util_encode_run_async_to_buffer()
* @see image_util_encode_destroy()
*/
int image_util_encode_set_png_compression(image_util_encode_h handle, image_util_png_compression_e compression);

/**
* @deprecated Deprecated since 7.0. Use image_util_encode_set_lossless() instead.
* @brief Sets lossless compression of WEBP images.
* @since_tizen 6.0
*
* @remarks If application does not set this, then the default value is @c false.
*
* @param[in] handle The handle of image util encoding
* @param[in] lossless The flag determining whether the compression is lossless or lossy: @c true for lossless, @c false for lossy
*
* @return @c 0 on success,
*                 otherwise a negative error value
*
* @retval #IMAGE_UTIL_ERROR_NONE Successful
* @retval #IMAGE_UTIL_ERROR_INVALID_PARAMETER Invalid parameter
* @retval #IMAGE_UTIL_ERROR_INVALID_OPERATION Invalid operation
* @retval #IMAGE_UTIL_ERROR_NOT_SUPPORTED_FORMAT Not supported format
*
* @pre image_util_encode_create()
*
* @post image_util_encode_run() / image_util_encode_run_async()
* @post image_util_encode_destroy()
*
* @see image_util_encode_create()
* @see image_util_encode_run_to_file()
* @see image_util_encode_run_to_buffer()
* @see image_util_encode_run_async_to_file()
* @see image_util_encode_run_async_to_buffer()
* @see image_util_encode_destroy()
*/
int image_util_encode_set_webp_lossless(image_util_encode_h handle, bool lossless) TIZEN_DEPRECATED_API;

/**
* @brief Sets lossless compression of WEBP and JPEG-XL images.
* @since_tizen 7.0
*
* @remarks If application does not set this, then the default value is @c false.
*
* @param[in] handle The handle of image util encoding
* @param[in] lossless The flag determining whether the compression is lossless or lossy: @c true for lossless, @c false for lossy
*
* @return @c 0 on success,
*                 otherwise a negative error value
*
* @retval #IMAGE_UTIL_ERROR_NONE Successful
* @retval #IMAGE_UTIL_ERROR_INVALID_PARAMETER Invalid parameter
* @retval #IMAGE_UTIL_ERROR_INVALID_OPERATION Invalid operation
* @retval #IMAGE_UTIL_ERROR_NOT_SUPPORTED_FORMAT Not supported format
*
* @pre image_util_encode_create()
*
* @post image_util_encode_run() / image_util_encode_run_async()
* @post image_util_encode_destroy()
*
* @see image_util_encode_create()
* @see image_util_encode_run_to_file()
* @see image_util_encode_run_to_buffer()
* @see image_util_encode_run_async_to_file()
* @see image_util_encode_run_async_to_buffer()
* @see image_util_encode_destroy()
*/
int image_util_encode_set_lossless(image_util_encode_h handle, bool lossless);

/**
* @deprecated Deprecated since 5.5. Use image_util_agif_encode_add_frame() instead.
* @brief Sets the time delay between each frame in the encoded animated GIF image.
* @since_tizen 3.0
*
* @remarks In case animated GIF image if this is not set then there will be no delay between each frame.\n
*             This should be set for each frame in the animated GIF image.\n
*             This can be set a different value for each frame, which results in different delay time between different frames.
*
* @param[in] handle The handle of image util encoding
* @param[in] delay_time Time delay between each frame in the encoded image, in 0.01sec units.
*
* @return @c 0 on success,
*                 otherwise a negative error value
*
* @retval #IMAGE_UTIL_ERROR_NONE Successful
* @retval #IMAGE_UTIL_ERROR_INVALID_PARAMETER Invalid parameter
* @retval #IMAGE_UTIL_ERROR_NOT_SUPPORTED_FORMAT Not supported format
*
* @pre image_util_encode_create()
*
* @post image_util_encode_run() / image_util_encode_run_async()
* @post image_util_encode_destroy()
*
* @see image_util_encode_create()
* @see image_util_encode_set_resolution()
* @see image_util_encode_set_input_buffer()
* @see image_util_encode_set_output_path()
* @see image_util_encode_set_output_buffer()
* @see image_util_encode_run()
* @see image_util_encode_run_async()
* @see image_util_encode_destroy()
*/
int image_util_encode_set_gif_frame_delay_time(image_util_encode_h handle, unsigned long long delay_time) TIZEN_DEPRECATED_API;

/**
* @deprecated Deprecated since 5.5. Use image_util_create_image() instead.
* @brief Sets the input buffer from which to encode.
* @since_tizen 3.0
*
* @remarks Either image_util_encode_set_output_path() or image_util_encode_set_output_buffer() should be set.\n
*             By default the input buffer colorspace will be considered as #IMAGE_UTIL_COLORSPACE_RGBA8888.\n
*             Use image_util_encode_set_colorspace() to change the colorspace.\n
*             While encoding animated GIF image, input buffer should be set for each frame.
*
* @param[in] handle The handle of image util decoding
* @param[in] src_buffer The input image buffer
*
* @return @c 0 on success,
*                 otherwise a negative error value
*
* @retval #IMAGE_UTIL_ERROR_NONE Successful
* @retval #IMAGE_UTIL_ERROR_INVALID_PARAMETER Invalid parameter
* @retval #IMAGE_UTIL_ERROR_INVALID_OPERATION Invalid operation
*
* @pre image_util_encode_create()
*
* @post image_util_encode_run() / image_util_encode_run_async()
* @post image_util_encode_destroy()
*
* @see image_util_encode_create()
* @see image_util_encode_set_resolution()
* @see image_util_encode_set_output_path()
* @see image_util_encode_set_output_buffer()
* @see image_util_encode_run()
* @see image_util_encode_run_async()
* @see image_util_encode_destroy()
*/
int image_util_encode_set_input_buffer(image_util_encode_h handle, const unsigned char *src_buffer) TIZEN_DEPRECATED_API;

/**
* @deprecated Deprecated since 5.5. Use image_util_encode_run_to_file() or image_util_encode_run_async_to_file() instead.
* @brief Sets the output path to which to encoded buffer will be written to.
* @since_tizen 3.0
*
* @remarks One of image_util_encode_set_output_path() or image_util_encode_set_output_buffer() should be set.\n
*             If both are set then the latest output set is considered.\n
*             %http://tizen.org/privilege/mediastorage is needed if input or output path are relevant to media storage.\n
*             %http://tizen.org/privilege/externalstorage is needed if input or output path are relevant to external storage.
*
* @param[in] handle The handle of image util encoding
* @param[in] path The output file path
*
* @return @c 0 on success,
*                 otherwise a negative error value
*
* @retval #IMAGE_UTIL_ERROR_NONE Successful
* @retval #IMAGE_UTIL_ERROR_PERMISSION_DENIED The application does not have the privilege to call this function
* @retval #IMAGE_UTIL_ERROR_INVALID_PARAMETER Invalid parameter
* @retval #IMAGE_UTIL_ERROR_INVALID_OPERATION Invalid operation
* @retval #IMAGE_UTIL_ERROR_NO_SUCH_FILE No such file
*
* @pre image_util_encode_create()
*
* @post image_util_encode_run() / image_util_encode_run_async()
* @post image_util_encode_destroy()
*
* @see image_util_encode_create()
* @see image_util_encode_set_resolution()
* @see image_util_encode_set_input_buffer()
* @see image_util_encode_run()
* @see image_util_encode_run_async()
* @see image_util_encode_destroy()
*/
int image_util_encode_set_output_path(image_util_encode_h handle, const char *path) TIZEN_DEPRECATED_API;

/**
* @deprecated Deprecated since 5.5. Use image_util_encode_run_to_buffer() or image_util_encode_run_async_to_buffer() instead.
* @brief Sets the output buffer to which to encoded buffer will be written to.
* @since_tizen 3.0
*
* @remarks One of image_util_encode_set_output_path() or image_util_encode_set_output_buffer() should be set.\n
*           If both are set then the latest output set is considered.\n
*           Before 4.0, output buffer setting is not supported for BMP. #IMAGE_UTIL_ERROR_NOT_SUPPORTED_FORMAT will be returned for BMP.\n
*           Since 4.0, output buffer setting has been supported for BMP. Applications can set the output buffer to write encoded BMP.\n
*           In case of GIF encoding, the output buffer will be completely available only after image_util_encode_destroy().
*           @a dst_buffer should be released after @c image_util_encode_run() or @c image_util_encode_run_async().
*
* @param[in] handle The handle of image util encoding
* @param[in] dst_buffer The output image buffer
*
* @return @c 0 on success,
*                 otherwise a negative error value
*
* @retval #IMAGE_UTIL_ERROR_NONE Successful
* @retval #IMAGE_UTIL_ERROR_INVALID_PARAMETER Invalid parameter
* @retval #IMAGE_UTIL_ERROR_INVALID_OPERATION Invalid operation
* @retval #IMAGE_UTIL_ERROR_NOT_SUPPORTED_FORMAT Not supported format
*
* @pre image_util_encode_create()
*
* @post image_util_encode_run() / image_util_encode_run_async()
* @post image_util_encode_destroy()
*
* @see image_util_encode_create()
* @see image_util_encode_set_resolution()
* @see image_util_encode_set_input_buffer()
* @see image_util_encode_run()
* @see image_util_encode_run_async()
* @see image_util_encode_destroy()
*/
int image_util_encode_set_output_buffer(image_util_encode_h handle, unsigned char **dst_buffer) TIZEN_DEPRECATED_API;

/**
* @deprecated Deprecated since 5.5. Use image_util_encode_run_to_file() or image_util_encode_run_to_buffer() instead.
* @brief Starts encoding of the image and fills the output buffer, set using image_util_encode_set_output_buffer() or image_util_encode_set_output_path().
* @since_tizen 3.0
*
* @remarks The output will be stored in the pointer set to image_util_encode_set_output_buffer() or image_util_encode_set_output_path().\n
*          The function executes synchronously.\n
*          When any of the functions at the pre-condition are not called, #IMAGE_UTIL_ERROR_INVALID_PARAMETER is returned.\n
*          In case of animated GIF encoding, image_util_encode_set_resolution(), image_util_encode_set_input_buffer() and\n
*          image_util_encode_set_gif_frame_delay_time() MUST be called for each frame.
*          In case of animated GIF encoding, call image_util_encode_run() for each frame to encode progressively.
*
* @param[in] handle The handle of image util encoding
* @param[out] size Size of the encoded image
*
* @return @c 0 on success,
*                 otherwise a negative error value
*
* @retval #IMAGE_UTIL_ERROR_NONE Successful
* @retval #IMAGE_UTIL_ERROR_INVALID_PARAMETER Invalid parameter
* @retval #IMAGE_UTIL_ERROR_INVALID_OPERATION Invalid operation
*
* @pre image_util_encode_create()
* @pre image_util_encode_set_resolution()
* @pre image_util_encode_set_input_buffer()
* @pre image_util_encode_set_output_buffer() / image_util_encode_set_output_path()
*
* @post image_util_encode_destroy()
*
* @see image_util_encode_create()
* @see image_util_encode_set_resolution()
* @see image_util_encode_set_input_buffer()
* @see image_util_encode_set_gif_frame_delay_time()
* @see image_util_encode_set_output_path()
* @see image_util_encode_set_output_buffer()
* @see image_util_encode_destroy()
*/
int image_util_encode_run(image_util_encode_h handle, unsigned long long *size) TIZEN_DEPRECATED_API;

/**
* @deprecated Deprecated since 5.5. Use image_util_encode_run_async_to_file() or image_util_encode_run_async_to_buffer() instead.
* @brief Starts encoding of the image and fills the output buffer, set using image_util_encode_set_output_buffer() or image_util_encode_set_output_path().
* @since_tizen 3.0
*
* @remarks The output will be stored in the pointer set to image_util_encode_set_output_buffer() or image_util_encode_set_output_path().\n
*          The function executes asynchronously, which contains complete callback.\n
*          When any of the functions at the pre-condition are not called, #IMAGE_UTIL_ERROR_INVALID_PARAMETER is returned.\n
*          In case of animated GIF encoding, image_util_encode_set_resolution(), image_util_encode_set_input_buffer() and\n
*          image_util_encode_set_gif_frame_delay_time() MUST be called for each frame.
*          In case of animated GIF encoding, call image_util_encode_run_async() for each frame to encode progressively.
*
* @param[in] handle The handle of image util encoding
* @param[in] callback The callback function to be invoked
* @param[in] user_data The user data to be passed to the callback function
*
* @return @c 0 on success,
*                 otherwise a negative error value
*
* @retval #IMAGE_UTIL_ERROR_NONE Successful
* @retval #IMAGE_UTIL_ERROR_INVALID_PARAMETER Invalid parameter
* @retval #IMAGE_UTIL_ERROR_INVALID_OPERATION Invalid operation
*
* @pre image_util_encode_create()
* @pre image_util_encode_set_resolution()
* @pre image_util_encode_set_input_buffer()
* @pre image_util_encode_set_output_buffer() / image_util_encode_set_output_path()
*
* @post image_util_encode_destroy()
*
* @see image_util_encode_create()
* @see image_util_encode_set_resolution()
* @see image_util_encode_set_input_buffer()
* @see image_util_encode_set_gif_frame_delay_time()
* @see image_util_encode_set_output_path()
* @see image_util_encode_set_output_buffer()
* @see image_util_encode_destroy()
*/
int image_util_encode_run_async(image_util_encode_h handle, image_util_encode_completed_cb callback, void *user_data) TIZEN_DEPRECATED_API;

/**
* @brief Encodes the image and save it to a file in the given encoding handle.
* @details The function executes synchronously.
* @since_tizen 5.5
*
* @remarks If any functions at the pre-condition are not called first, #IMAGE_UTIL_ERROR_INVALID_PARAMETER is returned.\n
*          The only supported colorspace for BMP and GIF is #IMAGE_UTIL_COLORSPACE_RGBA8888.\n
*          To get supported colorspaces for JPEG format, you can use image_util_foreach_supported_colorspace().\n
*          %http://tizen.org/privilege/mediastorage is required if @a file_path value is media storage.\n
*          %http://tizen.org/privilege/externalstorage is required if @a file_path value is external storage.\n
*          Before 6.0, #IMAGE_UTIL_COLORSPACE_RGBA8888 was the only supported colorspace for PNG. But since 6.0, this module supports more colorspaces for PNG.
*          To get supported colorspaces for PNG, you can use image_util_foreach_supported_colorspace().\n
*          Since 6.0, this module supports WEBP, you can use image_util_foreach_supported_colorspace() to get supported colorspaces for WEBP.\n
*          Since 7.0, this module supports JPEG-XL, you can use image_util_foreach_supported_colorspace() to get supported colorspaces for JPEG-XL.
*
* @param[in] handle The handle of image util encoding
* @param[in] image The image handle for encoding
* @param[in] file_path The file path for encoding image
*
* @return @c 0 on success,
*                 otherwise a negative error value
*
* @retval #IMAGE_UTIL_ERROR_NONE Successful
* @retval #IMAGE_UTIL_ERROR_PERMISSION_DENIED Permission denied
* @retval #IMAGE_UTIL_ERROR_INVALID_PARAMETER Invalid parameter
* @retval #IMAGE_UTIL_ERROR_INVALID_OPERATION Invalid operation
* @retval #IMAGE_UTIL_ERROR_OUT_OF_MEMORY Out of memory
* @retval #IMAGE_UTIL_ERROR_NOT_SUPPORTED_FORMAT Not supported format
*
* @pre image_util_encode_create()
*
* @post image_util_encode_destroy()
*
* @see image_util_encode_create()
* @see image_util_encode_destroy()
*/
int image_util_encode_run_to_file(image_util_encode_h handle, image_util_image_h image, const char *file_path);

/**
* @brief Encodes the image to save it to a buffer in the given encoding handle.
* @details The function executes synchronously.
* @since_tizen 5.5
*
* @remarks If any functions at the pre-condition are not called first, #IMAGE_UTIL_ERROR_INVALID_PARAMETER is returned.\n
*          The @a buffer should be released using free().\n
*          The only supported colorspace for BMP and GIF is #IMAGE_UTIL_COLORSPACE_RGBA8888.\n
*          To get supported colorspaces for JPEG format, you can use image_util_foreach_supported_colorspace().\n
*          Before 6.0, #IMAGE_UTIL_COLORSPACE_RGBA8888 was the only supported colorspace for PNG. But since 6.0, this module supports more colorspaces for PNG.
*          To get supported colorspaces for PNG, you can use image_util_foreach_supported_colorspace().\n
*          Since 6.0, this module supports WEBP, you can use image_util_foreach_supported_colorspace() to get supported colorspaces for WEBP.\n
*          Since 7.0, this module supports JPEG-XL, you can use image_util_foreach_supported_colorspace() to get supported colorspaces for JPEG-XL.
*
* @param[in] handle The handle of image util encoding
* @param[in] image The image handle for encoding
* @param[out] buffer The buffer that encoded image is stored
* @param[out] buffer_size The size of the buffer
*
* @return @c 0 on success,
*                 otherwise a negative error value
*
* @retval #IMAGE_UTIL_ERROR_NONE Successful
* @retval #IMAGE_UTIL_ERROR_INVALID_PARAMETER Invalid parameter
* @retval #IMAGE_UTIL_ERROR_INVALID_OPERATION Invalid operation
* @retval #IMAGE_UTIL_ERROR_OUT_OF_MEMORY Out of memory
* @retval #IMAGE_UTIL_ERROR_NOT_SUPPORTED_FORMAT Not supported format
*
* @pre image_util_encode_create()
*
* @post image_util_encode_destroy()
*
* @see image_util_encode_create()
* @see image_util_encode_destroy()
*/
int image_util_encode_run_to_buffer(image_util_encode_h handle, image_util_image_h image, unsigned char **buffer, size_t *buffer_size);

/**
* @brief Encodes the image and save it to a file in the given encoding handle asynchronously.
* @details The output will be stored in @a file_path. And the function executes asynchronously.
* @since_tizen 5.5
*
* @remarks If any functions at the pre-condition are not called first, #IMAGE_UTIL_ERROR_INVALID_PARAMETER is returned.\n
*          The only supported colorspace for BMP and GIF is #IMAGE_UTIL_COLORSPACE_RGBA8888.\n
*          To get supported colorspaces for JPEG format, you can use image_util_foreach_supported_colorspace().\n
*          %http://tizen.org/privilege/mediastorage is required if @a file_path value is media storage.\n
*          %http://tizen.org/privilege/externalstorage is required if @a file_path value is external storage.\n
*          Before 6.0, #IMAGE_UTIL_COLORSPACE_RGBA8888 was the only supported colorspace for PNG. But since 6.0, this module supports more colorspaces for PNG.
*          To get supported colorspaces for PNG, you can use image_util_foreach_supported_colorspace().\n
*          Since 6.0, this module supports WEBP, you can use image_util_foreach_supported_colorspace() to get supported colorspaces for WEBP.\n
*          Since 7.0, this module supports JPEG-XL, you can use image_util_foreach_supported_colorspace() to get supported colorspaces for JPEG-XL.
*
* @param[in] handle The handle of image util encoding
* @param[in] image The image handle for encoding
* @param[in] file_path The file path for encoding image
* @param[in] completed_cb The callback function to be invoked
* @param[in] user_data The user data to be passed to the callback function
*
* @return @c 0 on success,
*                 otherwise a negative error value
*
* @retval #IMAGE_UTIL_ERROR_NONE Successful
* @retval #IMAGE_UTIL_ERROR_PERMISSION_DENIED Permission denied
* @retval #IMAGE_UTIL_ERROR_INVALID_PARAMETER Invalid parameter
* @retval #IMAGE_UTIL_ERROR_INVALID_OPERATION Invalid operation
* @retval #IMAGE_UTIL_ERROR_OUT_OF_MEMORY Out of memory
*
* @pre image_util_encode_create()
*
* @post image_util_encode_destroy()
*
* @see image_util_encode_create()
* @see image_util_encode_destroy()
*/
int image_util_encode_run_async_to_file(image_util_encode_h handle, image_util_image_h image, const char *file_path, image_util_encode_to_file_completed_cb completed_cb, void *user_data);

/**
* @brief Encodes the image and save it to a buffer in the given encoding handle asynchronously.
* @details The output will be stored in a buffer provided by the @a completed_cb callback.
* @since_tizen 5.5
*
* @remarks If any functions at the pre-condition are not called first, #IMAGE_UTIL_ERROR_INVALID_PARAMETER is returned.\n
*          The only supported colorspace for BMP and GIF is #IMAGE_UTIL_COLORSPACE_RGBA8888.\n
*          To get supported colorspaces for JPEG format, you can use image_util_foreach_supported_colorspace().\n
*          Before 6.0, #IMAGE_UTIL_COLORSPACE_RGBA8888 was the only supported colorspace for PNG. But since 6.0, this module supports more colorspaces for PNG.
*          To get supported colorspaces for PNG, you can use image_util_foreach_supported_colorspace().\n
*          Since 6.0, this module supports WEBP, you can use image_util_foreach_supported_colorspace() to get supported colorspaces for WEBP.\n
*          Since 7.0, this module supports JPEG-XL, you can use image_util_foreach_supported_colorspace() to get supported colorspaces for JPEG-XL.
*
* @param[in] handle The handle of image util encoding
* @param[in] image The image handle for encoding
* @param[in] completed_cb The callback function to be invoked
* @param[in] user_data The user data to be passed to the callback function
*
* @return @c 0 on success,
*                 otherwise a negative error value
*
* @retval #IMAGE_UTIL_ERROR_NONE Successful
* @retval #IMAGE_UTIL_ERROR_INVALID_PARAMETER Invalid parameter
* @retval #IMAGE_UTIL_ERROR_INVALID_OPERATION Invalid operation
* @retval #IMAGE_UTIL_ERROR_OUT_OF_MEMORY Out of memory
*
* @pre image_util_encode_create()
*
* @post image_util_encode_destroy()
*
* @see image_util_encode_create()
* @see image_util_encode_destroy()
*/
int image_util_encode_run_async_to_buffer(image_util_encode_h handle, image_util_image_h image, image_util_encode_to_buffer_completed_cb completed_cb, void *user_data);

/**
* @brief Destroys the image encoding handle.
* @since_tizen 3.0
*
* @remarks Any image handle created should be destroyed.
*
* @param[in] handle The handle of image util encoding
*
* @return @c 0 on success,
*                 otherwise a negative error value
*
* @retval #IMAGE_UTIL_ERROR_NONE Successful
* @retval #IMAGE_UTIL_ERROR_INVALID_PARAMETER Invalid parameter
* @retval #IMAGE_UTIL_ERROR_INVALID_OPERATION Invalid operation
*
* @pre image_util_encode_create()
*
* @see image_util_encode_create()
*/
int image_util_encode_destroy(image_util_encode_h handle);

/**
* @brief Creates a handle for encoding an animated GIF.
* @since_tizen 5.5
*
* @remarks The @a handle should be released using image_util_agif_encode_destroy().
*
* @param[out] handle The handle of encoding an animated GIF
*
* @return @c 0 on success,
*                 otherwise a negative error value
*
* @retval #IMAGE_UTIL_ERROR_NONE Successful
* @retval #IMAGE_UTIL_ERROR_INVALID_PARAMETER Invalid parameter
* @retval #IMAGE_UTIL_ERROR_OUT_OF_MEMORY Out of memory
*
* @post image_util_agif_encode_destroy()
*
* @see image_util_agif_encode_add_frame()
* @see image_util_agif_encode_save_to_file()
* @see image_util_agif_encode_save_to_buffer()
* @see image_util_agif_encode_destroy()
*
*/
int image_util_agif_encode_create(image_util_agif_encode_h *handle);

/**
* @brief Encodes an image and adds the encoded image to the frames of the animated GIF.
* @since_tizen 5.5
*
* @remarks This function should be called for each @a image which you want to add to the animated GIF. Each @a image should be the same size.\n
*          The supported colorspace is #IMAGE_UTIL_COLORSPACE_RGBA8888.\n
*          You should call image_util_agif_encode_save_to_file() or image_util_agif_encode_save_to_buffer() to save the animated GIF.\n
*          If you call this function after image_util_agif_encode_save_to_file() or image_util_agif_encode_save_to_buffer() function is called,
*          this function will encode a new animated GIF.
*
* @param[in] handle The handle of encoding an animated GIF
* @param[in] image The handle of the image for each frame
* @param[in] time_delay The time delay between @a image and the next image (in 0.01sec increments)
*
* @return @c 0 on success,
*                 otherwise a negative error value
*
* @retval #IMAGE_UTIL_ERROR_NONE Successful
* @retval #IMAGE_UTIL_ERROR_INVALID_PARAMETER Invalid parameter
* @retval #IMAGE_UTIL_ERROR_INVALID_OPERATION Invalid operation
* @retval #IMAGE_UTIL_ERROR_OUT_OF_MEMORY Out of memory
* @retval #IMAGE_UTIL_ERROR_NOT_SUPPORTED_FORMAT Not supported format
*
* @pre image_util_agif_encode_create()
*
* @post image_util_agif_encode_save_to_file()
* @post image_util_agif_encode_save_to_buffer()
* @post image_util_agif_encode_destroy()
*
* @see image_util_agif_encode_create()
* @see image_util_agif_encode_save_to_file()
* @see image_util_agif_encode_save_to_buffer()
* @see image_util_agif_encode_destroy()
*
*/
int image_util_agif_encode_add_frame(image_util_agif_encode_h handle, image_util_image_h image, unsigned int time_delay);

/**
* @brief Saves the animated GIF image to the file.
* @details After the data has been written to a file,
*          the file cannot be modified. In other words,\n
*          it is not possible to add frames to the file.\n
*          Saving animation flushes the frames that has been added to the encoder.\n
*          However, the encoder can be still used after the data is written.\n
*          For example, although you write the data to one file(B0),\n
*          you can do the following operations:\n
*
*          1. Add frames 1-10 to the encoder.\n
*          2. Save the data in the encoder to another file(B1). 10 frames will be saved.\n
*          3. Add more frames (11-20) to the same encoder.\n
*          4. Save the data in the encoder to the other file (B2)\n
*             It is not possible to save the changed data to B1 file used in step 4.
* @since_tizen 5.5
*
* @remarks %http://tizen.org/privilege/mediastorage is required if @a file_path value is media storage.\n
*          %http://tizen.org/privilege/externalstorage is required if @a file_path value is external storage.
*
* @param[in] handle The handle of encoding an animated GIF
* @param[in] file_path The file path for saving the animated GIF
*
* @return @c 0 on success,
*                 otherwise a negative error value
*
* @retval #IMAGE_UTIL_ERROR_NONE Successful
* @retval #IMAGE_UTIL_ERROR_PERMISSION_DENIED Permission denied
* @retval #IMAGE_UTIL_ERROR_INVALID_PARAMETER Invalid parameter
* @retval #IMAGE_UTIL_ERROR_OUT_OF_MEMORY Out of memory
* @retval #IMAGE_UTIL_ERROR_NO_SUCH_FILE No such file
*
* @pre image_util_agif_encode_create()
*
* @post image_util_agif_encode_destroy()
*
* @see image_util_agif_encode_create()
* @see image_util_agif_encode_add_frame()
* @see image_util_agif_encode_destroy()
*
*/
int image_util_agif_encode_save_to_file(image_util_agif_encode_h handle, const char *file_path);

/**
* @brief Saves the animated GIF image to the buffer.
* @details After the data has been written to a buffer,
*          the file cannot be modified. In other words,\n
*          it is not possible to add frames to the buffer.\n
*          Saving animation flushes the frames that has been added to the encoder.\n
*          However, the encoder can be still used after the data is written.\n
*          For example, although you write the data to one buffer(B0),\n
*          you can do the following operations:\n
*
*          1. Add frames 1-10 to the encoder.\n
*          2. Save the data in the encoder to another buffer(B1). 10 frames will be saved.\n
*          3. Add more frames (11-20) to the same encoder.\n
*          4. Save the data in the encoder to the other buffer (B2)\n
*             It is not possible to save the changed data to B1 file used in step 4.
* @since_tizen 5.5
*
* @remarks The @a buffer should be released using free() after using it.
*
* @param[in] handle The handle of encoding an animated GIF
* @param[out] buffer The buffer in which the animated GIF is saved
* @param[out] buffer_size The size of the buffer
*
* @return @c 0 on success,
*                 otherwise a negative error value
*
* @retval #IMAGE_UTIL_ERROR_NONE Successful
* @retval #IMAGE_UTIL_ERROR_INVALID_PARAMETER Invalid parameter
* @retval #IMAGE_UTIL_ERROR_OUT_OF_MEMORY Out of memory
*
* @pre image_util_agif_encode_create()
*
* @post image_util_agif_encode_destroy()
*
* @see image_util_agif_encode_create()
* @see image_util_agif_encode_add_frame()
* @see image_util_agif_encode_destroy()
*
*/
int image_util_agif_encode_save_to_buffer(image_util_agif_encode_h handle, unsigned char **buffer, size_t *buffer_size);

/**
* @brief Destroys the handle for encoding an animated GIF.
* @since_tizen 5.5
*
* @remarks Any created animated GIF encoding handle created should be destroyed.
*
* @param[in] handle The handle of encoding an animated GIF
*
* @return @c 0 on success,
*                 otherwise a negative error value
*
* @retval #IMAGE_UTIL_ERROR_NONE Successful
* @retval #IMAGE_UTIL_ERROR_INVALID_PARAMETER Invalid parameter
*
* @pre image_util_agif_encode_create()
*
* @see image_util_agif_encode_create()
*
*/
int image_util_agif_encode_destroy(image_util_agif_encode_h handle);

/**
* @brief Creates a handle for encoding animation.
* @since_tizen 6.0
*
* @remarks The @a handle should be released using image_util_anim_encode_destroy().
*
* @param[in] image_type The type of output image to create animation
* @param[out] handle The handle of animation encoder
*
* @return @c 0 on success,
*                 otherwise a negative error value
*
* @retval #IMAGE_UTIL_ERROR_NONE Successful
* @retval #IMAGE_UTIL_ERROR_INVALID_PARAMETER Invalid parameter
* @retval #IMAGE_UTIL_ERROR_NOT_SUPPORTED_FORMAT Not supported format
*
* @see image_util_anim_encode_add_frame()
* @see image_util_anim_encode_save_to_file()
* @see image_util_anim_encode_save_to_buffer()
* @see image_util_anim_encode_destroy()
*
*/
int image_util_anim_encode_create(image_util_anim_type_e image_type, image_util_anim_encode_h *handle);

/**
* @brief Sets a number of times to repeat the animation.
* @since_tizen 6.0
*
* @remarks This function should be called before image_util_anim_encode_add_frame() is called.
*
* @param[in] handle The handle of animation encoder
* @param[in] loop_count The number of times to repeat the animation
*
* @return @c 0 on success,
*                 otherwise a negative error value
*
* @retval #IMAGE_UTIL_ERROR_NONE Successful
* @retval #IMAGE_UTIL_ERROR_INVALID_PARAMETER Invalid parameter
* @retval #IMAGE_UTIL_ERROR_NOT_SUPPORTED_FORMAT Not supported format
*
* @pre image_util_anim_encode_create()
*
*/
int image_util_anim_encode_set_loop_count(image_util_anim_encode_h handle, unsigned int loop_count);

/**
* @brief Sets a background color of the animation.
* @since_tizen 6.0
*
* @remarks This function should be called before image_util_anim_encode_add_frame() is called.
*
* @param[in] handle The handle of animation encoder
* @param[in] r The red color of the background
* @param[in] g The green color of the background
* @param[in] b The blue color of the background
* @param[in] a The alpha color of the background
*
* @return @c 0 on success,
*                 otherwise a negative error value
*
* @retval #IMAGE_UTIL_ERROR_NONE Successful
* @retval #IMAGE_UTIL_ERROR_INVALID_PARAMETER Invalid parameter
* @retval #IMAGE_UTIL_ERROR_NOT_SUPPORTED_FORMAT Not supported format
*
* @pre image_util_anim_encode_create()
*
*/
int image_util_anim_encode_set_background_color(image_util_anim_encode_h handle, unsigned char r, unsigned char g, unsigned char b, unsigned char a);

/**
* @brief Sets lossless compression of animation.
* @since_tizen 6.0
*
* @remarks This function should be called before image_util_anim_encode_add_frame() is called.
*
* @param[in] handle The handle of animation encoder
* @param[in] lossless The flag determining whether the compression is lossless or lossy: true for lossless, false for lossy
*
* @return @c 0 on success,
*                 otherwise a negative error value
*
* @retval #IMAGE_UTIL_ERROR_NONE Successful
* @retval #IMAGE_UTIL_ERROR_INVALID_PARAMETER Invalid parameter
* @retval #IMAGE_UTIL_ERROR_NOT_SUPPORTED_FORMAT Not supported format
*
* @pre image_util_anim_encode_create()
*
*/
int image_util_anim_encode_set_lossless(image_util_anim_encode_h handle, bool lossless);

/**
* @brief Encodes an image and adds image to the frame of the animated image.
* @since_tizen 6.0
*
* @remarks This function should be called for each @a image which you want to add to the animated WEBP or GIF. Each @a image should be the same size.\n
*          The supported colorspaces are #IMAGE_UTIL_COLORSPACE_RGBA8888, #IMAGE_UTIL_COLORSPACE_ARGB8888, #IMAGE_UTIL_COLORSPACE_BGRA8888 and #IMAGE_UTIL_COLORSPACE_RGBA8888 for animated WEBP.\n
*          The supported colorspace is #IMAGE_UTIL_COLORSPACE_RGBA8888 for animated GIF.\n
*          You should call image_util_anim_encode_save_to_file() or image_util_anim_encode_save_to_buffer() to save the animated WEBP.\n
*          If you call this function after image_util_anim_encode_save_to_file() or image_util_anim_encode_save_to_buffer() function is called,
*          this function will encode a new animated WEBP and GIF.
*
* @param[in] handle The handle of animation encoder
* @param[in] image The handle of the image for each frame
* @param[in] time_delay The time delay between @a image and the next image (in 1ms increments for WEBP and 10ms increments for GIF)
*
* @return @c 0 on success,
*                 otherwise a negative error value
*
* @retval #IMAGE_UTIL_ERROR_NONE Successful
* @retval #IMAGE_UTIL_ERROR_INVALID_PARAMETER Invalid parameter
* @retval #IMAGE_UTIL_ERROR_INVALID_OPERATION Invalid operation
* @retval #IMAGE_UTIL_ERROR_OUT_OF_MEMORY Out of memory
*
* @pre image_util_anim_encode_create()
*
* @post image_util_anim_encode_save_to_file()
* @post image_util_anim_encode_save_to_buffer()
*
* @see image_util_anim_encode_save_to_file()
* @see image_util_anim_encode_save_to_buffer()
*
*/
int image_util_anim_encode_add_frame(image_util_anim_encode_h handle, image_util_image_h image, unsigned int time_delay);

/**
* @brief Saves the animated WEBP or GIF image to the file.
* @details After the data has been written to a file,
*          the file cannot be modified. In other words,\n
*          it is not possible to add frames to the file.\n
*          Saving animation flushes the frames that has been added to the encoder.\n
*          However, the encoder can be still used after the data is written.\n
*          For example, although you write the data to one file(B0),\n
*          you can do the following operations:\n
*
*          1. Add frames 1-10 to the encoder.\n
*          2. Save the data in the encoder to another file(B1). 10 frames will be saved.\n
*          3. Add more frames (11-20) to the same encoder.\n
*          4. Save the data in the encoder to the other file (B2)\n
*             It is not possible to save the changed data to B1 file used in step 4.
* @since_tizen 6.0
*
* @remarks %http://tizen.org/privilege/mediastorage is required if @a file_path value is media storage.\n
*          %http://tizen.org/privilege/externalstorage is required if @a file_path value is external storage.
*
* @param[in] handle The handle of encoding an animated WEBP or GIF
* @param[in] file_path The file path for saving the animated WEBP or GIF
*
* @return @c 0 on success,
*                 otherwise a negative error value
*
* @retval #IMAGE_UTIL_ERROR_NONE Successful
* @retval #IMAGE_UTIL_ERROR_PERMISSION_DENIED Permission denied
* @retval #IMAGE_UTIL_ERROR_INVALID_PARAMETER Invalid parameter
* @retval #IMAGE_UTIL_ERROR_INVALID_OPERATION Invalid operation
* @retval #IMAGE_UTIL_ERROR_NO_SUCH_FILE No such file
*
* @pre image_util_anim_encode_create()
*
* @see image_util_anim_encode_add_frame()
*
*/
int image_util_anim_encode_save_to_file(image_util_anim_encode_h handle, const char *file_path);

/**
* @brief Saves the animation WEBP or GIF image to the buffer.
* @details After the data has been written to a buffer,
*          the file cannot be modified. In other words,\n
*          it is not possible to add frames to the buffer.\n
*          Saving animation flushes the frames that has been added to the encoder.\n
*          However, the encoder can be still used after the data is written.\n
*          For example, although you write the data to one buffer(B0),\n
*          you can do the following operations:\n
*
*          1. Add frames 1-10 to the encoder.\n
*          2. Save the data in the encoder to another buffer(B1). 10 frames will be saved.\n
*          3. Add more frames (11-20) to the same encoder.\n
*          4. Save the data in the encoder to the other buffer (B2)\n
*             It is not possible to save the changed data to B1 file used in step 4.
* @since_tizen 6.0
*
* @remarks The @a buffer should be released using free() after using it.
*
* @param[in] handle The handle of encoding an animated WEBP or GIF
* @param[out] buffer The buffer in which the animated WEBP or GIF is saved
* @param[out] buffer_size The size of the buffer
*
* @return @c 0 on success,
*                 otherwise a negative error value
*
* @retval #IMAGE_UTIL_ERROR_NONE Successful
* @retval #IMAGE_UTIL_ERROR_INVALID_PARAMETER Invalid parameter
* @retval #IMAGE_UTIL_ERROR_INVALID_OPERATION Invalid operation
*
* @pre image_util_anim_encode_create()
*
* @see image_util_anim_encode_add_frame()
*
*/
int image_util_anim_encode_save_to_buffer(image_util_anim_encode_h handle, unsigned char **buffer, size_t *buffer_size);

/**
* @brief Destroys the handle for encoding animation.
* @since_tizen 6.0
*
* @remarks Any created animated encoder handle created should be destroyed.
*
* @param[in] handle The handle of animation encoder
*
* @return @c 0 on success,
*                 otherwise a negative error value
*
* @retval #IMAGE_UTIL_ERROR_NONE Successful
* @retval #IMAGE_UTIL_ERROR_INVALID_PARAMETER Invalid parameter
*
* @pre image_util_anim_encode_create()
*
* @see image_util_anim_encode_create()
*
*/
int image_util_anim_encode_destroy(image_util_anim_encode_h handle);

/**
 * @}
 */

#ifdef __cplusplus
}
#endif

#endif /* __TIZEN_MULTIMEDIA_IMAGE_UTIL_ENCODE_H__ */
