/*
* Copyright (c) 2011 Samsung Electronics Co., Ltd All Rights Reserved
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
* http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

#ifndef __TIZEN_MULTIMEDIA_THUMBNAIL_UTIL_H__
#define __TIZEN_MULTIMEDIA_THUMBNAIL_UTIL_H__

#include <thumbnail_util_type.h>

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/**
 * @file thumbnail_util.h
 * @brief This file contains the thumbnail util API and functions related with handling thumbnail. \n
 *        Functions include operations to get the various size thumbnail from media file.
 */

/**
 * @addtogroup CAPI_MEDIA_THUMBNAIL_UTIL_MODULE
 * @{
 */

/**
 * @deprecated Deprecated since 5.0.
 * @brief Creates thumbnail info handle.
 * @since_tizen @if MOBILE 2.4 @elseif WEARABLE 3.0 @endif
 * @remarks The @a thumb info should be released using thumbnail_util_destroy().
 *
 * @param[in] thumb The handle to thumbnail info
 *
 * @return @c 0 on success, otherwise a negative error value
 *
 * @retval #THUMBNAIL_UTIL_ERROR_NONE Successful
 * @retval #THUMBNAIL_UTIL_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #THUMBNAIL_UTIL_ERROR_OUT_OF_MEMORY Out of memory
 *
 * @see thumbnail_util_destroy()
 */
int thumbnail_util_create(thumbnail_h *thumb) TIZEN_DEPRECATED_API;

/**
 * @deprecated Deprecated since 5.0. Use thumbnail_util_extract_to_file() or thumbnail_util_extract_to_buffer() instead.
 * @brief Extracts the thumbnail for the given media, asynchronously.
 * @details This function extracts the thumbnail for given media item and calls registered
 *         callback function for completion of extracting the thumbnail.\n
 *         You can distinguish generated thumbnails by @a request_id. Therefore, the @a request_id must be maintained until the request is completed. \n
 *         And the @a request_id is also used to cancel the request.
 *
 * @since_tizen @if MOBILE 2.4 @elseif WEARABLE 3.0 @endif
 *
 * @remarks The @a request_id should be released using free(). \n
 *                %http://tizen.org/privilege/mediastorage is needed if input or output path are relevant to media storage. \n
 *                %http://tizen.org/privilege/externalstorage is needed if input or output path are relevant to external storage. \n
 *                If you don't set the size, the thumbnail size will be set default size. Default size is 320x240. \n
 *                Color space of the generated thumbnail is BGRA. \n
 *                Since 3.0, %http://tizen.org/privilege/content.write privilege is not required. \n
 *
 * @param[in] thumb     The thumbnail info handle
 * @param[in] callback  The callback function to be invoked
 * @param[in] user_data The user data to be passed to the callback function
 * @param[out] request_id The request id for the thumbnail extraction request
 *
 * @return @c 0 on success, otherwise a negative error value
 *
 * @retval #THUMBNAIL_UTIL_ERROR_NONE              Successful
 * @retval #THUMBNAIL_UTIL_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #THUMBNAIL_UTIL_ERROR_PERMISSION_DENIED Permission denied
 *
 * @pre Create a thumbnail info handle by calling thumbnail_util_create(). \n
 *           The content path that you want to generate thumbnail must be set by calling thumbnail_util_set_path().
 * @post Thumbnail extraction result except canceled is provided by thumbnail_extracted_cb().
 * @see thumbnail_util_create()
 * @see thumbnail_util_set_path()
 * @see thumbnail_util_set_size()
 * @see thumbnail_extracted_cb()
 * @see thumbnail_util_cancel()
 * @see thumbnail_util_destroy()
 */
int thumbnail_util_extract(thumbnail_h thumb, thumbnail_extracted_cb callback, void *user_data, char **request_id) TIZEN_DEPRECATED_API;

/**
 * @deprecated Deprecated since 5.0.
 * @brief Sets the path of original media file.
 * @since_tizen @if MOBILE 2.4 @elseif WEARABLE 3.0 @endif
 *
 * @param[in] thumb     The thumbnail info handle
 * @param[in] path        The path of the original media file
 *
 * @return @c 0 on success, otherwise a negative error value
 *
 * @retval #THUMBNAIL_UTIL_ERROR_NONE              Successful
 * @retval #THUMBNAIL_UTIL_ERROR_INVALID_PARAMETER Invalid parameter
 *
 * @pre Create a thumbnail info handle by calling thumbnail_util_create().
 * @see thumbnail_util_create()
 * @see thumbnail_util_set_size()
 * @see thumbnail_util_destroy()
 */
int thumbnail_util_set_path(thumbnail_h thumb, const char *path) TIZEN_DEPRECATED_API;

/**
 * @deprecated Deprecated since 5.0.
 * @brief Sets the size of thumbnail to be extracted.
 * @since_tizen @if MOBILE 2.4 @elseif WEARABLE 3.0 @endif
 * @remarks If you don't set the size, the thumbnail size will be set default size. Default size is 320x240. \n
 *                  If the set width is not a multiple of 8, it can be changed by inner process. the width will be a multiple of 8 greater than the set value.
 *
 * @param[in] thumb    The thumbnail info handle
 * @param[in] width      The width of the thumbnail
 * @param[in] height     The height of the thumbnail
 *
 * @return @c 0 on success, otherwise a negative error value
 *
 * @retval #THUMBNAIL_UTIL_ERROR_NONE              Successful
 * @retval #THUMBNAIL_UTIL_ERROR_INVALID_PARAMETER Invalid parameter
 *
 * @pre Create a thumbnail info handle by calling thumbnail_util_create().
 * @see thumbnail_util_create()
 * @see thumbnail_util_set_path()
 * @see thumbnail_util_destroy()
 *
 */
int thumbnail_util_set_size(thumbnail_h thumb, int width, int height) TIZEN_DEPRECATED_API;

/**
 * @deprecated Deprecated since 5.0.
 * @brief Cancels the thumbnail extraction request for the given media.
 * @since_tizen @if MOBILE 2.4 @elseif WEARABLE 3.0 @endif
 * @remarks If there is no request to be canceled, this function returns #THUMBNAIL_UTIL_ERROR_INVALID_OPERATION
 *
 * @param[in] thumb         The media thumbnail info handle
 * @param[in] request_id   The generated request id by thumbnail extraction request
 *
 * @return @c 0 on success, otherwise a negative error value
 *
 * @retval #THUMBNAIL_UTIL_ERROR_NONE              Successful
 * @retval #THUMBNAIL_UTIL_ERROR_INVALID_PARAMETER Invalid parameter
 *
 * @pre It is available before thumbnail_extracted_cb() is called.
 * @see thumbnail_util_extract()
 */
int thumbnail_util_cancel(thumbnail_h thumb, const char *request_id) TIZEN_DEPRECATED_API;

/**
 * @deprecated Deprecated since 5.0.
 * @brief Destroys thumbnail info handle.
 * @details The function frees all resources related to the thumbnail info handle. This handle
 *          can no longer be used to perform any operations. New thumbnail info handle has to
 *          be created before the next usage.
 *
 * @since_tizen @if MOBILE 2.4 @elseif WEARABLE 3.0 @endif
 *
 * @param[in]  thumb  The thumbnail info handle
 *
 * @return @c 0 on success, otherwise a negative error value
 *
 * @retval #THUMBNAIL_UTIL_ERROR_NONE              Successful
 * @retval #THUMBNAIL_UTIL_ERROR_INVALID_PARAMETER Invalid parameter
 *
 * @pre Create a thumbnail info handle by calling thumbnail_util_create().
 * @see thumbnail_util_create()
 */
int thumbnail_util_destroy(thumbnail_h thumb) TIZEN_DEPRECATED_API;

/**
 * @brief Extracts the thumbnail for the given media into a file, synchronously.
 * @details This function creates thumbnail from a given media file. \n
 *          The generated thumbnail is saved in the given thumbnail path. \n
 *          The format of the thumbnail file is determined by the extension of the thumbnail path
 *          (e.g. path is input.jpg and thumbnail_path is thumbnail.png then thumbnail.png is encoded in the PNG format). \n
 *          If the extension is missing or the format is not supported, the thumbnail is encoded in the same format as the given media
 *          (e.g. path is input.jpg and thumbnail_path is thumbnail.out then thumbnail.out is encoded in the JPEG format). \n
 *          Supported image formats are as follows: \n
 *           - GIF, JPEG, PNG, BMP, WBMP \n
 *          Video files only support JPEG format thumbnail.
 *
 * @since_tizen 5.0
 *
 * @remarks %http://tizen.org/privilege/mediastorage is needed if input or output path points to media storage. \n
 *          %http://tizen.org/privilege/externalstorage is needed if input or output path points to external storage. \n
 *          The width and height of the thumbnail to be generated cannot exceed 2000. \n
 *          The width and height of the thumbnail to be generated cannot exceed the original resolution. \n
 *          Since 5.5, if the width and height of the thumbnail to be generated exceeds the original resolution, the value changes to the original resolution. \n
 *          In order to maintain the ratio, the requested size and generated size may be different. (Modify based on short axis) \n
 *          If the parent directory of the thumbnail path to be generated does not exist, an error is returned.
 *
 * @param[in] path       The path of the original media file
 * @param[in] width      The width of the thumbnail
 * @param[in] height     The height of the thumbnail
 * @param[in] thumbnail_path   The path to save the generated thumbnail
 *
 * @return @c 0 on success, otherwise a negative error value
 *
 * @retval #THUMBNAIL_UTIL_ERROR_NONE              Successful
 * @retval #THUMBNAIL_UTIL_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #THUMBNAIL_UTIL_ERROR_INVALID_OPERATION Invalid operation
 * @retval #THUMBNAIL_UTIL_ERROR_OUT_OF_MEMORY Out of memory
 * @retval #THUMBNAIL_UTIL_ERROR_PERMISSION_DENIED Permission denied
 *
 */
int thumbnail_util_extract_to_file(const char *path, unsigned int width, unsigned int height, const char *thumbnail_path);

/**
 * @brief Extracts the thumbnail for the given media into a buffer, synchronously.
 * @details This function creates thumbnails from a given media file. \n
 *          The generated thumbnail is returned as a raw data stream.
 *
 * @since_tizen 5.0
 *
 * @remarks %http://tizen.org/privilege/mediastorage is needed if input or output path points to media storage. \n
 *          %http://tizen.org/privilege/externalstorage is needed if input or output path points to external storage. \n
 *          The width and height of the thumbnail to be generated cannot exceed 2000. \n
 *          The width and height of the thumbnail to be generated cannot exceed the original resolution. \n
 *          In the case of video file, color space of the generated thumbnail is RGB. \n
 *          In the case of image file, color space of the generated thumbnail is BGRA. \n
 *          The @a thumb_buffer should be released using free().
 *          Since 5.5, if the width and height of the thumbnail to be generated exceeds the original resolution, the value changes to the original resolution. \n
 *          In order to maintain the ratio, the requested size and generated size may be different. (Modify based on short axis)
 *
 * @param[in] path       The path of the original media file
 * @param[in] width      The width of the thumbnail
 * @param[in] height     The height of the thumbnail
 * @param[out] thumb_buffer   The raw data of the thumbnail which is generated
 * @param[out] thumb_size     The size of the thumbnail which is generated
 * @param[out] thumb_width    The width of the thumbnail which is generated
 * @param[out] thumb_height   The height of the thumbnail which is generated
 *
 * @return @c 0 on success, otherwise a negative error value
 *
 * @retval #THUMBNAIL_UTIL_ERROR_NONE              Successful
 * @retval #THUMBNAIL_UTIL_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #THUMBNAIL_UTIL_ERROR_INVALID_OPERATION Invalid operation
 * @retval #THUMBNAIL_UTIL_ERROR_OUT_OF_MEMORY Out of memory
 * @retval #THUMBNAIL_UTIL_ERROR_PERMISSION_DENIED Permission denied
 *
 */
int thumbnail_util_extract_to_buffer(const char *path, unsigned int width, unsigned int height, unsigned char **thumb_buffer, size_t *thumb_size, unsigned int *thumb_width, unsigned int *thumb_height);

/**
 * @}
 */

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* __TIZEN_MULTIMEDIA_THUMBNAIL_UTIL_H__ */
