/*
 * Copyright (c) 2013 - 2016 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the License);
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef _MINICTRL_PROVIDER_H_
#define _MINICTRL_PROVIDER_H_

#include <Evas.h>
#include <bundle.h>
#include <minicontrol-error.h>
#include <minicontrol-type.h>

#ifdef __cplusplus
extern "C" {
#endif

/**
 * @file minicontrol-provider.h
 * @brief This minicontrol provider library is used to create evas socket window. \n
 *        This library provides functions to create a remote evas object to draw a minicontrol on a minicontrol viewer and \n
 *        sending some requests to the minicontrol viewer. \n
 *        Drawings on this remote evas object will be shown on the place of the minicontrol viewer.
 */

/**
 * @addtogroup MINICONTROL_PROVIDER_MODULE
 * @{
 */

/**
 * @brief Called when a event comes from viewer.
 * @since_tizen @if MOBILE 2.4 @elseif WEARABLE 5.0 @endif
 * @remarks The @a event_arg should not be released.
 * @param[in] event_type The type of fired event
 * @param[in] event_arg Argument of the event
 *            The @a event_arg can be used only in the callback. To use outside, make a copy.
 * @pre minicontrol_viewer_register_event_callback() used to register this callback.
 * @see #minicontrol_create_window
 * @see #minicontrol_viewer_event_e
 */
typedef void (*minicontrol_event_cb)(minicontrol_viewer_event_e event_type, bundle *event_arg);

/**
 * @brief Creates a window for minicontrol.
 * @since_tizen @if MOBILE 2.4 @elseif WEARABLE 5.0 @endif
 * @remarks The specific error code can be obtained using the gat_last_result() method. Error codes are described in Exception section.
 *          The returned value should be released using evas_object_del().
 * @param[in] name Name of minicontrol socket window
 * @param[in] target_viewer Target viewer for minicontrol. You can select multiple viewers by using bitwise OR operator
 * @param[in] callback A callback function for events originated by minicontrol viewer
 * @return Evas object of minicontrol. NULL on error
 * @exception #MINICONTROL_ERROR_NONE Success
 * @exception #MINICONTROL_ERROR_NOT_SUPPORTED Not supported
 * @exception #MINICONTROL_ERROR_INVALID_PARAMETER Invalid argument
 * @exception #MINICONTROL_ERROR_ELM_FAILURE Some error occurred when creating a  minicontrol window
 * @exception #MINICONTROL_ERROR_OUT_OF_MEMORY Out of memory
 * @see #minicontrol_target_viewer_e
 * @see #minicontrol_event_cb
 * @see get_last_result()
 */
Evas_Object *minicontrol_create_window(const char *name, minicontrol_target_viewer_e target_viewer, minicontrol_event_cb callback);

/**
 * @brief Sends a event to the viewer.
 * @since_tizen @if MOBILE 2.4 @elseif WEARABLE 5.0 @endif
 * @remarks When a viewer doesn't handle some events, it can be ignored.
 * @param[in] minicontrol Minicontrol window
 * @param[in] event Type of the event
 * @param[in] event_arg Bundle argument of the event
 * @return #MINICONTROL_ERROR_NONE on success,
 *         otherwise an error code on failure
 * @retval #MINICONTROL_ERROR_NONE Success
 * @retval #MINICONTROL_ERROR_NOT_SUPPORTED Not supported
 * @retval #MINICONTROL_ERROR_INVALID_PARAMETER Invalid argument
 * @see #minicontrol_provider_event_e
 * @see #minicontrol_create_window
 */
int minicontrol_send_event(Evas_Object *minicontrol, minicontrol_provider_event_e event, bundle *event_arg);

/**
 * @}
 */

#ifdef __cplusplus
}
#endif
#endif /* _MINICTRL_PROVIDER_H_ */

