/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef SAMPLENATIVEIMEUI_H_
#define SAMPLENATIVEIMEUI_H_

#include <Elementary.h>
#include <stdio.h>
#include "samplenativeimeuiproperty.h"
#include "samplenativeime.h"

/**
 * @brief set screen rotation degree which is (0,90,180,270)
 *
 * @param degree the screen rotation degree
 *
 * @return void
 */
void sample_native_ime_set_screen_rotation_degree(int degree);

/**
 * @brief register callback functions for key events
 *
 * @param callback the callback function to register
 *
 * @return set the callback events success or not
 */
Eina_Bool sample_native_ime_set_key_events_callback(IMEKeyEventCallBack *callback);

/**
 * @brief create ime layout
 *
 * @param main_win the main window for keyboard
 * @param layout the layout to create
 * @param edje_path the path for edje file
 * @param scale_ratio the scale ratio for ui
 * @param shift_state the shift status for keyboard
 *
 * @return the layout create result, 0 success, others failed.
 */
int sample_native_ime_layout_create(Evas_Object *main_win, IMELayout *layout, char *edje_path, double scale_ratio, IMEShiftState shift_state);

/**
 * @brief show ime layout
 *
 * @param layout the pointer of layout
 *
 * @return show layout result, 0 success, others failed.
 */
int sample_native_ime_layout_show(IMELayout *layout);

/**
 * @brief hide ime layout
 *
 * @param layout the pointer of layout
 *
 * @return hide layout result, 0 success, others failed.
 */
int sample_native_ime_layout_hide(IMELayout *layout);

/**
 * @brief set return key disable status
 *
 * @param layout the pointer of layout
 * @param disabled the disable status of return key
 *
 * @return set the disable status, success or not
 */
Eina_Bool sample_native_ime_set_return_disable(IMELayout *layout, Eina_Bool disabled);

/**
 * @brief set return key disable type
 *
 * @param layout the pointer of layout
 * @param type the type of return key
 *
 * @return set the return key type, success or not
 */
Eina_Bool sample_native_ime_set_return_type(IMELayout *layout, Ecore_IMF_Input_Panel_Return_Key_Type type);

/**
 * @brief update shift key state
 *
 * @param layout the pointer of layout
 * @param shift_state the shift state
 *
 * @return update shift key state, success or not
 */
Eina_Bool sample_native_ime_update_shift_state(IMELayout *layout, IMEShiftState shift_state);

/**
 * @brief update the whole layout
 *
 * @param layout the pointer of layout
 * @param shift_state the shift state of layout
 *
 * @return update layout, success or not
 */
Eina_Bool sample_native_ime_update_layout(IMELayout *layout, IMEShiftState shift_state);

/**
 * @brief create the magnifier window
 *
 * @param main_win the main window for keyboard
 * @param edje_path the path for edje file
 * @param scale_ratio the scale ratio for ui
 *
 * @return the pointer for magnifier window
 */
Evas_Object *sample_native_ime_magnifier_win_create(Evas_Object *main_win, char *edje_path, double scale_ratio);

/**
 * @brief show the magnifier window
 *
 * @param x the x coordinate for magnifier window
 * @param y the y coordinate for magnifier window
 * @param width the width for magnifier window
 * @param height the height for magnifier window
 * @param label_text the label text
 * @param scale_ratio the scale_ratio for ui
 *
 * @return void
 */
void sample_native_ime_magnifier_win_show(int x, int y, int width, int height, const char *label_text, double scale_ratio);

/**
 * @brief hide the magnifier window
 *
 * @param void
 *
 * @return void
 */
void sample_native_ime_magnifier_win_hide();

#endif /* SAMPLENATIVEIMEUI_H_ */
