/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "samplenativeimefeedback.h"
#include <dlog.h>

#ifdef LOG_TAG
#undef LOG_TAG
#endif
#define LOG_TAG "$(appNameUpper)"

static Eina_Bool g_sound_feedback_status = EINA_FALSE;         /* the sound feedback status */
static Eina_Bool g_vibration_feedback_status = EINA_FALSE;     /* the vibration feedback status */

/**
 * @brief set the sound feedback status
 *
 * @param status sound feedback status
 */
void sample_native_ime_set_sound_feedback_status(Eina_Bool status)
{
	g_sound_feedback_status = status;
}

/**
 * @brief set the vibration feedback status
 *
 * @param status vibration feedback status
 */
void sample_native_ime_set_vibration_feedback_status(Eina_Bool status)
{
	g_vibration_feedback_status = status;
}

/**
 * @brief get the sound feedback status
 *
 * @return the sound feedback status
 */
Eina_Bool sample_native_ime_get_sound_feedback_status()
{
	return g_sound_feedback_status;
}

/**
 * @brief get the vibration feedback status
 *
 * @return the vibration feedback
 */
Eina_Bool sample_native_ime_get_vibration_feedback_status()
{
	return g_vibration_feedback_status;
}

/**
 * @brief initialize the feedback
 *        feedback must be initialized before usage
 *
 * @param
 *
 * @return
 */
void sample_native_ime_feedback_initialize(void)
{
	/* feedback should be initialized before using */
	int res = feedback_initialize();
	if (res != FEEDBACK_ERROR_NONE)
		dlog_print(DLOG_DEBUG, LOG_TAG, "Feedback initialize failed:%d", res);
	else
		dlog_print(DLOG_DEBUG, LOG_TAG, "Feedback initialize successful");
}

/**
 * @brief deinitialize the feedback
 *        feedback must be deinitialized after usage
 *
 * @param
 *
 * @return
 */
void sample_native_ime_feedback_deinitialize(void)
{
	/* stop feedback */
	feedback_stop();
	/* deinitialize feedback */
	int res = feedback_deinitialize();
	if (res != FEEDBACK_ERROR_NONE)
		dlog_print(DLOG_DEBUG, LOG_TAG, "Feedback initialize failed:%d", res);
	else
		dlog_print(DLOG_DEBUG, LOG_TAG, "Feedback initialize successful");
}

/**
 * @brief play sound feedback
 *
 * @param
 *
 * @return
 */
static void sample_native_ime_play_sound_feedback()
{
	if (g_sound_feedback_status) {
		/* play sound feedback */
		int res = feedback_play_type(FEEDBACK_TYPE_SOUND, FEEDBACK_PATTERN_SIP);
		if (res != FEEDBACK_ERROR_NONE)
			dlog_print(DLOG_DEBUG, LOG_TAG, "can not play sound");
	}
}

/**
 * @brief play vibration feedback
 *
 * @param
 *
 * @retrun
 */
static void sample_native_ime_play_vibration_feedback()
{
	if (g_vibration_feedback_status) {
		/* play vibration feedback */
		int res = feedback_play_type(FEEDBACK_TYPE_VIBRATION, FEEDBACK_PATTERN_SIP);
		if (res != FEEDBACK_ERROR_NONE)
			dlog_print(DLOG_DEBUG, LOG_TAG, "can not play vibration");
	}
}

/**
 * @brief play feedback
 *
 * @param
 *
 * @return
 */
void sample_native_ime_feedback()
{
	/* play sound and vibration feedback */
	sample_native_ime_play_sound_feedback();
	sample_native_ime_play_vibration_feedback();
}
