/*
 * Copyright (c) 2018 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Licensed under the Flora License, Version 1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://floralicense.org/license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef __EFL_EXTENSION_PANEL_H__
#define __EFL_EXTENSION_PANEL_H__

#ifdef __cplusplus
extern "C" {
#endif

/**
 * @defgroup CAPI_EFL_EXTENSION_PANEL_MODULE Efl Extension Panel
 * @ingroup CAPI_EFL_EXTENSION_COMMON_UI_MODULE
 * @brief Eext_panel is composed of primary panel and assist panel.
 *        Assist panel becomes visible by dragging the edge of the screen.
 *
 * @{
 */

/**
 * @WEARABLE_ONLY
 * @brief Add a new Eext_Panel
 * @details Because the return value is a handle of elm_layout object,
 *          the elm_layout APIs are applicable to eext_panel.
 *
 *          This widget emits the following signals:
 *          @li "assist,panel,visible": When the assist panel becomes draggable
 *                                      and visible
 *          @li "assist,panel,hidden": When the assist panel becomes completely
 *                                     invisible and not draggable
 *
 * @param[in] parent The parent object
 *
 * @return A new eext_panel handle, otherwise @c NULL if it cannot be created
 * @if WEARABLE @since_tizen 4.0
 * @endif
 */
EAPI Evas_Object* eext_panel_add(Evas_Object *parent);


/**
 * @WEARABLE_ONLY
 * @brief Set the content at a part of a given eext_panel widget
 *
 * @remarks Eext panel holds two (primary and assist) child objects as its
 *          contents in the given parts. This API sets a new content to a given
 *          part. If any object is already set as a content object in the given
 *          part, the previous object is automatically deleted with this call.
 *          If the @p content is NULL, this call will just delete the previous
 *          object.
 *
 *          The part names works as follows:
 *          @li "primary": If you set an object as the content of this part,
 *                         the object is visible by default before assist panel
 *                         is dragged.
 *          @li "assist": If you set an object as the content of this part,
 *                        the object is hidden by default and becomes visible
 *                        by dragging the edge of the screen.
 *
 * @param[in] obj Eext panel
 * @param[in] part_name Eext panel's part name to set
 * @param[in] content The new content for that part
 *
 * @if WEARABLE @since_tizen 4.0
 * @endif
 */
EAPI void eext_panel_part_content_set(Evas_Object *obj, const char *part_name, Evas_Object *content);

/**
 * @WEARABLE_ONLY
 * @brief Get the content from the part of the given eext_panel widget
 *
 * @param[in] obj Eext panel
 * @param[in] part_name Eext panel part name
 * @return The content of the object at the given part, otherwise @c NULL in case of an error
 *
 * @if WEARABLE @since_tizen 4.0
 * @endif
 */
EAPI Evas_Object* eext_panel_part_content_get(const Evas_Object *obj, const char *part_name);


/**
 * @}
 */

#ifdef __cplusplus
}
#endif

#endif /* __EFL_EXTENSION_PANEL_H__ */
