/*
 * Copyright (c) 2024 Samsung Electronics Co., Ltd All Rights Reserved
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *               http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 */

#ifndef __TIZEN_UIX_MMI_H__
#define __TIZEN_UIX_MMI_H__

#include <mmi-error.h>
#include <mmi-data.h>
#include <mmi-attribute.h>
#include <mmi-signal.h>
#include <mmi-workflow.h>

#ifdef __cplusplus
extern "C" {
#endif


/**
* @file mmi.h
* @brief This file contains the MMI API.
*/

/**
* @addtogroup CAPI_UIX_MMI_MODULE
* @{
*/

/**
 * @brief Enumeration for MMI state.
 * @details This enumeration defines the different states of the MMI client.
 * @since_tizen 9.0
*/
typedef enum {
    MMI_STATE_NONE = 0, /**< Indicates that there is no active state. */
    MMI_STATE_READY,    /**< Indicates that the MMI client is ready to use. */
} mmi_state_e;


/**
 * @brief Callback function type for MMI state change notifications.
 * @details This callback function is called whenever the state of the MMI client changes.
 * @since_tizen 9.0
 *
 * @param[in] state The new state of the MMI client.
 * @param[in] user_data User-provided data passed to the callback function.
 *
 * @return An integer value indicating the result of the callback execution.
 * @retval 0 on success, otherwise a negative error value.
 *
 * @see mmi_state_e
*/
typedef int (*mmi_state_changed_cb)(mmi_state_e state, void *user_data);


/**
 * @brief Handle for MMI workflow instance.
 * @details This handle represents a workflow instance created within the MMI framework.
 * @since_tizen 9.0
*/
typedef void* mmi_workflow_instance_h;


/**
 * @brief Callback function type for handling workflow output.
 * @details This callback function is called when the MMI framework generates output for a workflow instance.
 * @since_tizen 9.0
 * @remarks The @a instance should not be released.
 *          The @a data should not be released.
 *          The @a data can be used only in the callback. To use outside, make a copy.
 *
 * @param[in] instance The workflow instance handle for which the output is generated.
 *                     The @a instance is the same object for which the callback was set/added.
 *                     The @a instance is available until the workflow instance is released.
 * @param[in] name The name associated with the workflow output.
 *                 The @a name can be used only in the callback. To use outside, make a copy.
 * @param[in] data The output data generated by the workflow.
 * @param[in] user_data User-provided data passed to the callback function.
 *
 * @see mmi_workflow_instance_set_output_cb()
 * @see mmi_workflow_instance_unset_output_cb()
*/
typedef void (*mmi_workflow_output_cb)(mmi_workflow_instance_h instance, const char *name, mmi_data_h data, void *user_data);

/**
 * @brief Initializes the MMI framework.
 * @details This function initializes the MMI (Multimodal Interaction) framework. It must be called before any other MMI functions.
 * @since_tizen 9.0
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #MMI_ERROR_NONE Successful
 * @retval #MMI_ERROR_NOT_SUPPORTED Not supported
 * @retval #MMI_ERROR_OPERATION_FAILED Operation failed
 * @retval #MMI_ERROR_OUT_OF_MEMORY Out of memory
 *
 * @see mmi_deinitialize(void)
 */
int mmi_initialize(void);

/**
 * @brief Deinitializes the MMI framework.
 * @details This function deinitializes the MMI (Multimodal Interaction) framework. It must be called after all MMI operations are completed.
 * @since_tizen 9.0
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #MMI_ERROR_NONE Successful
 * @retval #MMI_ERROR_NOT_SUPPORTED Not supported
 * @retval #MMI_ERROR_OPERATION_FAILED Operation failed
 *
 * @see mmi_initialize(void)
 */
int mmi_deinitialize(void);

/**
 * @brief Sets a callback function to be invoked when the MMI state changes.
 * @details This function sets a callback function that will be called whenever the state of the MMI framework changes.
 * @since_tizen 9.0
 *
 * @param[in] callback The callback function to be called when state is changed
 * @param[in] user_data The user data to be passed to the callback function
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #MMI_ERROR_NONE Successful
 * @retval #MMI_ERROR_NOT_SUPPORTED Not supported
 * @retval #MMI_ERROR_INVALID_PARAMETER Invalid parameter
 *
 * @see mmi_state_changed_cb()
 * @see mmi_unset_state_changed_cb()
 */
int mmi_set_state_changed_cb(mmi_state_changed_cb callback, void *user_data);

/**
 * @brief Unsets the previously set callback function for MMI state changes.
 * @details This function removes the previously set callback function that was called whenever the state of the MMI framework changes.
 * @since_tizen 9.0
 *
 * @param[in] callback The callback function to be unset
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #MMI_ERROR_NONE Successful
 * @retval #MMI_ERROR_NOT_SUPPORTED Not supported
 * @retval #MMI_ERROR_INVALID_PARAMETER Invalid parameter
 *
 * @pre A callback function must have been set using mmi_set_state_changed_cb() before calling this function.
 * @post The callback function will no longer be called when the MMI state changes.
 *
 * @see mmi_state_changed_cb()
 * @see mmi_set_state_changed_cb()
 */
int mmi_unset_state_changed_cb(mmi_state_changed_cb callback);

/**
 * @brief Creates a new workflow instance from a standard workflow prototype.
 * @details This function creates a new workflow instance from a predefined standard workflow prototype.
 * @since_tizen 9.0
 * @remarks The @a instance should be released using mmi_workflow_instance_destroy().
 *
 * @param[in] type The type of the standard workflow prototype
 * @param[out] instance A pointer to the created workflow instance handle
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #MMI_ERROR_NONE Successful
 * @retval #MMI_ERROR_NOT_SUPPORTED Not supported
 * @retval #MMI_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #MMI_ERROR_OUT_OF_MEMORY Out of memory
 *
 * @par Example
 * @code
   #include <mmi.h>
   ...
   mmi_workflow_instance_h instance;
   mmi_standard_workflow_instance_create(MMI_STANDARD_WORKFLOW_VOICE_TOUCH, &instance);
   ...
   mmi_workflow_instance_destroy(instance);
 * @endcode
 *
 * @see mmi_standard_workflow_type_e
 * @see mmi_workflow_instance_destroy()
 */
int mmi_standard_workflow_instance_create(mmi_standard_workflow_type_e type, mmi_workflow_instance_h *instance);

/**
 * @brief Instantiates a workflow from a custom workflow prototype.
 * @details This function creates an instance of a workflow based on a custom workflow prototype.
 * @since_tizen 9.0
 * @remarks The @a instance should be released using mmi_workflow_instance_destroy().
 *
 * @param[in] workflow The handle to the custom workflow prototype.
 * @param[out] instance The handle to the newly created workflow instance.
 *
 * @return 0 on success, otherwise a negative error value
 * @retval #MMI_ERROR_NONE Successful
 * @retval #MMI_ERROR_NOT_SUPPORTED Not supported
 * @retval #MMI_ERROR_INVALID_PARAMETER Invalid parameter
 * @retval #MMI_ERROR_OUT_OF_MEMORY Out of memory
 *
 * @pre The custom workflow prototype must be valid and properly configured.
 * @post A new workflow instance will be created and assigned to the provided instance handle.
 *
 * @par Example
 * @code
   #include <mmi.h>
   ...
   const char *workflow_script = "@workflow\n...";
   mmi_workflow_h workflow;
   mmi_workflow_create_from_script(workflow_script, &workflow);
   ...
   mmi_workflow_instance_h instance;
   mmi_custom_workflow_instance_create(workflow, &instance);
   ...
   mmi_workflow_instance_destroy(instance);
 * @endcode
 *
 * @see mmi_workflow_h
 * @see mmi_workflow_instance_h
 */
int mmi_custom_workflow_instance_create(mmi_workflow_h workflow, mmi_workflow_instance_h *instance);

/**
 * @brief Destroys a workflow instance.
 * @details This function destroys a workflow instance and releases all associated resources.
 * @since_tizen 9.0
 *
 * @param[in] instance The handle to the workflow instance to be destroyed.
 *
 * @return 0 on success, otherwise a negative error value.
 * @retval #MMI_ERROR_NONE Successful.
 * @retval #MMI_ERROR_NOT_SUPPORTED Not supported
 * @retval #MMI_ERROR_INVALID_PARAMETER Invalid parameter.
 *
 * @pre The workflow instance must have been created and activated before calling this function.
 * @post The workflow instance is destroyed and its handle becomes invalid.
 *
 * @see mmi_standard_workflow_instance_create()
 * @see mmi_custom_workflow_instance_create()
 * @see mmi_workflow_instance_activate()
*/
int mmi_workflow_instance_destroy(mmi_workflow_instance_h instance);

/**
 * @brief Activates a workflow instance.
 * @details This function starts the execution of a workflow instance.
 * @since_tizen 9.0
 *
 * @param[in] instance The handle to the workflow instance to be activated.
 *
 * @return 0 on success, otherwise a negative error value.
 * @retval #MMI_ERROR_NONE Successful.
 * @retval #MMI_ERROR_NOT_SUPPORTED Not supported
 * @retval #MMI_ERROR_INVALID_PARAMETER Invalid parameter.
 *
 * @pre The workflow instance must have been created before calling this function.
 * @post The workflow instance is ready to process inputs and emit outputs.
 *
 * @see mmi_standard_workflow_instance_create()
 * @see mmi_custom_workflow_instance_create()
 * @see mmi_workflow_instance_deactivate()
*/
int mmi_workflow_instance_activate(mmi_workflow_instance_h instance);

/**
 * @brief Deactivates a workflow instance.
 * @details This function stops the execution of a workflow instance.
 * @since_tizen 9.0
 *
 * @param[in] instance The handle to the workflow instance to be deactivated.
 *
 * @return 0 on success, otherwise a negative error value.
 * @retval #MMI_ERROR_NONE Successful.
 * @retval #MMI_ERROR_NOT_SUPPORTED Not supported
 * @retval #MMI_ERROR_INVALID_PARAMETER Invalid parameter.
 *
 * @pre The workflow instance must have been created and activated before calling this function.
 * @post The workflow instance is stopped and cannot process inputs or emit outputs.
 *
 * @see mmi_workflow_instance_activate()
 * */
int mmi_workflow_instance_deactivate(mmi_workflow_instance_h instance);

/**
 * @brief Sets an attribute of a workflow instance.
 * @details This function sets an attribute to the workflow instance.
 * @since_tizen 9.0
 *
 * @param[in] instance The handle to the workflow instance.
 * @param[in] attribute The handle to the attribute to be set.
 *
 * @return 0 on success, otherwise a negative error value.
 * @retval #MMI_ERROR_NONE Successful.
 * @retval #MMI_ERROR_NOT_SUPPORTED Not supported
 * @retval #MMI_ERROR_INVALID_PARAMETER Invalid parameter.
 *
 * @pre The workflow instance must have been created before calling this function.
 * @post The attribute is set to the workflow instance.
 *
 * @see mmi_attribute_create()
*/
int mmi_workflow_instance_set_attribute(mmi_workflow_instance_h instance, mmi_attribute_h attribute);

/**
 * @brief Emits a signal to a workflow instance.
 * @details This function sends a signal to the workflow instance.
 * @since_tizen 9.0
 *
 * @param[in] instance The handle to the workflow instance.
 * @param[in] signal The handle to the signal to be emitted.
 *
 * @return 0 on success, otherwise a negative error value.
 * @retval #MMI_ERROR_NONE Successful.
 * @retval #MMI_ERROR_NOT_SUPPORTED Not supported
 * @retval #MMI_ERROR_INVALID_PARAMETER Invalid parameter.
 *
 * @pre The workflow instance must have been created and activated before calling this function.
 * @post The signal is processed by the workflow instance.
 *
 * @see mmi_signal_create()
*/
int mmi_workflow_instance_emit_signal(mmi_workflow_instance_h instance, mmi_signal_h signal);

/**
 * @brief Sets a callback function to receive workflow output.
 * @details This function sets a callback function that will be called when the workflow instance emits output.
 * @since_tizen 9.0
 *
 * @param[in] instance The handle to the workflow instance.
 * @param[in] name The name associated with the callback.
 * @param[in] callback The callback function to be set.
 * @param[in] user_data The user data to be passed to the callback function.
 *
 * @return 0 on success, otherwise a negative error value.
 * @retval #MMI_ERROR_NONE Successful.
 * @retval #MMI_ERROR_NOT_SUPPORTED Not supported
 * @retval #MMI_ERROR_INVALID_PARAMETER Invalid parameter.
 *
 * @pre The workflow instance must have been created before calling this function.
 * @post The callback function will be called when the workflow instance emits output.
 *
 * @see mmi_workflow_output_cb()
 * @see mmi_workflow_instance_unset_output_cb()
*/
int mmi_workflow_instance_set_output_cb(mmi_workflow_instance_h instance, const char *name, mmi_workflow_output_cb callback, void *user_data);

/**
 * @brief Unsets a callback function for an output of a workflow instance.
 * @details This function unsets a callback function that was set by mmi_workflow_instance_set_output_cb().
 *          After this function is called, the callback function will not be called when the workflow instance emits output.
 * @since_tizen 9.0
 *
 * @param[in] instance The handle to the workflow instance
 * @param[in] callback The callback function pointer to unset
 *
 * @return @c 0 on success, otherwise a negative error value
 * @retval #MMI_ERROR_NONE Successful.
 * @retval #MMI_ERROR_NOT_SUPPORTED Not supported
 * @retval #MMI_ERROR_INVALID_PARAMETER Invalid parameter.
 *
 * @pre The workflow instance must have been created before calling this function.
 * @post The callback function will not be called when the workflow instance emits output.
 *
 * @see mmi_workflow_output_cb()
 * @see mmi_workflow_instance_set_output_cb()
*/
int mmi_workflow_instance_unset_output_cb(mmi_workflow_instance_h instance, mmi_workflow_output_cb callback);

/**
 * @}
 */

#ifdef __cplusplus
}
#endif



#endif /* __TIZEN_UIX_MMI_H__ */
