/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "samplenativeimeconfig.h"
#include <dlog.h>

/**
 * @brief ime layout initialize
 *
 * @param keyboard_layout the pointer of layout
 * @param mode the display mode of keyboard
 * @param type the layout type
 *
 * @return initialize ime layout, success or not
 */
Eina_Bool sample_native_ime_layout_init(IMELayout *keyboard_layout, IMEDisplayMode mode, IMEKeyboardLayoutType type)
{
	if (NULL == keyboard_layout)
		return EINA_FALSE;

	keyboard_layout->fvalid = EINA_TRUE;
	keyboard_layout->name = g_keyboard_layout_type[mode][type].name;
	keyboard_layout->display_mode = g_keyboard_layout_type[mode][type].display_mode;
	keyboard_layout->width = g_keyboard_layout_type[mode][type].width;
	keyboard_layout->height = g_keyboard_layout_type[mode][type].height;
	keyboard_layout->key_width = g_keyboard_layout_type[mode][type].key_width;
	keyboard_layout->key_height = g_keyboard_layout_type[mode][type].key_height;
	keyboard_layout->key_spacing = g_keyboard_layout_type[mode][type].key_spacing;
	keyboard_layout->row_spacing = g_keyboard_layout_type[mode][type].row_spacing;
	keyboard_layout->row_number = g_keyboard_layout_type[mode][type].row_number;

	int i = 0;
	int j = 0;
	int k = 0;
	/* initialize row information for layout */
	for (i = 0; i < keyboard_layout->row_number; ++i)
	{
		keyboard_layout->row_infos[i].x = g_keyboard_layout_type[mode][type].row_info[i].x;
		keyboard_layout->row_infos[i].y = g_keyboard_layout_type[mode][type].row_info[i].y;
		if (SAMPLE_NATIVE_IME_AUTO_CAL == keyboard_layout->row_infos[i].y)
			keyboard_layout->row_infos[i].y = keyboard_layout->row_infos[i - 1].y + keyboard_layout->key_height + keyboard_layout->row_spacing;

		keyboard_layout->row_infos[i].key_number = g_keyboard_layout_type[mode][type].row_info[i].key_number;
	}

	/* initialize key row by row */
	for (i = 0; i < keyboard_layout->row_number; ++i)
	{
		int key_number = keyboard_layout->row_infos[i].key_number;
		int curPosX = keyboard_layout->row_infos[i].x;
		for (j = 0; j < key_number; ++j)
		{
			keyboard_layout->button_key[i][j].x = curPosX;
			keyboard_layout->button_key[i][j].y = keyboard_layout->row_infos[i].y;
			keyboard_layout->button_key[i][j].width = g_keyboard_layout_type[mode][type].key_label[i][j].width;
			/* key width is default value */
			if (SAMPLE_NATIVE_IME_DEFAULT_VAL == keyboard_layout->button_key[i][j].width)
				keyboard_layout->button_key[i][j].width = keyboard_layout->key_width;

			curPosX += keyboard_layout->button_key[i][j].width + keyboard_layout->key_spacing;

			keyboard_layout->button_key[i][j].height = g_keyboard_layout_type[mode][type].key_label[i][j].height;
			/* key height is default value */
			if (SAMPLE_NATIVE_IME_DEFAULT_VAL == keyboard_layout->button_key[i][j].height)
				keyboard_layout->button_key[i][j].height = keyboard_layout->key_height;

			/* initialize key value */
			for (k = 0; k < IME_SHIFT_STATE_MAX; ++k)
			{
				keyboard_layout->button_key[i][j].key_value[k][0] = g_keyboard_layout_type[mode][type].key_label[i][j].main_key_value[k];
				keyboard_layout->button_key[i][j].key_value[k][1] = g_keyboard_layout_type[mode][type].key_label[i][j].sub_key;
			}

			keyboard_layout->button_key[i][j].button_type = g_keyboard_layout_type[mode][type].key_label[i][j].button_type;
			keyboard_layout->button_key[i][j].key_type = g_keyboard_layout_type[mode][type].key_label[i][j].key_type;
			keyboard_layout->button_key[i][j].long_key_type = g_keyboard_layout_type[mode][type].key_label[i][j].long_key_type;

			IMEKeyRowInfo rowinfo = {0};
			rowinfo.row = i;
			rowinfo.col = j;
			keyboard_layout->button_key[i][j].row_info = rowinfo;

			/* initialize key label */
			for (k = 0; k < IME_SHIFT_STATE_MAX; ++k)
			{
				keyboard_layout->button_key[i][j].main_label[k].label_text = g_keyboard_layout_type[mode][type].key_label[i][j].main_key[k];
				keyboard_layout->button_key[i][j].main_label[k].font_size = g_keyboard_layout_type[mode][type].key_label[i][j].main_font_size[k];
				keyboard_layout->button_key[i][j].sub_label[k].label_text = g_keyboard_layout_type[mode][type].key_label[i][j].sub_key;
				keyboard_layout->button_key[i][j].sub_label[k].font_size = g_keyboard_layout_type[mode][type].key_label[i][j].sub_font_size[k];
			}
		}
	}
	return EINA_TRUE;
}

/**
 * @brief get keyboard layout by keyboard name
 *
 * @param name name of layout
 *
 * @return the keyboard layout type
 */
IMEKeyboardLayoutType sample_native_ime_get_layout_type_by_name(char *name)
{
	if (NULL == name)
		return IME_KEYBOARD_LAYOUT_MAX;

	/* search layout type by layout name */
	int i = 0;
	for (; i < IME_KEYBOARD_LAYOUT_MAX; ++i)
	{
		if (!strcmp(name, g_keyboard_layout_name_to_value[i].name))
			break;
	}

	return (IMEKeyboardLayoutType)i;
}
