/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "samplenativeimeoption.h"
#include "samplenativeimefeedback.h"
#include <efl_extension.h>
#include <dlog.h>

/* option item id */
typedef enum SETTING_ITEM_ID
{
	SETTING_ITEM_ID_SOUND,
	SETTING_ITEM_ID_VIBRATION,
	SETTING_ITEM_ID_MAX,
} SETTING_ITEM_ID;

#define SAMPLE_NATIVE_IME_OPTION "Keyboard settings"
#define SAMPLE_NATIVE_IME_TAP_FEEDBACK "Key-tap feedback"
#define SAMPLE_NATIVE_IME_SOUND "Sound"
#define SAMPLE_NATIVE_IME_VIBRATION "Vibration"
#define MAX_LABEL_NAME_LENGTH 20

/* the data for items in option window */
typedef struct setting_item_data
{
	char text[MAX_LABEL_NAME_LENGTH];
	SETTING_ITEM_ID id;
} setting_item_data;

static Evas_Object *g_option_window = NULL;							   /* the option window */
static Elm_Genlist_Item_Class *g_genlist_group_title_style = NULL;	   /* the style for group title */
static Elm_Genlist_Item_Class *g_genlist_main_text_radio_style = NULL; /* the style for item */
static setting_item_data g_setting_item_data[SETTING_ITEM_ID_MAX];	   /* the data for item*/

/* the callback for naviframe back */
static void sample_native_ime_naviframe_back_cb(void *data, Evas_Object *obj, void *event_info)
{
	eext_object_event_callback_del(obj, EEXT_CALLBACK_BACK, sample_native_ime_naviframe_back_cb);
	ime_destroy_option_window(g_option_window);
	g_option_window = NULL;
}

static void sample_native_ime_naviframe_back_button_cb(void *data, Evas_Object *obj, void *event_info)
{
	evas_object_smart_callback_del(obj, "clicked", sample_native_ime_naviframe_back_button_cb);
	ime_destroy_option_window(g_option_window);
	g_option_window = NULL;
}

/**
 * @brief setting the text for item
 *
 * @param data the pointer of data
 * @param obj  the pointer of Evas_Object
 * @param part the part which want to set text
 *
 * @return the text for item
 */
static char *sample_native_ime_main_gl_text_get(void *data, Evas_Object *obj, const char *part)
{
	setting_item_data *item_data = (setting_item_data *)data;
	if (item_data)
	{
		if (!strcmp(part, "elm.text.main.left.top") || !strcmp(part, "elm.text.main.left") || !strcmp(part, "elm.text.main") || !strcmp(part, "elm.text") || !strcmp(part, "elm.text.1"))
			return strdup(item_data->text);
	}

	return NULL;
}

/**
 * @brief set feedback status
 *
 * @param data the pointer of data
 * @param state the feedback state
 *
 */
static void sample_native_ime_set_feedback_status(void *data, Eina_Bool state)
{
	SETTING_ITEM_ID id = (SETTING_ITEM_ID)data;
	switch (id)
	{
	case SETTING_ITEM_ID_SOUND:
	{
		sample_native_ime_set_sound_feedback_status(state);
		break;
	}
	case SETTING_ITEM_ID_VIBRATION:
	{
		sample_native_ime_set_vibration_feedback_status(state);
		break;
	}
	default:
	{
		dlog_print(DLOG_DEBUG, LOG_TAG, "wrong setting item id: %d", id);
		break;
	}
	}
}

static void sample_native_ime_sound_check_button_change_cb(void *data, Evas_Object *obj, void *event_info)
{
	Eina_Bool state = elm_check_state_get(obj);
	sample_native_ime_set_feedback_status(data, state);
}

/**
 * @brief set item content in option window
 *
 * @param data the pointer of data
 * @param obj the pointer of Evas_Object*
 * @param part the part to get content
 *
 * @return the pointer of the content
 */
static Evas_Object *sample_native_ime_main_text_radio_content_get(void *data, Evas_Object *obj, const char *part)
{
	setting_item_data *item_data = (setting_item_data *)data;
	SETTING_ITEM_ID id = (SETTING_ITEM_ID)item_data->id;
	Eina_Bool state = EINA_FALSE;
	/* get the initial state for sound and vibration */
	switch (id)
	{
	case SETTING_ITEM_ID_SOUND:
	{
		state = sample_native_ime_get_sound_feedback_status();
		break;
	}
	case SETTING_ITEM_ID_VIBRATION:
	{
		state = sample_native_ime_get_vibration_feedback_status();
		break;
	}
	default:
	{
		dlog_print(DLOG_DEBUG, LOG_TAG, "wrong id of setting item: %d", id);
		break;
	}
	}
	/* set the status for check */
	Evas_Object *ck = NULL;
	if (obj)
	{
		if (!strcmp(part, "elm.swallow.end"))
		{
			ck = elm_check_add(obj);
			elm_object_style_set(ck, "on&off");
			elm_check_state_set(ck, state);
			evas_object_propagate_events_set(ck, EINA_FALSE);
			evas_object_show(ck);
			evas_object_smart_callback_add(ck, "changed", sample_native_ime_sound_check_button_change_cb, (void *)id);
		}
	}
	return ck;
}

/**
 * @brief create item style for genlist
 *
 * @param type the type of option window
 *
 */
static void sample_native_ime_create_genlist_item_classes(ime_option_window_type_e type)
{
	/* style for group title */
	g_genlist_group_title_style = elm_genlist_item_class_new();
	if (g_genlist_group_title_style)
	{
		g_genlist_group_title_style->item_style = "group_index";
		g_genlist_group_title_style->func.text_get = sample_native_ime_main_gl_text_get;
		g_genlist_group_title_style->func.content_get = NULL;
		g_genlist_group_title_style->func.state_get = NULL;
		g_genlist_group_title_style->func.del = NULL;
	}

	/* style for items */
	g_genlist_main_text_radio_style = elm_genlist_item_class_new();
	if (g_genlist_main_text_radio_style)
	{
		g_genlist_main_text_radio_style->item_style = "type1";
		g_genlist_main_text_radio_style->func.text_get = sample_native_ime_main_gl_text_get;
		g_genlist_main_text_radio_style->func.content_get = sample_native_ime_main_text_radio_content_get;
		g_genlist_main_text_radio_style->func.state_get = NULL;
		g_genlist_main_text_radio_style->func.del = NULL;
	}
}

/**
 * @brief callback function when item selected
 *
 * @param data the pointer of data
 * @param obj the pointer of Evas_Object
 * @param event_info the event information
 *
 */
static void sample_native_ime_genlist_item_sel_cb(void *data, Evas_Object *obj, void *event_info)
{
	Elm_Object_Item *item = (Elm_Object_Item *)event_info;
	if (item)
		elm_genlist_item_selected_set(item, EINA_FALSE);

	Evas_Object *ck = elm_object_item_part_content_get(item, "elm.swallow.end");
	if (ck == NULL)
		ck = elm_object_item_part_content_get(item, "elm.icon");

	/* reverse the current state */
	Eina_Bool state = !elm_check_state_get(ck);
	elm_check_state_set(ck, state);
	sample_native_ime_set_feedback_status(data, state);
}

/**
 * @brief create main view for option window
 *
 * @param parent the pointer of parent object
 * @param naviframe the pointer of naviframe object
 * @param type the option windwo type
 *
 * @return the pointer of main view object
 */
Evas_Object *sample_native_ime_create_option_main_view(Evas_Object *parent, Evas_Object *naviframe, ime_option_window_type_e type)
{
	sample_native_ime_create_genlist_item_classes(type);

	/* add genlist */
	Evas_Object *genlist = elm_genlist_add(naviframe);
	elm_genlist_mode_set(genlist, ELM_LIST_COMPRESS);
	evas_object_size_hint_weight_set(genlist, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_align_set(genlist, EVAS_HINT_FILL, EVAS_HINT_FILL);

	Elm_Object_Item *item = elm_genlist_item_append(genlist, g_genlist_group_title_style, SAMPLE_NATIVE_IME_TAP_FEEDBACK, NULL, ELM_GENLIST_ITEM_NONE, NULL, NULL);
	elm_genlist_item_select_mode_set(item, ELM_OBJECT_SELECT_MODE_DISPLAY_ONLY);

	/* add sound item */
	strncpy(g_setting_item_data[SETTING_ITEM_ID_SOUND].text, SAMPLE_NATIVE_IME_SOUND, MAX_LABEL_NAME_LENGTH - 1);
	g_setting_item_data[SETTING_ITEM_ID_SOUND].id = SETTING_ITEM_ID_SOUND;
	elm_genlist_item_append(genlist, g_genlist_main_text_radio_style, (void *)&g_setting_item_data[SETTING_ITEM_ID_SOUND], NULL, ELM_GENLIST_ITEM_NONE, sample_native_ime_genlist_item_sel_cb, (void *)SETTING_ITEM_ID_SOUND);
	/* add vibration item */
	strncpy(g_setting_item_data[SETTING_ITEM_ID_VIBRATION].text, SAMPLE_NATIVE_IME_VIBRATION, MAX_LABEL_NAME_LENGTH - 1);
	g_setting_item_data[SETTING_ITEM_ID_VIBRATION].id = SETTING_ITEM_ID_VIBRATION;
	elm_genlist_item_append(genlist, g_genlist_main_text_radio_style, (void *)&g_setting_item_data[SETTING_ITEM_ID_VIBRATION], NULL, ELM_GENLIST_ITEM_NONE, sample_native_ime_genlist_item_sel_cb, (void *)SETTING_ITEM_ID_VIBRATION);

	return genlist;
}

/**
 * @brief option window destroyed
 *
 * @param
 *
 * @return
 */
void sample_native_ime_option_window_destroyed()
{
	/* free the style */
	if (g_genlist_group_title_style)
	{
		elm_genlist_item_class_free(g_genlist_group_title_style);
		g_genlist_group_title_style = NULL;
	}

	if (g_genlist_main_text_radio_style)
	{
		elm_genlist_item_class_free(g_genlist_main_text_radio_style);
		g_genlist_main_text_radio_style = NULL;
	}
}

/**
 * @brief option window created
 *
 * @param window the pointer of option window
 * @param the option window type
 */
void sample_native_ime_option_window_created(Evas_Object *window, ime_option_window_type_e type)
{
	if (window == NULL)
		return;
	if (NULL == g_option_window)
		g_option_window = window;

	/* set indicator */
	elm_win_indicator_mode_set(window, ELM_WIN_INDICATOR_SHOW);
	elm_win_indicator_opacity_set(window, ELM_WIN_INDICATOR_OPAQUE);

	/* create conformant */
	Evas_Object *conformant = elm_conformant_add(window);
	evas_object_size_hint_weight_set(conformant, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_align_set(conformant, EVAS_HINT_FILL, EVAS_HINT_FILL);
	elm_win_resize_object_add(window, conformant);
	elm_win_conformant_set(window, EINA_TRUE);
	evas_object_show(conformant);

	/* create header bg */
	Evas_Object *bg = elm_bg_add(conformant);
	elm_object_style_set(bg, "indicator/headerbg");
	elm_object_part_content_set(conformant, "elm.swallow.indicator_bg", bg);
	evas_object_show(bg);

	/* add naviframe */
	Evas_Object *naviframe = elm_naviframe_add(conformant);
	elm_naviframe_prev_btn_auto_pushed_set(naviframe, EINA_FALSE);
	eext_object_event_callback_add(naviframe, EEXT_CALLBACK_BACK, sample_native_ime_naviframe_back_cb, NULL);
	evas_object_size_hint_weight_set(naviframe, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
	evas_object_size_hint_align_set(naviframe, EVAS_HINT_FILL, EVAS_HINT_FILL);

	/* add list */
	Evas_Object *list = sample_native_ime_create_option_main_view(conformant, naviframe, type);

	/* add a back button to naviframe */
	Evas_Object *back_button = elm_button_add(naviframe);
	elm_object_style_set(back_button, "naviframe/back_btn/default");
	evas_object_smart_callback_add(back_button, "clicked", sample_native_ime_naviframe_back_button_cb, NULL);
	elm_naviframe_item_push(naviframe, SAMPLE_NATIVE_IME_OPTION, back_button, NULL, list, NULL);

	elm_object_content_set(conformant, naviframe);
	evas_object_show(naviframe);
	evas_object_show(window);
}
