/*
 * Web IDE - beautifier
 *
 * Copyright (c) 2012 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Contact:
 * Changhyun Lee <changhyun1.lee@samsung.com>
 * Hyeongseok Heo <hyeongseok.heo@samsung.com>
 * BonYong Lee <bonyong.lee@samsung.com>
 * Kangho Kim <kh5325.kim@samsung.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Contributors:
 * - S-Core Co., Ltd
 *
 */
package org.tizen.web.compressor.beautifier.core;

import static org.tizen.common.util.IOUtil.getString;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.HashSet;
import java.util.Set;

import org.mozilla.javascript.Context;
import org.mozilla.javascript.Scriptable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.tizen.common.util.FileUtil;
import org.tizen.common.util.FilenameUtil;

/**
 * AbstractBeautifier.
 *
 * Abstract class for {@link IBeautifier}
 *
 * @author ChangHyun Lee{@literal <changhyun1.lee@samsung.com>} (S-Core)
 * @author BonYong Lee{@literal <bonyong.lee@samsung.com>} (S-Core)
 */
abstract public class AbstractBeautifier implements IBeautifier {
    /**
     * Logger for this
     */
    protected final Logger logger = LoggerFactory.getLogger( getClass() );

    /**
     * Extension {@link Set} for Extension-base operation
     */
    protected final Set<String> extensions;

    /**
     * Context for javascript enginge
     */
    protected final Context context;

    /**
     * Scope for javascript engine
     */
    protected final Scriptable scope;

    /**
     * Excutable client code of triggering
     */
    protected final String executable;

    /**
     * Beautifier code
     */
    protected final String beautifier;

    /**
     * Constructor with extensions, beautifier code and client code
     * 
     * @param extensions files' extensions
     * @param beautifier beautifier code in javascript
     * @param executable client code in javascript
     */
    public
    AbstractBeautifier( final String[] extensions, final String beautifier, final String executable ) {
        this.extensions = new HashSet<String>();
        this.beautifier = beautifier;
        this.executable = executable;
        this.context = Context.enter();
        this.context.setLanguageVersion( Context.VERSION_1_6 );
        this.scope = context.initStandardObjects();

        for ( final String ext : extensions ) {
            this.extensions.add( ext.toLowerCase() );
        }

        context.evaluateString( scope, getBeautifier(), "beautifier", 1, null ); //$NON-NLS-1$
    }

    /**
     * Load beautifier code from same package in classpath
     * 
     * @param name file name
     * @return contents of javascript
     * 
     * @throws IOException If file does NOT exist in path
     */
    protected static String load( final String name ) throws IOException {
        final ClassLoader cl = AbstractBeautifier.class.getClassLoader();
        final String path = FilenameUtil.addTailingPath( AbstractBeautifier.class.getPackage().getName().replace( '.', '/' ), name );
        final InputStream in = cl.getResourceAsStream( path );
        return getString( in, true );
    }

    /**
     * Return extensions of file
     * 
     * @return {@link #extensions}
     * 
     */
    protected Set<String> getExtensions() {
        return this.extensions;
    }

    /**
     * Return beautifier code
     * @return {@link #beautifier}
     */
    protected String getBeautifier() {
        return this.beautifier;
    }

    /**
     * Return client code
     * 
     * @return {@link #executable}
     */
    protected String getExecutableCode() {
        return this.executable;
    }

    @Override
    public boolean canHandle( final String name ) {
        logger.trace( "Name: {}", name );
        String fileExt = FileUtil.getFileExtension( name );
        logger.trace( "Extension: {}", fileExt );
        if ( fileExt == null ) {
            logger.debug( "No extension" );
            return false;
        }

        fileExt = fileExt.toLowerCase();

        Set<String> exts = getExtensions();
        logger.debug( "Extension candidate: {}", exts );
        return exts.contains( fileExt );
    }

    @Override
    public void decompress( final InputStream source, final OutputStream target ) throws IOException {
        scope.put( "contents", scope, getString( source ) ); //$NON-NLS-1$
        context.evaluateString( scope, getExecutableCode(), "beautifier", 1, null );

        final Object result = scope.get( "result", scope ); //$NON-NLS-1$
        if ( null == result ) {
            throw new IOException( "Invalid result" );
        }

        target.write( result.toString().getBytes() );
    }

}
