/*
 * Web IDE - beautifier
 *
 * Copyright (c) 2012 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Contact:
 * Changhyun Lee <changhyun1.lee@samsung.com>
 * Hyeongseok Heo <hyeongseok.heo@samsung.com>
 * BonYong Lee <bonyong.lee@samsung.com>
 * Kangho Kim <kh5325.kim@samsung.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Contributors:
 * - S-Core Co., Ltd
 *
 */
package org.tizen.web.compressor.beautifier.core;

import static org.tizen.common.util.IOUtil.getBytes;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.HashSet;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Beautifier.
 *
 * beautify code to be minified html, js and css file.
 *
 * @author ChangHyun Lee{@literal <changhyun1.lee@samsung.com>} (S-Core)
 */
public class Beautifier {
    /**
     * Logger for this
     */
    protected final Logger logger = LoggerFactory.getLogger( getClass() );

    /**
     * Workers to handle
     */
    protected final Set<IBeautifier> beautifiers = new HashSet<IBeautifier>();

    /**
     * Check if file can be handled
     * 
     * @param name file name
     * 
     * @return If file can be handle
     */
    public boolean canHandle( final String name ) {
        logger.debug( "Name: {}, Beautifiers: {}", name, this.beautifiers );
        for ( final IBeautifier beautifier : this.beautifiers ) {
            if ( beautifier.canHandle( name ) ) {
                return true;
            }
        }
        return false;
    }

    /**
     * Add beautifier
     * 
     * @param beautifier beautifier to add
     */
    public void addBeautifier( final IBeautifier beautifier ) {
        this.beautifiers.add( beautifier );
    }

    /**
     * beautify code
     *
     * @param name target file name
     * @param in {@link InputStream} to read
     * @param out {@link OutputStream} to write
     *
     * @throws Exception
     */
    public void decompress( String name, InputStream in, OutputStream out ) throws Exception {
        boolean bProcess = false;
        byte[] bytes = getBytes( in, true );
        logger.debug( "Beautifiers: {}", this.beautifiers );

        for ( final IBeautifier beautifier : this.beautifiers ) {
            logger.trace( "Beautifier: {}", beautifier );
            if ( !beautifier.canHandle( name ) ) {
                continue;
            }

            final ByteArrayInputStream byteIn = new ByteArrayInputStream( bytes );
            final ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
            beautifier.decompress( byteIn, byteOut );
            bytes = byteOut.toByteArray();
            bProcess = true;
        }

        if ( !bProcess ) {
            throw new Exception( name + " is not supported file."); //$NON-NLS-1$
        }

        if ( null != out ) {
            out.write( bytes );
        }
    }

}
