//===--- tools/APIChecker/APICheckerTool.cpp - APIChecker tool
//--------------===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
//===----------------------------------------------------------------------===//
//
//  This file implements a APICheckerTool that checks for use of deprecated and
//  unprivileged functions in Tizen.
//
//  This file contains the Frontend of the Tool and implements the
//  ASTFrontendAction.
//
//===----------------------------------------------------------------------===//

#include "clang/AST/ASTConsumer.h"
#include "clang/Tooling/Tooling.h"
#include "clang/Frontend/FrontendActions.h"
#include "clang/Tooling/CommonOptionsParser.h"
#include "clang/Frontend/CompilerInstance.h"
#include "APICheckerConsumer.h"
#include <stdio.h>

using namespace clang;
using namespace clang::tooling;
using namespace llvm;

// Mark all our options with this category, everything else (except for -version
// and -help) will be hidden.
cl::OptionCategory APICheckerOptions("APIChecker options");

cl::opt<std::string> OutputFilename("o", cl::desc("Specify output filename"),
                                    cl::value_desc("filename"));
static cl::opt<std::string>
    FilePath("file-path",
             cl::desc("The file corresponding to the file buffer."),
             cl::cat(APICheckerOptions));

static cl::opt<std::string> CheckUnused("check-unused",
                                        cl::desc("Check for unused privilege"),
                                        cl::cat(APICheckerOptions));

cl::opt<std::string> PrivilegeListOpt("privilege-list",
                                      cl::desc("Current privileges"),
                                      cl::cat(APICheckerOptions));

void CheckAndReportUnusedPrivileges(std::vector<std::string> PrivFileInput) {

  std::unordered_map<std::string, bool> DeclaredPrivileges;
  std::string PrivilegeList(PrivilegeListOpt.c_str());
  char *PrivString = const_cast<char *>(PrivilegeList.c_str());
  char *pch = strtok(PrivString, " ,");
  while (pch != NULL) {
    DeclaredPrivileges[std::string(pch)] = true;
    pch = strtok(NULL, " ,");
  }
  for (auto I = PrivFileInput.begin(), E = PrivFileInput.end(); I != E; ++I) {
    std::ifstream infile(*I);
    std::string privilege;
    while (infile >> privilege) {
      if ((DeclaredPrivileges.find(privilege) != DeclaredPrivileges.end()) &&
          (DeclaredPrivileges[privilege] == true))
        DeclaredPrivileges[privilege] = false;
    }
  }

  for (auto I = DeclaredPrivileges.begin(), E = DeclaredPrivileges.end();
       I != E; ++I) {
    if (I->second == true) {
      std::string Privilege = "../tizen-manifest.xml:0:0: warning: Privilege " +
                              I->first + " unused in project\n";
      fputs(Privilege.c_str(), stderr);
    }
  }
}

class APICheckerFrontendAction : public ASTFrontendAction {

protected:
  std::unique_ptr<ASTConsumer> CreateASTConsumer(CompilerInstance &CI,
                                                 StringRef infile) {

    APICheckerConsumer *Consumer =
        new APICheckerConsumer(&CI.getASTContext(), infile);
    return std::unique_ptr<ASTConsumer>(Consumer);
  }
};

int main(int argc, const char **argv) {
  std::vector<std::string> PrivFileInput;
  CommonOptionsParser OptionsParser(argc, argv, APICheckerOptions);

  if (!OutputFilename.empty()) {
    std::ofstream myfile;
    myfile.open(OutputFilename);
    myfile.close();
  }

  if (!CheckUnused.empty()) {
    for (int i = 0; i < argc; ++i) {
      std::string option = std::string(argv[i]);
      if (option.find(".txt") != std::string::npos)
        PrivFileInput.push_back(option);
    }
  }

  if (!PrivFileInput.empty()) {
    CheckAndReportUnusedPrivileges(std::move(PrivFileInput));
    return 0;
  }

  std::vector<std::string> srcList = OptionsParser.getSourcePathList();
  ClangTool Tool(OptionsParser.getCompilations(), std::move(srcList));
  if (srcList.size() != 1)
    return 0;

  std::ifstream ifs(FilePath.c_str());
  std::string content((std::istreambuf_iterator<char>(ifs)),
                      (std::istreambuf_iterator<char>()));

  if (!content.empty())
    Tool.mapVirtualFile(srcList[0], content);

  std::unique_ptr<FrontendActionFactory> FrontendFactory =
      newFrontendActionFactory<APICheckerFrontendAction>();
  Tool.run(FrontendFactory.get());

  return 0;
}
