/**
 * @internal
 * @defgroup Thumb Thumbnail
 * @ingroup Elementary
 *
 * @image html thumb_inheritance_tree.png
 * @image latex thumb_inheritance_tree.eps
 *
 * @image html img/widget/thumb/preview-00.png
 * @image latex img/widget/thumb/preview-00.eps
 *
 * A thumbnail object is used for displaying the thumbnail of an image
 * or video. You must have compiled Elementary with @c Ethumb_Client
 * support. Also, Ethumb's DBus service must be present and
 * auto-activated in order to have thumbnails generated. You must also
 * have a @b session bus, not a @b system one.
 *
 * Once the thumbnail object becomes visible, it checks if there
 * is a previously generated thumbnail image for the file set on
 * it. If not, it starts generating this thumbnail.
 *
 * Different configuration settings causes different thumbnails to
 * be generated even on the same file.
 *
 * Generated thumbnails are stored under @c $HOME/.thumbnails/. Check
 * Ethumb's documentation to change this path, and to see other
 * configuration options.
 *
 * This widget emits the following signals:
 * - @c "clicked" - This is called when a user has clicked the
 *                  thumbnail object without dragging it around.
 * - @c "clicked,double" - This is called when a user has double-clicked
 *                         the thumbnail object.
 * - @c "press" - This is called when a user has pressed down over the
 *                thumbnail object.
 * - @c "generate,start" - The thumbnail generation has started.
 * - @c "generate,stop" - The generation process has stopped.
 * - @c "generate,error" - The thumbnail generation failed.
 * - @c "load,error" - The thumbnail image loading failed.
 *
 * Available styles:
 * - @c "default"
 * - @c "noframe"
 *
 * @{
 */

/**
 * @enum Elm_Thumb_Animation_Setting
 * @typedef Elm_Thumb_Animation_Setting
 *
 * @brief Enumeration to set if a video thumbnail is animating.
 *
 * @ingroup Thumb
 */
typedef enum
{
   ELM_THUMB_ANIMATION_START = 0, /**< Play animation once */
   ELM_THUMB_ANIMATION_LOOP, /**< Keep playing animation until stop is requested */
   ELM_THUMB_ANIMATION_STOP, /**< Stop playing the animation */
   ELM_THUMB_ANIMATION_LAST
} Elm_Thumb_Animation_Setting;

/**
 * @brief Adds a new thumb object to the parent.
 *
 * @param[in] parent The parent object
 * @return The new object, otherwise @c NULL if it cannot be created
 *
 * @see elm_thumb_file_set()
 * @see elm_thumb_ethumb_client_get()
 *
 * @ingroup Thumb
 */
EAPI Evas_Object                *elm_thumb_add(Evas_Object *parent);

/**
 * @brief Reloads a thumbnail if it is generated before.
 *
 * @remarks This is useful if the ethumb client configuration changed, like its
 *          size, aspect, or any other property one set in the handle returned
 *          by elm_thumb_ethumb_client_get().
 *
 * @remarks If the options didn't change, the thumbnail won't be generated again, but
 *          the old one is still used.
 *
 * @param[in] obj The thumb object to reload
 *
 * @see elm_thumb_file_set()
 *
 * @ingroup Thumb
 */
EAPI void                        elm_thumb_reload(Evas_Object *obj);

/**
 * @brief Sets the file that is used as a thumbnail @b source.
 *
 * @remarks The file can be an image or a video (in that case, acceptable
 *          extensions are: avi, mp4, ogv, mov, mpg and wmv). To start the
 *          video animation, use the function elm_thumb_animate().
 *
 * @param[in] obj The thumb object
 * @param[in] file The path to the file that is used as a thumbnail source
 * @param[in] key The key used in case of an EET file
 *
 * @see elm_thumb_file_get()
 * @see elm_thumb_reload()
 * @see elm_thumb_animate()
 *
 * @ingroup Thumb
 */
EAPI void                        elm_thumb_file_set(Evas_Object *obj, const char *file, const char *key);

/**
 * @brief Gets the image or video path and key used to generate the thumbnail.
 *
 * @param[in] obj The thumb object
 * @param[out] file A pointer to the filename
 * @param[out] key A pointer to the key
 *
 * @see elm_thumb_file_set()
 * @see elm_thumb_path_get()
 *
 * @ingroup Thumb
 */
EAPI void                        elm_thumb_file_get(const Evas_Object *obj, const char **file, const char **key);

/**
 * @brief Gets the path and key to the image or video thumbnail generated by ethumb.
 *
 * @remarks One just needs to make sure that the thumbnail is generated before getting
 *          its path, otherwise the path is @c NULL. One way to do that is by asking
 *          for the path when/after the "generate,stop" smart callback is called.
 *
 * @param[in] obj The thumb object
 * @param[out] file A pointer to the thumb path
 * @param[out] key A pointer to the thumb key
 *
 * @see elm_thumb_file_get()
 *
 * @ingroup Thumb
 */
EAPI void                        elm_thumb_path_get(const Evas_Object *obj, const char **file, const char **key);

/**
 * @brief Sets the animation state for the thumb object. If its content is an animated
 *        video, you may start/stop the animation or tell it to play continuously and
 *        looping.
 *
 * @param[in] obj The thumb object
 * @param[in] s The animation setting
 *
 * @see elm_thumb_file_set()
 *
 * @ingroup Thumb
 */
EAPI void                        elm_thumb_animate_set(Evas_Object *obj, Elm_Thumb_Animation_Setting s);

/**
 * @brief Gets the animation state for the thumb object.
 *
 * @param[in] obj The thumb object
 * @return The animation setting, otherwise @c ELM_THUMB_ANIMATION_LAST
 *         on errors
 *
 * @see elm_thumb_animate_set()
 *
 * @ingroup Thumb
 */
EAPI Elm_Thumb_Animation_Setting elm_thumb_animate_get(const Evas_Object *obj);

/**
 * @brief Gets the ethumb_client handle so that custom configuration can be made.
 *
 * @remarks This must be called before the objects are created to be sure that no object is
 *          visible and no generation has started.
 *
 * Example of usage:
 *
 * @code
 * #include <Elementary.h>
 * #ifndef ELM_LIB_QUICKLAUNCH
 * EAPI_MAIN int
 * elm_main(int argc, char **argv)
 * {
 *    Ethumb_Client *client;
 *
 *    elm_need_ethumb();
 *
 *    // ... your code
 *
 *    client = elm_thumb_ethumb_client_get();
 *    if (!client)
 *      {
 *         ERR("could not get ethumb_client");
 *         return 1;
 *      }
 *    ethumb_client_size_set(client, 100, 100);
 *    ethumb_client_crop_align_set(client, 0.5, 0.5);
 *    // ... your code
 *
 *    // Create elm_thumb objects here
 *
 *    elm_run();
 *    elm_shutdown();
 *    return 0;
 * }
 * #endif
 * ELM_MAIN()
 * @endcode
 *
 * @remarks There's only one client handle for Ethumb, so once a configuration
 *          change is done to it, any other request for thumbnails (for any thumbnail
 *          object) uses that configuration. Thus, this configuration is global.
 *
 * @return An Ethumb_Client instance, otherwise @c NULL
 *
 * @ingroup Thumb
 */
EAPI void                       *elm_thumb_ethumb_client_get(void);

/**
 * @brief Gets the ethumb_client connection state.
 *
 * @return @c EINA_TRUE if the client is connected to the server, otherwise @c EINA_FALSE
 * 
 */
EAPI Eina_Bool                   elm_thumb_ethumb_client_connected_get(void);

/**
 * @brief Makes the thumbnail 'editable'.
 *
 * @remarks This means the thumbnail is a valid drag target for drag and drop, and can be
 *          cut or pasted.
 *
 * @param[in] obj The thumb object
 * @param[in] edit The boolean value that turns on or turns off editability \n
 *             Default is @c EINA_FALSE.
 *
 * @see elm_thumb_editable_get()
 *
 * @ingroup Thumb
 */
EAPI Eina_Bool                   elm_thumb_editable_set(Evas_Object *obj, Eina_Bool edit);

/**
 * @brief Makes the thumbnail 'editable'.
 *
 * @remarks This means the thumbnail is a valid drag target for drag and drop, and can be
 *          cut or pasted.
 *
 * @param[in] obj The thumb object
 * @return The boolean value that turns on or turns off editability
 *
 * @see elm_thumb_editable_set()
 *
 * @ingroup Thumb
 */
EAPI Eina_Bool                   elm_thumb_editable_get(const Evas_Object *obj);

/**
 * @}
 */
