#ifndef _EFL_FILE_EO_H_
#define _EFL_FILE_EO_H_

#ifndef _EFL_FILE_EO_CLASS_TYPE
#define _EFL_FILE_EO_CLASS_TYPE

typedef Eo Efl_File;

#endif

#ifndef _EFL_FILE_EO_TYPES
#define _EFL_FILE_EO_TYPES


#endif
#define EFL_FILE_INTERFACE efl_file_interface_get()

const Eo_Class *efl_file_interface_get(void) EINA_CONST;

/**
 *
 * Set the source mmaped file from where an image object must fetch the real
 * image data (it must be an Eina_File).
 *
 * If the file supports multiple data stored in it (as Eet files do),
 * you can specify the key to be used as the index of the image in
 * this file.
 *
 * @since 1.8
 *
 * @param[in] f The mmaped file
 * @param[in] key The image key in @p file (if its an Eet one), or @c
NULL, otherwise.
 *
 */
EOAPI Eina_Bool  efl_file_mmap_set(const Eina_File *f, const char *key);

/**
 *
 * Get the source mmaped file from where an image object must fetch the real
 * image data (it must be an Eina_File).
 *
 * If the file supports multiple data stored in it (as Eet files do),
 * you can get the key to be used as the index of the image in
 * this file.
 *
 * @since 1.10
 *
 * @param[out] f The mmaped file
 * @param[out] key The image key in @p file (if its an Eet one), or @c
NULL, otherwise.
 *
 */
EOAPI void  efl_file_mmap_get(const Eina_File **f, const char **key);

/**
 *
 * Set the source file from where an image object must fetch the real
 * image data (it may be an Eet file, besides pure image ones).
 *
 * If the file supports multiple data stored in it (as Eet files do),
 * you can specify the key to be used as the index of the image in
 * this file.
 *
 * Example:
 * @code
 * img = evas_object_image_add(canvas);
 * evas_object_image_file_set(img, "/path/to/img", NULL);
 * err = evas_object_image_load_error_get(img);
 * if (err != EVAS_LOAD_ERROR_NONE)
 * {
 * fprintf(stderr, "could not load image '%s'. error string is \"%s\"\n",
 * valid_path, evas_load_error_str(err));
 * }
 * else
 * {
 * evas_object_image_fill_set(img, 0, 0, w, h);
 * evas_object_resize(img, w, h);
 * evas_object_show(img);
 * }
 * @endcode
 *
 * @param[in] file The image file path.
 * @param[in] key The image key in @p file (if its an Eet one), or @c
NULL, otherwise.
 *
 */
EOAPI Eina_Bool  efl_file_set(const char *file, const char *key);

/**
 *
 * Retrieve the source file from where an image object is to fetch the
 * real image data (it may be an Eet file, besides pure image ones).
 *
 * You must @b not modify the strings on the returned pointers.
 *
 * @note Use @c NULL pointers on the file components you're not
 * interested in: they'll be ignored by the function.
 *
 * @param[out] file The image file path.
 * @param[out] key The image key in @p file (if its an Eet one), or @c
NULL, otherwise.
 *
 */
EOAPI void  efl_file_get(const char **file, const char **key);

/**
 *
 * If true, file open will happen asynchronously allowing for better
 * performance in some situations. The file will be opened from a
 * different thread. Classes implementing async open might then block
 * and wait when querying information from the file (eg. image size).
 *
 * Only a few objects implement this feature, and this flag may
 * be ignored by EFL. In that case, get() will always return false.
 *
 * @param[in] async Flag for asynchronous open.
 *
 */
EOAPI void  efl_file_async_set(Eina_Bool async);

/**
 *
 * Retrieves the asynchronous open flag, which will be true only if
 * enabled and supported by the object.
 *
 *
 */
EOAPI Eina_Bool  efl_file_async_get(void);

/**
 *
 * Save the given image object's contents to an (image) file.
 *
 * The extension suffix on @p file will determine which <b>saver
 * module</b> Evas is to use when saving, thus the final file's
 * format. If the file supports multiple data stored in it (Eet ones),
 * you can specify the key to be used as the index of the image in it.
 *
 * You can specify some flags when saving the image.  Currently
 * acceptable flags are @c quality and @c compress. Eg.: @c
 * "quality=100 compress=9"
 * 
 *
 * @param[in] file The filename to be used to save the image (extension
obligatory).
 * @param[in] key The image key in the file (if an Eet one), or @c NULL,
otherwise.
 * @param[in] flags String containing the flags to be used (@c NULL for
none).
 *
 */
EOAPI Eina_Bool  efl_file_save(const char *file, const char *key, const char *flags);

/**
 *
 * Eject the represented object.
 *
 * Get rid of and clean the pointed resource.
 * 
 *
 *
 */
EOAPI void  efl_file_eject(void);

/**
 *
 * Block and wait until all asynchronous operations are completed. Unless
 * the async flag was set on this object, this method has no effect.
 *
 * Returns false if an error occured.
 * 
 *
 *
 */
EOAPI Eina_Bool  efl_file_async_wait(void);

EOAPI extern const Eo_Event_Description _EFL_FILE_EVENT_ASYNC_OPENED;
EOAPI extern const Eo_Event_Description _EFL_FILE_EVENT_ASYNC_ERROR;

/**
 * The file was successfully opened asynchronously
 */
#define EFL_FILE_EVENT_ASYNC_OPENED (&(_EFL_FILE_EVENT_ASYNC_OPENED))

/**
 * Error occured in asynchronous file operation
 */
#define EFL_FILE_EVENT_ASYNC_ERROR (&(_EFL_FILE_EVENT_ASYNC_ERROR))

#endif
