/**
 * @defgroup General General
 * @ingroup elm_infra_group
 *
 * @brief General Elementary API. Functions that don't relate to
 * Elementary objects specifically.
 *
 * Here are documented functions which init/shutdown the library,
 * that apply to generic Elementary objects, that deal with
 * configuration, et cetera.
 *
 */

/**
 * @addtogroup General
 * @{
 */

/**
 * Defines couple of standard Evas_Object layers to be used
 * with evas_object_layer_set().
 *
 * @note whenever extending with new values, try to keep some padding
 *       to siblings so there is room for further extensions.
 */
typedef enum
{
   ELM_OBJECT_LAYER_BACKGROUND = EVAS_LAYER_MIN + 64, /**< where to place backgrounds */
   ELM_OBJECT_LAYER_DEFAULT = 0, /**< Evas_Object default layer (and thus for Elementary) */
   ELM_OBJECT_LAYER_FOCUS = EVAS_LAYER_MAX - 128, /**< where focus object visualization is */
   ELM_OBJECT_LAYER_TOOLTIP = EVAS_LAYER_MAX - 64, /**< where to show tooltips */
   ELM_OBJECT_LAYER_CURSOR = EVAS_LAYER_MAX - 32, /**< where to show cursors */
   ELM_OBJECT_LAYER_LAST /**< last layer known by Elementary */
} Elm_Object_Layer;

/**************************************************************************/
EAPI extern int ELM_ECORE_EVENT_ETHUMB_CONNECT;

/**
 * @internal
 *
 * Emitted when the application has reconfigured elementary settings due
 * to an external configuration tool asking it to.
 */
EAPI extern int ELM_EVENT_CONFIG_ALL_CHANGED;

/**
 * @internal
 *
 * Emitted when any Elementary's policy value is changed.
 */
EAPI extern int ELM_EVENT_POLICY_CHANGED;

/**
 * @internal
 *
 * Emitted when nothing is visible and the process as a whole should go into
 * a background state.
 * @since 1.12
 */
EAPI extern int ELM_EVENT_PROCESS_BACKGROUND;

/**
 * @internal
 *
 * Emitted when going from nothing being visible to at least one window
 * being visible.
 * @since 1.12
 */
EAPI extern int ELM_EVENT_PROCESS_FOREGROUND;

/**
 * @typedef Elm_Event_Policy_Changed
 *
 * Data on the event when an Elementary policy has changed
 */
typedef struct _Elm_Event_Policy_Changed Elm_Event_Policy_Changed;

/**
 * @struct _Elm_Event_Policy_Changed
 *
 * Data on the event when an Elementary policy has changed
 */
struct _Elm_Event_Policy_Changed
{
   unsigned int policy; /**< the policy identifier */
   int          new_value; /**< value the policy had before the change */
   int          old_value; /**< new value the policy got */
};

/**
 * Policy identifiers.
 */
typedef enum
{
   ELM_POLICY_QUIT, /**< under which circumstances the application
                     * should quit automatically. @see
                     * Elm_Policy_Quit.
                     */
   ELM_POLICY_EXIT, /**< defines elm_exit() behaviour. @see Elm_Policy_Exit.
                     * @since 1.8
                     */
   ELM_POLICY_THROTTLE, /**< defines how throttling should work @see Elm_Policy_Throttle
                         * @since 1.8
                         */
   ELM_POLICY_LAST
} Elm_Policy; /**< Elementary policy identifiers/groups enumeration.  @see elm_policy_set() */

/**
 * Possible values for the #ELM_POLICY_QUIT policy
 */
typedef enum
{
   ELM_POLICY_QUIT_NONE = 0, /**< never quit the application
                              * automatically */
   ELM_POLICY_QUIT_LAST_WINDOW_CLOSED /**< quit when the
                                       * application's last
                                       * window is closed */
} Elm_Policy_Quit;

/**
 * Possible values for the #ELM_POLICY_EXIT policy.
 * @since 1.8
 */
typedef enum
{
   ELM_POLICY_EXIT_NONE = 0, /**< just quit the main loop on elm_exit() */
   ELM_POLICY_EXIT_WINDOWS_DEL /**< delete all the windows after quitting
                                * the main loop */
} Elm_Policy_Exit;

/**
 * Possible values for the #ELM_POLICY_THROTTLE policy.
 * @since 1.8
 */
typedef enum
{
   ELM_POLICY_THROTTLE_CONFIG = 0, /**< do whatever elementary config is configured to do */
   ELM_POLICY_THROTTLE_HIDDEN_ALWAYS, /**< always throttle when all windows are no longer visible */
   ELM_POLICY_THROTTLE_NEVER /**< never throttle when windows are all hidden, regardless of config settings */
} Elm_Policy_Throttle;

/**
 * Possible values for the #ELM_OBJECT_SELECT_MODE policy.
 * @since 1.7
 */
typedef enum
{
   ELM_OBJECT_SELECT_MODE_DEFAULT = 0, /**< default select mode. Once an item is selected, it would stay highlighted and not going to call selected callback again even it was clicked. Items can get focus. */
   ELM_OBJECT_SELECT_MODE_ALWAYS, /**< always select mode. Item selected callbacks will be called every time for click events, even after the item was already selected. Items can get focus. */
   ELM_OBJECT_SELECT_MODE_NONE, /**< no select mode. Items will never be highlighted and selected but the size will be adjusted by the finger size configuration. Items can't get focus. */
   ELM_OBJECT_SELECT_MODE_DISPLAY_ONLY, /**< no select mode with no finger size rule. Items will never be highlighted and selected and ignore the finger size. So the item size can be reduced below than the finger size configuration. Items can't get focus. */
   ELM_OBJECT_SELECT_MODE_MAX /**< canary value: any value greater or equal to ELM_OBJECT_SELECT_MODE_MAX is forbidden. */
} Elm_Object_Select_Mode;

/**
 * Possible values for the #ELM_OBJECT_MULTI_SELECT_MODE policy.
 * @since 1.8
 */
typedef enum
{
   ELM_OBJECT_MULTI_SELECT_MODE_DEFAULT = 0, /**< default multiple select mode */
   ELM_OBJECT_MULTI_SELECT_MODE_WITH_CONTROL, /**< disallow mutiple selection when clicked without control key pressed */
   ELM_OBJECT_MULTI_SELECT_MODE_MAX /**< canary value: any value greater or equal to ELM_OBJECT_MULTI_SELECT_MODE_MAX is forbidden. */
} Elm_Object_Multi_Select_Mode;

typedef Eina_Bool             (*Elm_Event_Cb)(void *data, Evas_Object *obj, Evas_Object *src, Evas_Callback_Type type, void *event_info); /**< Function prototype definition for callbacks on input events happening on Elementary widgets. @a data will receive the user data pointer passed to elm_object_event_callback_add(). @a src will be a pointer to the widget on which the input event took place. @a type will get the type of this event and @a event_info, the struct with details on this event. */

extern EAPI double _elm_startup_time;

#ifndef ELM_LIB_QUICKLAUNCH
#define ELM_MAIN() int main(int argc, char **argv) { int ret; _elm_startup_time = ecore_time_unix_get(); elm_init(argc, argv); ret = elm_main(argc, argv); elm_shutdown(); return ret; } /**< macro to be used after the elm_main() function */
#else
/** @deprecated macro to be used after the elm_main() function.
 * Do not define ELM_LIB_QUICKLAUNCH
 * Compile your programs with -fpie and -pie -rdynamic instead, to generate a single binary (linkable executable).
 */
#define ELM_MAIN() int main(int argc, char **argv) { int ret; _elm_startup_time = ecore_time_unix_get(); ret = elm_quicklaunch_fallback(argc, argv); elm_shutdown(); return ret; }
#endif

/**************************************************************************/
/* General calls */

/**
 * @brief Initialize Elementary
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @param[in] argc System's argument count value
 * @param[in] argv System's pointer to array of argument strings
 * @return The init counter value.
 *
 * @remark This function initializes Elementary and increments a counter of
 * the number of calls to it. It returns the new counter's value.
 *
 * @warning This call is exported only for use by the @c ELM_MAIN()
 * macro. There is no need to use this if you use this macro (which
 * is highly advisable). An elm_main() should contain the entry
 * point code for your application, having the same prototype as
 * elm_init(), and @b not being static (putting the @c EAPI_MAIN symbol
 * in front of its type declaration is advisable). The @c
 * ELM_MAIN() call should be placed just after it.
 *
 * @see elm_shutdown().
 * @ingroup General
 */
EAPI int       elm_init(int argc, char **argv);

/**
 * @brief Shut down Elementary
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @return The init counter value.
 *
 * @remark This should be called at the end of your application, just
 * before it ceases to do any more processing. This will clean up
 * any permanent resources your application may have allocated via
 * Elementary that would otherwise persist.
 *
 * @see elm_init() for an example
 *
 * @remark elm_shutdown() will iterate main loop until all ecore_evas are freed.
 * There is a possibility to call your ecore callbacks(timer, animator, event,
 * job, and etc.) in elm_shutdown()
 *
 * @ingroup General
 */
EAPI int       elm_shutdown(void);

/**
 * @brief Run Elementary's main loop
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @remark This call should be issued just after all initialization is
 * completed. This function will not return until elm_exit() is
 * called. It will keep looping, running the main
 * (event/processing) loop for Elementary.
 *
 * @see elm_init() for an example
 *
 * @ingroup General
 */
EAPI void      elm_run(void);

/**
 * @brief Ask to exit Elementary's main loop
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @remark If this call is issued, it will flag the main loop to cease
 * processing and return back to its parent function (usually your
 * elm_main() function). This does not mean the main loop instantly quits.
 * So your ecore callbacks(timer, animator, event, job, and etc.) have chances
 * to be called even after elm_exit().
 *
 * @see elm_init() for an example. There, just after a request to
 * close the window comes, the main loop will be left.
 *
 * @remark By using the appropriate #ELM_POLICY_QUIT on your Elementary
 * applications, you'll be able to get this function called automatically for you.
 *
 * @ingroup General
 */
EAPI void      elm_exit(void);

/**
 * @brief Exposed symbol used only by macros and should not be used by apps
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @param[in] ql_on The quicklaunch mode to be set.
 *
 * @deprecated Deprecated since Tizen 2.4 application should not use this function
 */
EAPI void      elm_quicklaunch_mode_set(Eina_Bool ql_on);

/**
 * @brief Exposed symbol used only by macros and should not be used by apps
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @deprecated Deprecated since Tizen 2.4 application should not use this function
 */
EAPI Eina_Bool elm_quicklaunch_mode_get(void);

/**
 * @brief Exposed symbol used only by macros and should not be used by apps
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @param[in] argc The system's argument count value.
 * @param[in] argv The system's pointer to the array of argument strings.
 *
 * @deprecated Deprecated since Tizen 2.4 application should not use this function
 */
EAPI int       elm_quicklaunch_init(int argc, char **argv);

/**
 * @brief Exposed symbol used only by macros and should not be used by apps
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @param[in] argc The system's argument count value.
 * @param[in] argv The system's pointer to the array of argument strings.
 *
 * @deprecated Deprecated since Tizen 2.4 application should not use this function
 */
EAPI int       elm_quicklaunch_sub_init(int argc, char **argv);

/**
 * @brief Exposed symbol used only by macros and should not be used by apps
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @deprecated Deprecated since Tizen 2.4 application should not use this function
 */
EAPI int       elm_quicklaunch_sub_shutdown(void);

/**
 * @brief Exposed symbol used only by macros and should not be used by apps
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @deprecated Deprecated since Tizen 2.4 application should not use this function
 */
EAPI int       elm_quicklaunch_shutdown(void);

/**
 * @brief Exposed symbol used only by macros and should not be used by apps
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @deprecated Deprecated since Tizen 2.4 application should not use this function
 */
EAPI void      elm_quicklaunch_seed(void);

/**
 * @brief Exposed symbol used only by macros and should not be used by apps
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @param[in] argc The system's argument count value.
 * @param[in] argv The system's pointer to the array of argument strings.
 * @param[in] cwd The current working directory.
 *
 * @deprecated Deprecated since Tizen 2.4 application should not use this function
 */
EAPI Eina_Bool elm_quicklaunch_prepare(int argc, char **argv, const char *cwd);

/**
 * @brief Exposed symbol used only by macros and should not be used by apps
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @param[in] argc The system's argument count value.
 * @param[in] argv The system's pointer to the array of argument strings.
 * @param[in] cwd The current working directory.
 * @param[in] postfork_func The function will be called after fork.
 * @param[in] postfork_data The used data for postfork_func.
 *
 * @deprecated Deprecated since Tizen 2.4 application should not use this function
 */
EAPI Eina_Bool elm_quicklaunch_fork(int argc, char **argv, char *cwd, void (*postfork_func) (void *data), void *postfork_data);

/**
 * @brief Exposed symbol used only by macros and should not be used by apps
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @deprecated Deprecated since Tizen 2.4 application should not use this function
 */
EAPI void      elm_quicklaunch_cleanup(void);

/**
 * @brief Exposed symbol used only by macros and should not be used by apps
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @param[in] argc The system's argument count value.
 * @param[in] argv The system's pointer to the array of argument strings.
 *
 * @deprecated Deprecated since Tizen 2.4 application should not use this function
 */
EAPI int       elm_quicklaunch_fallback(int argc, char **argv);

/**
 * @brief Exposed symbol used only by macros and should not be used by apps
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @param[in] exe The executable path.
 * @param[in] cwd The current working directory.
 *
 * @deprecated Deprecated since Tizen 2.4 application should not use this function
 */
EAPI char     *elm_quicklaunch_exe_path_get(const char *exe, const char *cwd);

/**
 * @brief Set a new policy's value (for a given policy group/identifier).
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @param[in] policy policy identifier, as in @ref Elm_Policy.
 * @param[in] value policy value, which depends on the identifier
 *
 * @return @c EINA_TRUE on success or @c EINA_FALSE, on error.
 *
 * @remark Elementary policies define applications' behavior,
 * somehow. These behaviors are divided in policy groups
 * (see #Elm_Policy enumeration). This call will emit the Ecore
 * event #ELM_EVENT_POLICY_CHANGED, which can be hooked at with
 * handlers. An #Elm_Event_Policy_Changed struct will be passed,
 * then.
 *
 * @remark Currently, we have only one policy identifier/group
 * (#ELM_POLICY_QUIT), which has two possible values.
 *
 * @ingroup General
 */
EAPI Eina_Bool elm_policy_set(unsigned int policy, int value);

/**
 * @brief Get the policy value for given policy identifier.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @param[in] policy policy identifier, as in #Elm_Policy.
 * @return The currently set policy value, for that
 * identifier. Will be @c 0 if @p policy passed is invalid.
 *
 * @ingroup General
 */
EAPI int       elm_policy_get(unsigned int policy);

/**
 * @brief Change the language of the current application
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @remark The @p lang passed must be the full name of the locale to use, for
 * example "en_US.utf8" or "es_ES@euro".
 *
 * @remark Changing language with this function will make Elementary run through
 * all its widgets, translating strings set with
 * elm_object_domain_translatable_part_text_set(). This way, an entire
 * UI can have its language changed without having to restart the program.
 *
 * @remark For more complex cases, like having formatted strings that need
 * translation, widgets will also emit a "language,changed" signal that
 * the user can listen to and manually translate the text.
 *
 * @param[in] lang Language to set, must be the full name of the locale
 *
 * @ingroup General
 */
EAPI void      elm_language_set(const char *lang);

typedef enum _Elm_Process_State
{
   ELM_PROCESS_STATE_FOREGROUND, /*< The process is in a foreground/active/running state - work as normal. @since 1.12 */
   ELM_PROCESS_STATE_BACKGROUND /*< The process is in the bacgkround, so you may want to stop animating, fetching data as often etc. @since 1.12 */
} Elm_Process_State; /** @internal The state of the process as a whole. @since 1.12 */

/**
 * @internal
 *
 * Get the process state as a while
 *
 * @return The current process state
 *
 * The process may logically be some runnable state. a "foreground" application
 * runs as normal and may be user-visible or "active" in some way. A
 * background application is not user-visible or otherwise important and
 * likely should release resources and not wake up often or process much.
 *
 * @ingroup General
 * @since 1.12
 */
EAPI Elm_Process_State  elm_process_state_get(void);

/**
 * @}
 */
