/*
 * 
 *
 * Copyright (c) 2000 - 2011 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Contact: 
 * BonYong Lee <bonyong.lee@samsung.com>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Contributors:
 * - S-Core Co., Ltd
 *
 */
package org.tizen.common;

import static org.tizen.common.util.IOUtil.getBytes;
import static org.tizen.common.util.IOUtil.tryClose;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.lang.instrument.ClassFileTransformer;
import java.lang.instrument.IllegalClassFormatException;
import java.util.ArrayList;
import java.util.Arrays;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.tizen.common.classloader.ClassSource;
import org.tizen.common.util.IOUtil;


/**
 * <p>
 * UnsafeClassLoader.
 * 
 * Classloader for untestable class
 * 
 * FIXME add {@link Package} info and {@link #getResource(String)}, {@link #getResourceAsStream(String)}, {@link #getResources(String)}
 * </p>
 * 
 * @author BonYong Lee{@literal <bonyong.lee@samsung.com>} (S-Core)
 */
public class
UnsafeClassLoader
extends ClassLoader
{
	/**
	 * Logger for this instance
	 */
	protected final Logger logger = LoggerFactory.getLogger( getClass() );
	
	/**
	 * Parent classloader
	 */
	protected final ClassLoader parent;
	
	protected ClassFileTransformer transformer = new Transformer();

	/**
	 * {@link ClassSource}s providing class contents
	 */
	protected final ArrayList<ClassSource> sources = new ArrayList<ClassSource>();

	/**
	 * Constructor with {@link ClassSource}s
	 * @param source
	 */
	public
	UnsafeClassLoader(
		final ClassSource... source
	)
	{
		this( null, source );
	}

	/**
	 * Constructor with parent class loader and class source
	 * 
	 * @param parent {@link ClassLoader}
	 * @param sources {@link ClassSource}s
	 */
	public
	UnsafeClassLoader(
		final ClassLoader parent,
		final ClassSource... sources
	)
	{
		this.parent = parent;
		this.sources.addAll( Arrays.asList( sources ) );
	}

	/* (non-Javadoc)
	 * @see java.lang.ClassLoader#loadClass(java.lang.String, boolean)
	 */
	@Override
	synchronized protected
	Class<?>
	loadClass(
		final String name,
		final boolean resolve
	)
	throws ClassNotFoundException
	{
		// First, check if the class has already been loaded
		Class<?> c = findLoadedClass( name );
		if ( c == null )
		{
			if ( name.startsWith( "java." ) )
			{
				return ClassLoader.getSystemClassLoader().loadClass( name );
			}
			try
			{
				c = findClass( name );
			} catch (ClassNotFoundException e) {
				if ( null != parent )
				{
					c = parent.loadClass( name );
				}
				else
				{
					throw new ClassNotFoundException();
				}
			}
		}
		if ( resolve )
		{
			resolveClass( c );
		}
		return c;
	}

	/* (non-Javadoc)
	 * @see java.lang.ClassLoader#findClass(java.lang.String)
	 */
	@Override
	protected
	Class<?>
	findClass(
		final String name
	)
	throws ClassNotFoundException
	{
		logger.trace( "Class name :{}", name );
		final String path = name.replace( '.', '/' ) + ".class";
		for ( final ClassSource source : sources )
		{
			logger.trace( "Source :{}", source );
			InputStream in = null;;
			try
			{
				in = source.getResourceAsStream( path );
				final Class<?> clazz = define( name, in );
				if ( null != clazz )
				{
					return clazz;
				}
			}
			catch ( final IOException e )
			{
			}
			finally
			{
				tryClose( in );
			}
		}
		
//		final ClassLoader loader = getClass().getClassLoader();
//		
//		final InputStream in = loader.getResourceAsStream( path );
//		try
//		{
//			final Class<?> clazz = define( name, in );
//			if ( null != clazz )
//			{
//				return clazz;
//			}
//		}
//		finally
//		{
//			tryClose( in );
//		}
//
		throw new ClassNotFoundException();

	}
	
	protected Class<?> define(
		final String name,
		final InputStream in
	)
	{
		try {
			if ( null == in )
			{
				return null;
			}
			byte[] bytes = getBytes( in );
			final String qn = name.replace( '.', '/' );
			bytes = transformer.transform( this, qn, null, null, bytes );
			logger.trace( "{} bytes read", bytes.length );
			return defineClass( name, bytes, 0, bytes.length );
		} catch ( final IOException e )
		{
		} catch ( final IllegalClassFormatException e )
		{
		}
		
		return null;
	}

	/**
	 * Write log file about converted class
	 * 
	 * @param path file path to write
	 * @param bytes class contents
	 * 
	 * @throws IOException If file can't be write
	 */
	protected
	void
	writeLog(
		final String path,
		byte[] bytes
	)
	throws IOException
	{
		FileOutputStream fileOut = null;
		try
		{
			File logFile = new File( "test/log/" + path );
            File parent = logFile.getParentFile();
            if (parent != null) {
                parent.mkdirs();
                fileOut = new FileOutputStream(logFile);
                IOUtil.redirect(new ByteArrayInputStream(bytes), fileOut);
            }
		}
		finally
		{
			tryClose( fileOut);
		}
	}

}
