/*
*  Common
*
* Copyright (c) 2000 - 2015 Samsung Electronics Co., Ltd. All rights reserved.
*
* Contact: 
* Jihoon Song <jihoon80.song@samsung.com>
* 
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
* http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
* Contributors:
* - S-Core Co., Ltd
*
*/
package org.tizen.common.core.application;

import static org.junit.Assert.*;

import org.junit.Test;

/**
 * Test case for {@link Version}
 * 
 * @author Jihoon Song {@literal<jihoon80.song@samsung.com>}
 * @see Version
 */
public class VersionTest {

    /**
     * Test method for {@link org.tizen.common.core.application.Version#Version(java.lang.String)}.
     */
    @Test
    public void testVersion() {
        final Object[][] TEST_CASES = {
                // argument, exception type (if null, success)
                { null, IllegalArgumentException.class },
                { "0", IllegalArgumentException.class },
                { "1#.to", IllegalArgumentException.class },
                { "0.0", null },
                { "1.0", null },
                { "1.0.1", null },
                { "-1.0.0", IllegalArgumentException.class },
                { "a.b", IllegalArgumentException.class },
                { "1.0.", IllegalArgumentException.class }
        };
        
        for ( Object[] TEST_CASE : TEST_CASES ) {
            String argument = (String) TEST_CASE[0];
            Object exception = TEST_CASE[1];
            
            try {
                new Version( argument );
                if ( exception != null ) {
                    fail( "Exception is not occurred (" + argument + ")" );
                }
            } catch(IllegalArgumentException e) {
                assertTrue( "Exception is not matched (" + argument + ")",
                        e.getClass().equals( exception ) );
            }
        }
    }

    /**
     * Test method for {@link org.tizen.common.core.application.Version#get()}.
     */
    @Test
    public void testGet() {
        final String[] TEST_CASES = {
                // argument
                "0.0",
                "1.0",
                "1.0.1",
                "10.40.38"
        };
        
        for ( String TEST_CASE : TEST_CASES ) {
            Version ver = new Version( TEST_CASE );
            assertEquals( TEST_CASE,
                    ver.get(), TEST_CASE );
        }
    }

    /**
     * Test method for {@link org.tizen.common.core.application.Version#compareTo(org.tizen.common.core.application.Version)}.
     */
    @Test
    public void testCompareTo() {
        final Object[][] TEST_CASES = {
                // source, compare, result (int)
                { "0.0", "0.0", 0 },
                { "1.0", "0.5", 1 },
                { "0.1", "0.10", -1},
                { "1.0.1", "1.0", 1 },
        };
        
        for ( Object[] TEST_CASE : TEST_CASES ) {
            String argument1 = (String) TEST_CASE[0];
            String argument2 = (String) TEST_CASE[1];
            int expected = (Integer) TEST_CASE[2];
            
            Version ver1 = new Version( argument1 );
            Version ver2 = new Version( argument2 );
            assertEquals( argument1 + ", " + argument2,
                    ver1.compareTo( ver2 ), expected );
        }
    }

    /**
     * Test method for {@link org.tizen.common.core.application.Version#equals(java.lang.Object)}.
     */
    @Test
    public void testEqualsObject() {
        final String TEST_VERSION_NUMBER = "1.0";
        final Version TEST_VERSION = new Version( TEST_VERSION_NUMBER );
        
        final Object[][] TEST_CASES = {
                // source, compare, result (boolean)
                { TEST_VERSION, TEST_VERSION, true }, // same instance
                { TEST_VERSION, new Version( TEST_VERSION_NUMBER ), true }, // same attribute
                { TEST_VERSION, new Version( "0.1" ), false }, // not same attribute
                { TEST_VERSION, TEST_VERSION_NUMBER, false }, // not same class
                { TEST_VERSION, null, false } // null
        };
        
        for ( Object[] TEST_CASE : TEST_CASES ) {
            Version ver1 = (Version) TEST_CASE[0];
            Object ver2 = TEST_CASE[1];
            boolean expected = (Boolean) TEST_CASE[2];
            
            assertEquals( ver1.equals( ver2 ), expected );
        }
    }

}
