/*
 * Common
 *
 * Copyright (c) 2000 - 2011 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Contact:
 * Ho Namkoong <ho.namkoong@samsung.com> 
 * BonYong Lee <bonyong.lee@samsung.com>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Contributors:
 * - S-Core Co., Ltd
 *
 */

package org.tizen.common.sdb;

import static org.junit.Assert.*;
import static org.tizen.common.Shell.*;
import static org.tizen.common.sdb.SdbTestUtil.*;

import java.io.File;
import java.text.MessageFormat;
import java.util.List;

import org.junit.Test;
import org.tizen.common.util.FileUtil;
import org.tizen.common.util.FileUtilTest;

/**
 * <p>
 * PullPushTest
 * 
 * Test case for sdb pull, push
 *
 * </p>
 * 
 * @author Ho Namkoong{@literal <ho.namkoong@samsung.com>} (S-Core)
 */
public class PullPushTest {

    /**
     * Test {sdb push, sdb pull (file)}
     * 
     * @author Ho Namkoong{@literal <ho.namkoong@samsung.com>} (S-Core)
     * 
     * command
     * 1. create random text file 'push_file'
     * 2. sdb shell rm -rf /tmp/push_file
     * 3. sdb push push_file /tmp/push_file
     * 4. sdb pull /tmp/push_file pull_file
     * expected result
     * contents of push_file and pull_file are same
     */
    @Test
    public void test_pushPullFile() throws Exception {
        
        final String pushFileName = "push_file";
        final String pullFileName = "pull_file";
        final String tmpDir = "/tmp";
        final String targetDst = tmpDir + "/" + pushFileName;
        
        try {
            pushPullAndCheck(pushFileName, targetDst, pullFileName, true);
        }
        finally {
            File pushFile = new File(pushFileName);
            if(pushFile.exists()) {
                FileUtil.recursiveDelete(pushFile);
            }
            File pullFile = new File(pullFileName);
            if(pullFile.exists()) {
                FileUtil.recursiveDelete(pullFile);
            }
        }
    }
    
    /**
     * Test {sdb push, sdb pull (directory)}
     * 
     * @author Ho Namkoong{@literal <ho.namkoong@samsung.com>} (S-Core)
     * 
     * command
     * 1. create random directory 'push_dir'
     * 2. sdb shell rm -rf /tmp/push_dir
     * 3. sdb push push_dir /tmp/push_dir
     * 4. sdb pull /tmp/push_dir pull_dir
     * expected result
     * contents of push_dir and pull_dir are same
     */
    @Test
    public void test_pushPullDir() throws Exception {
        
        final String pushDirName = "push_dir";
        final String pullDirName = "pull_dir";
        final String tmpDir = "/tmp";
        final String targetDst = tmpDir + "/" + pushDirName;
        try {
            pushPullAndCheck(pushDirName, targetDst, pullDirName, false);
        }
        finally {
            File pushDir = new File(pushDirName);
            if(pushDir.exists()) {
                FileUtil.recursiveDelete(pushDir);
            }
            File pullDir = new File(pullDirName);
            if(pullDir.exists()) {
                FileUtil.recursiveDelete(pullDir);
            }
        }
    }
    
    private void pushPullAndCheck(String pushSrc, String pushDst, String pullDst, boolean file) throws Exception {
        
        String[] results = null;
        
        File pushFile = new File(pushSrc);
        if(pushFile.exists()) {
            FileUtil.recursiveDelete(pushFile);
        }
        
        File pullFile = new File(pullDst);
        if(pullFile.exists()) {
            FileUtil.recursiveDelete(pullFile);
        }
        
        String fileContent = "";
        if(file) {
            fileContent = generateRandomString(400, FILE_CONTENT_CHAR_LIST);
            FileUtil.createTextFile(pushFile, fileContent, null);
        }
        else {
            generateRandomDir(pushFile, 5, 5, 3);
        }
        
        assertTrue(MessageFormat.format("fail to create file {0}", pushFile.getAbsolutePath()), pushFile.exists());
        String serialNumber = getSerialNumber()[0];
        runAndWait( SDB_PATH + String.format(" -s %s shell rm -rf %s", serialNumber, pushDst));
        assertTrue(MessageFormat.format("rm fail to remove {0}", pushDst), 
                checkTargetFileExist(pushDst, serialNumber, file, false));
        results = runAndGetWholeString( SDB_PATH + String.format(" -s %s push %s %s", serialNumber, pushFile.getAbsolutePath(), pushDst), false);
        if(results != null) {
            for(String result: results) {
                assertFalse(SDB_PATH + String.format(" -s %s push %s %s", serialNumber, pushFile.getAbsolutePath(), pushDst) 
                        + "makes error message: " + result, result.contains("error"));
            }
        }
        
        assertTrue(MessageFormat.format("{0} does not exist in target {1}", pushDst, serialNumber), 
                checkTargetFileExist(pushDst, serialNumber, file, true));
        results = runAndGetWholeString( SDB_PATH + String.format(" -s %s pull %s %s", serialNumber, pushDst, pullFile.getAbsolutePath()), false);
        
        if(results != null) {
            for(String result: results) {
                assertFalse( SDB_PATH + String.format(" -s %s pull %s %s", serialNumber, pushDst, pullFile.getAbsolutePath()) 
                        + "makes error message: " + result, result.contains("error"));
            }
        }
        
        assertTrue(MessageFormat.format("{0} does not exist in host", pullFile.getAbsolutePath()), pullFile.exists());
        
        if( file ) {
            String result = FileUtil.readTextFile(pullFile, null);
            assertEquals(MessageFormat.format("pushed file {0} and pull file {1} are not same", pushFile.getAbsolutePath(), pullFile.getAbsolutePath()), 
                    result, fileContent);
        }
        else {
            List<File> notPushPullList = FileUtilTest.compareFiles(pushFile.getAbsolutePath(), pullFile.getAbsolutePath());
            assertTrue(String.format("Following files are not pushed or pulled\n%s", 
                    listFiles(notPushPullList)), notPushPullList.size() == 0);
            List<File> morePushPullList = FileUtilTest.compareFiles(pullFile.getAbsolutePath(), pushFile.getAbsolutePath());
            assertTrue(String.format("Following files are pushed or pulled but not in source\n%s", listFiles(morePushPullList)), 
                    FileUtilTest.compareFiles(pullFile.getAbsolutePath(), pushFile.getAbsolutePath()).size() == 0);
        }
    }
    
    private String listFiles(List<File> fileList) {
        StringBuffer buffer = new StringBuffer();
        for(File file: fileList) {
            buffer.append(file.getAbsolutePath() + "\n");
        }
        return buffer.toString();
    }
    
    private boolean checkTargetFileExist(String fileName, String serial, boolean file, boolean exist) throws Exception {
        String[] results = null;
        for( int i =0; i < MAX_TRIAL; i++) {
            if(file) {
                results = runAndGetWholeString(SDB_PATH + String.format(" -s %s shell ls %s", serial, fileName), true);
            }
            else {
                results = runAndGetWholeString(SDB_PATH + String.format(" -s %s shell ls -d %s", serial, fileName), true);
            }
            for(String result: results) {
                if(exist) {
                    if(result.equals(fileName)) {
                        return true;
                    }
                }
                else {
                    if(result.contains("No such file or directory")) {
                        return true;
                    }
                }
            }
            Thread.sleep(1000);
        }
        return false;
    }
}
