/*
 * Common
 *
 * Copyright (c) 2000 - 2011 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Contact: 
 * BonYong Lee <bonyong.lee@samsung.com>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Contributors:
 * - S-Core Co., Ltd
 *
 */
package org.tizen.common.util;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedMap;
import java.util.SortedSet;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.Vector;
import java.util.concurrent.atomic.AtomicInteger;

import org.junit.Test;
import org.slf4j.LoggerFactory;


/**
 * CollectionUtilsTest
 *
 * Test case for {@link CollectionUtil}
 * 
 * @author BonYong Lee{@literal <bonyong.lee@samsung.com>} (S-Core)
 * @author GyeongSeok Seo{@literal <gyeongseok.seo@samsung.com>} (S-Core)
 * 
 * @see CollectionUtil
 */
public class
CollectionUtilTest
{
    /**
     * Test {@link CollectionUtil#isEmpty(java.util.Collection)}
     * 
     * @throws Exception in case of failure in test
     * 
     * @see CollectionUtil#isEmpty(java.util.Collection)
     */
    @Test
    @SuppressWarnings("serial")
    public void test_isEmpty() throws Exception
    {
        assertTrue( CollectionUtil.isEmpty( null ) );
        assertTrue( CollectionUtil.isEmpty( new HashSet<Object>() ) );
        assertTrue( CollectionUtil.isEmpty( new HashMap<String, Object>().keySet() ) );
        assertFalse( CollectionUtil.isEmpty( new ArrayList<Object>() {{ add("Hello"); }} ) );
        assertFalse( CollectionUtil.isEmpty( new ArrayList<Object>( Arrays.asList( "Hello" ) ) ) );
    }

    /**
     *  Test {@link CollectionUtil#size(java.util.Collection)}
     * 
     * @throws Exception in case of failure in test
     * 
     * @see CollectionUtil#size(java.util.Collection)
     */
    @SuppressWarnings("unchecked")
	@Test
    public void test_size() throws Exception
    {
        assertEquals( 0, CollectionUtil.size( null ) );

        /* Mock Object using Mockito */
        final ArrayList<Object> list = mock( ArrayList.class );

        when( list.size() ).thenReturn( 3 );
        assertEquals( 3, CollectionUtil.size(list) );
        verify( list ).size();

        final HashMap<String, Object> map = new HashMap<String, Object>();
        assertEquals( 0, CollectionUtil.size( map.keySet() ) );

        map.put("Hello", 1);
        map.put("Tizen", 2);
        assertEquals( 2, CollectionUtil.size( map.keySet() ) );
    }

    /**
     * Test {@link CollectionUtil#iterate(java.util.Collection, IteratingRunner)}
     * and {@link CollectionUtil#iterate(java.util.Collection, IteratingRunner, boolean)}
     * 
     * @throws Exception in case of failure in test
     * 
     * @see {@link CollectionUtil#iterate(java.util.Collection, IteratingRunner)}
     * @see {@link CollectionUtil#iterate(java.util.Collection, IteratingRunner, boolean)}
     */
    @Test
    public void test_iterate() throws Exception
    {
        final AtomicInteger count = new AtomicInteger();

        CollectionUtil.iterate(
                new ArrayList<String>( Arrays.asList(
                        "hello", "Hello", "World", null, "Test"
                ) ),
                new IteratingRunner<String>()
                {
                    public void run(String arg)
                    {
                        if ( Character.isUpperCase( arg.charAt( 0 ) ) )
                        {
                            count.incrementAndGet();
                        }
                    }
                }
        );

        assertEquals( 3, count.get() );

        count.set( 0 );
        CollectionUtil.iterate(
                new ArrayList<String>( Arrays.asList(
                        "hello", "Hello", "World", null, "Test"
                ) ),
                new IteratingRunner<String>()
                {
                    public void run( final String arg)
                    {
                        throw new IllegalStateException();
                    }
                },
                true
        );
        assertEquals( 0, count.get() );
    }

    /**
     * Test {@link CollectionUtil#filter(java.util.Collection, java.util.Collection, IteratingAcceptor, boolean)}
     * 
     * @throws Exception in case of failure in test
     * 
     * @see {@link CollectionUtil#filter(java.util.Collection, java.util.Collection, IteratingAcceptor, boolean)}
     */
    @Test
    public void test_filter() throws Exception
    {
        final ArrayList<String> results = new ArrayList<String>();

        CollectionUtil.filter(
                null,
                results,
                new IteratingAcceptor<String>()
                {
                    @Override
                    public boolean accept(String arg) {
                        return true;
                    }
                },
                true
        );
        assertEquals( 0, CollectionUtil.size(results) );

        results.clear();
        CollectionUtil.filter(
                new ArrayList<String>( Arrays.asList(
                        "hello", "Hello", "World", null, "Test"
                ) ),
                results,
                new IteratingAcceptor<String>()
                {
                    @Override
                    public boolean accept(String arg) {
                        if ( arg != null && Character.isUpperCase( arg.charAt( 0 ) ) )
                        {
                            return true;
                        }
                        return false;
                    }
                },
                true
        );
        assertEquals( 3, CollectionUtil.size(results) );

        results.clear();
        CollectionUtil.filter(
                new ArrayList<String>( Arrays.asList(
                        "hello", "Hello", "World", null, "Test"
                ) ),
                results,
                new IteratingAcceptor<String>()
                {
                    @Override
                    public boolean accept(String arg) {
                        if ( arg != null && Character.isUpperCase( arg.charAt( 0 ) ) )
                        {
                            return true;
                        }
                        throw new IllegalStateException();
                    }
                },
                true
        );
        assertEquals( 3, CollectionUtil.size(results) );
    }

    /**
     * Test {@link CollectionUtil#pickupFirst(Collection)}
     * 
     * @throws Exception in case of failure in test
     * 
     * @see {@link CollectionUtil#pickupFirst(Collection)}
     */
    @Test
    public void
    test_pickupFirst()
    throws Exception
    {
        final Object[][] TEST_CASES = new Object[][] {
                new Object[] { null, null },
                new Object[] { new Object[] { null }, null },
                new Object[] { new Object[] { "hello", null }, "hello" },
                new Object[] { new Object[] { null, "hello" }, null },
                new Object[] { new Object[] { "A", "B" }, "A" },
        };

        int iTestCase = 0;
        for ( final Object[] TEST_CASE : TEST_CASES )
        {
			++iTestCase;
            final Object[] closeables = (Object[]) TEST_CASE[0];
            final Object expected = (Object) TEST_CASE[1];

            Collection<Object> collection = null;
            if ( null != closeables )
            {
                collection = Arrays.asList( closeables );
            }

            assertEquals( iTestCase + " th test case", expected, CollectionUtil.pickupFirst( collection ) );
        }
    }

    /**
     * Test {@link CollectionUtil#pickupLast(Collection)}
     * 
     * @throws Exception in case of failure in test
     * 
     * @see {@link CollectionUtil#pickupLast(Collection)}
     */
    @Test
    public void
    test_pickupLast()
    throws Exception
    {
        final Object[][] TEST_CASES = new Object[][] {
                new Object[] { null, null },
                new Object[] { new Object[] { null }, null },
                new Object[] { new Object[] { "hello", null }, null },
                new Object[] { new Object[] { null, "hello" }, "hello" },
                new Object[] { new Object[] { "A", "B" }, "B" },
        };

        int iTestCase = 0;
        for ( final Object[] TEST_CASE : TEST_CASES )
        {
			++iTestCase;
            final Object[] closeables = (Object[]) TEST_CASE[0];
            final Object expected = (Object) TEST_CASE[1];

            Collection<Object> collection = null;
            if ( null != closeables )
            {
                collection = Arrays.asList( closeables );
            }

            assertEquals( iTestCase + " th test case", expected, CollectionUtil.pickupLast( collection ) );
        }
    }

    /**
     * Test {@link CollectionUtil#iterator(Enumeration)}
     * 
     * @throws Exception in case of failure in test
     * 
     * @see {@link CollectionUtil#iterator(Enumeration)}
     */
    @Test
    public
    void
    test_iterator()
    throws Exception
    {
        final Object[][] TEST_CASES = new Object[][] {
                new Object[] { null },
                new Object[] { "hello" },
                new Object[] { null, "hello" },
                new Object[] { "A", "B" },
        };

        int iTestCase = 0;
        for ( final Object[] TEST_CASE : TEST_CASES )
        {
			++iTestCase;
            final Object[] array = TEST_CASE;
            final Vector<Object> list = new Vector<Object>( Arrays.asList( array ) );
            final Iterator<Object> iter = list.iterator();
            final Enumeration<Object> enumeration = list.elements();

            final Iterator<Object> testTarget = CollectionUtil.iterator( enumeration );
            int iElement = 0;
            while ( iter.hasNext() && testTarget.hasNext() )
            {
                assertEquals( iTestCase + " th test case :" + (++iElement) + " th element", iter.next(), testTarget.next() );
            }
            assertEquals( iter.hasNext(), testTarget.hasNext() );
        }
    }

    /**
     * Test {@link CollectionUtil#asList(Object)}
     * 
     * @throws Exception in case of failure in test
     * 
     * @see {@link CollectionUtil#asList(Object)}
     */
    @Test
    public
    void
    test_asList()
    throws Exception
    {
        final Object[][] TEST_CASES = new Object[][] {
                new Object[] { null, null },
                new Object[] { new Object[] { null }, null },
                new Object[] { new Object[] { "hello", null }, null },
                new Object[] { new Object[] { null, "hello" }, "hello" },
                new Object[] { new Object[] { "A", "B" }, "B" },
        };

        for ( final Object[] TEST_CASE : TEST_CASES )
        {
            assertTrue( CollectionUtil.asList(TEST_CASE) instanceof List );
        }
    }

    /**
     * Test {@link CollectionUtil#mergeArrayIntoCollection(Object, Collection)}
     * 
     * @throws Exception is case of failure in test
     * 
     * @see {@link CollectionUtil#mergeArrayIntoCollection(Object, Collection)}
     */
    @Test
    public
    void
    test_mergeArrayIntoCollection()
    throws Exception
    {
        final ArrayList<Object> list = new ArrayList<Object>();
        list.add( "Tizen" );
        list.add( "Test" );

        final Object[][] TEST_CASES = new Object[][] {
                new Object[] { null, null },
                new Object[] { new Object[] { null }, null },
                new Object[] { new Object[] { "hello", null }, null },
                new Object[] { new Object[] { null, "hello" }, "hello" },
                new Object[] { new Object[] { "A", "B" }, "B" },
        };

        try {
            CollectionUtil.mergeArrayIntoCollection( TEST_CASES[0], null );
            fail( "mergeArrayIntoCollection must throw exception" );
        } catch ( IllegalArgumentException e )
        {
            LoggerFactory.getLogger( getClass() ).info( "Test case successful" );
        }

        for ( final Object[] TEST_CASE : TEST_CASES )
        {
            final int count = TEST_CASE.length + CollectionUtil.size( list );
            CollectionUtil.mergeArrayIntoCollection( TEST_CASE, list );
            assertEquals( count, CollectionUtil.size( list ) );
        }
    }

    /**
     * Test {@link CollectionUtil#contains(Collection, Object)}
     * and {@link CollectionUtil#contains(Enumeration, Object)}
     * and {@link CollectionUtil#contains(Iterator, Object)}
     * 
     * @throws Exception is case of failure in test
     * 
     * @see {@link CollectionUtil#contains(Collection, Object)}
     * @see {@link CollectionUtil#contains(Enumeration, Object)}
     * @see {@link CollectionUtil#contains(Iterator, Object)}
     */
    @Test
    public
    void
    test_contains()
    throws Exception
    {
        final Object[][] TEST_CASES = new Object[][] {
                new Object[] { null },
                new Object[] { "hello" },
                new Object[] { null, "hello" },
                new Object[] { "Tizen", "Test" },
        };

        assertFalse( CollectionUtil.contains( (Collection<Object>)null , null) );
        assertFalse( CollectionUtil.contains( (Enumeration<Object>)null , null) );
        assertFalse( CollectionUtil.contains( (Iterator<Object>)null , null) );

        for ( final Object[] TEST_CASE : TEST_CASES )
        {
            final Object[] array = TEST_CASE;
            final Vector<Object> list = new Vector<Object>( CollectionUtil.asList( array ) );
            final Enumeration<Object> enumeration = list.elements();
            final Iterator<Object> iter = CollectionUtil.iterator( list.elements() );
            final Object element = list.get(0);
            final Object element2 = "False";

            assertTrue( CollectionUtil.contains( list, element ) );
            assertTrue( CollectionUtil.contains( enumeration, element ) );
            assertTrue( CollectionUtil.contains( iter, element ) );

            assertFalse( CollectionUtil.contains( list, element2 ) );
            assertFalse( CollectionUtil.contains( enumeration, element2 ) );
            assertFalse( CollectionUtil.contains( iter, element2 ) );
        }
    }

    /**
     * Test {@link CollectionUtil#containsAny(Collection, Collection)}
     * 
     * @throws Exception is case of failure in test
     * 
     * @see {@link CollectionUtil#containsAny(Collection, Collection)}
     */
    @Test
    public
    void
    test_containsAny()
    throws Exception
    {
        final Object[][] TEST_CASES = new Object[][] {
                new Object[] { null },
                new Object[] { "hello" },
                new Object[] { null, "hello" },
                new Object[] { "Tizen", "Test" },
        };

        final ArrayList<String> candidates = new ArrayList<String>();
        candidates.add(null);
        candidates.add("hello");
        candidates.add("Test");

        final ArrayList<String> candidates2 = new ArrayList<String>();
        candidates2.add("False");

        final ArrayList<String> source = new ArrayList<String>();
        source.add("Tizen");
        source.add("Test");
        source.add(null);

        assertFalse( CollectionUtil.containsAny( (Collection<Object>)null , candidates) );
        assertFalse( CollectionUtil.containsAny( source, (Collection<Object>)null ) );

        for ( final Object[] TEST_CASE : TEST_CASES )
        {
            final Object[] array = TEST_CASE;
            final Vector<Object> source2 = new Vector<Object>( CollectionUtil.asList( array ) );

            assertTrue( CollectionUtil.containsAny( source2, candidates ) );
            assertFalse( CollectionUtil.containsAny( source2, candidates2 ) );
        }
    }

    /**
     * Test {@link CollectionUtil#isApproximableCollectionType(Class)}
     * 
     * @throws Exception is case of failure in test
     * 
     * @see {@link CollectionUtil#isApproximableCollectionType(Class)}
     */
    @Test
    public
    void
    test_isApproximableCollectionType()
    throws Exception
    {
        final List<Class<?>> collectionTypeList = Arrays.asList( new Class<?>[] {
                Collection.class,
                Set.class, HashSet.class, SortedSet.class, LinkedHashSet.class, TreeSet.class,
                List.class, LinkedList.class, ArrayList.class
            } );

        final List<Class<?>> mapTypeList = Arrays.asList( new Class<?>[] {
                Map.class, SortedMap.class, HashMap.class, LinkedHashMap.class, TreeMap.class
            } );

        final List<Class<?>> generalTypeList = Arrays.asList( new Class<?>[] {
                String.class, Integer.class, Exception.class, File.class
            } );

        assertFalse( CollectionUtil.isApproximableCollectionType( (Class<?>)null ) );

        final Iterator<Class<?>> iter = collectionTypeList.iterator();
        while( iter.hasNext() ) {
            assertTrue( CollectionUtil.isApproximableCollectionType( iter.next() ) );
        }

        final Iterator<Class<?>> iter2 = mapTypeList.iterator();
        while( iter2.hasNext() ) {
            assertFalse( CollectionUtil.isApproximableCollectionType( iter2.next() ) );
        }

        final Iterator<Class<?>> iter3 = generalTypeList.iterator();
        while( iter3.hasNext() ) {
            assertFalse( CollectionUtil.isApproximableCollectionType( iter3.next() ) );
        }
    }

    /**
     * Test {@link CollectionUtil#isApproximableMapType(Class)}
     * 
     * @throws Exception is case of failure in test
     * 
     * @see {@link CollectionUtil#isApproximableMapType(Class)}
     */
    @Test
    public
    void
    test_isApproximableMapType()
    throws Exception
    {
        final List<Class<?>> collectionTypeList = Arrays.asList( new Class<?>[] {
                Collection.class,
                Set.class, HashSet.class, SortedSet.class, LinkedHashSet.class, TreeSet.class,
                List.class, LinkedList.class, ArrayList.class
            } );

        final List<Class<?>> mapTypeList = Arrays.asList( new Class<?>[] {
                Map.class, SortedMap.class, HashMap.class, LinkedHashMap.class, TreeMap.class
            } );

        final List<Class<?>> generalTypeList = Arrays.asList( new Class<?>[] {
                String.class, Integer.class, Exception.class, File.class
            } );

        assertFalse( CollectionUtil.isApproximableMapType( (Class<?>)null ) );

        final Iterator<Class<?>> iter = collectionTypeList.iterator();
        while( iter.hasNext() ) {
            assertFalse( CollectionUtil.isApproximableMapType( iter.next() ) );
        }

        final Iterator<Class<?>> iter2 = mapTypeList.iterator();
        while( iter2.hasNext() ) {
            assertTrue( CollectionUtil.isApproximableMapType( iter2.next() ) );
        }

        final Iterator<Class<?>> iter3 = generalTypeList.iterator();
        while( iter3.hasNext() ) {
            assertFalse( CollectionUtil.isApproximableMapType( iter3.next() ) );
        }
    }

    /**
     * Test {@link CollectionUtil#createApproximableCollection(Collection, int)}
     * 
     * @throws Exception is case of failure in test
     * 
     * @see {@link CollectionUtil#createApproximableCollection(Collection, int)}
     */
    @Test
    public
    void
    test_createApproximableCollection()
    throws Exception
    {
        final ArrayList<String> list = new ArrayList<String>( Arrays.asList(
                "hello", "Hello", "World", null, "Test"
                ) );

        final HashMap<String, Object> map = new HashMap<String, Object>();
        map.put("Hello", 1);
        map.put("Tizen", 2);

        Collection<?> result = CollectionUtil.createApproximableCollection( null, 5 );
        assertTrue( CollectionUtil.isApproximableCollectionType( result.getClass() ) );
        result.clear();

        result = CollectionUtil.createApproximableCollection( list, 3 );
        assertTrue( CollectionUtil.isApproximableCollectionType( result.getClass() ) );
        result.clear();

        result = CollectionUtil.createApproximableCollection( map.keySet(), 2 );
        assertTrue( CollectionUtil.isApproximableCollectionType( result.getClass() ) );
    }

    /**
     * Test {@link CollectionUtil#createApproximableMap(Map, int)}
     * 
     * @throws Exception is case of failure in test
     * 
     * @see {@link CollectionUtil#createApproximableMap(Map, int)}
     */
    @Test
    public
    void
    test_createApproximableMap()
    throws Exception
    {
        final HashMap<String, Object> map = new HashMap<String, Object>();
        map.put("Hello", 1);
        map.put("Tizen", 2);

        Map<?,?> result = CollectionUtil.createApproximableMap( null, 5 );
        assertTrue( CollectionUtil.isApproximableMapType( result.getClass() ) );
        result.clear();

        result = CollectionUtil.createApproximableMap( map, 1 );
        assertTrue( CollectionUtil.isApproximableMapType( result.getClass() ) );
    }

    /**
     * Test {@link CollectionUtil#toString(Object)}
     * and {@link CollectionUtil#toString(boolean[])}
     * and {@link CollectionUtil#toString(char[])}
     * 
     * @throws Exception is case of failure in test
     * 
     * @see {@link CollectionUtil#toString(Object)}
     * @see {@link CollectionUtil#toString(boolean[])}
     * @see {@link CollectionUtil#toString(char[])}
     */
    @Test
    public
    void
    test_toString()
    throws Exception
    {
        Object obj = (Object)new ArrayList<String>( Arrays.asList(
                "hello", "Hello", "World", null, "Test"
                ) );
        boolean [] booleanArray = { true, false };
        boolean [] booleanArray2 = {};
        char [] charArray = { 'T', 'i', 'Z', 'e', 'N' };
        char [] charArray2 = {};

        // toString(Object)
        String result = CollectionUtil.toString( (Object)null );
        assertEquals( StringUtil.NULL_STRING, result);
        result = CollectionUtil.toString( (Object)booleanArray );
        assertEquals( "{true, false}", result );
        result = CollectionUtil.toString( (Object)charArray );
        assertEquals( "{T, i, Z, e, N}", result );
        result = CollectionUtil.toString(obj);
        assertEquals( "[hello, Hello, World, null, Test]", result);

        // toString(boolean[])
        result = CollectionUtil.toString( (boolean[])null );
        assertEquals( StringUtil.NULL_STRING, result);
        result = CollectionUtil.toString( booleanArray );
        assertEquals( "{true, false}", result );
        result = CollectionUtil.toString( booleanArray2 );
        assertEquals( "{}", result );

        // toString(char[])
        result = CollectionUtil.toString( (char[])null );
        assertEquals( StringUtil.NULL_STRING, result);
        result = CollectionUtil.toString( charArray );
        assertEquals( "{T, i, Z, e, N}", result );
        result = CollectionUtil.toString( charArray2 );
        assertEquals( "{}", result );
        
        assertEquals( "{true, false}", CollectionUtil.toString( new boolean[] { true, false } ) );
        assertEquals( "{a, b}", CollectionUtil.toString( new Object[] { "a", "b" } ) );
    }

    /**
     * Test {@link CollectionUtil#concatenate(Collection, String)}
     * and {@link CollectionUtil#concatenate(Object[], String)}
     * and {@link CollectionUtil#concatenate(Iterator, String)}
     * 
     * @throws Exception is case of failure in test
     * 
     * @see {@link CollectionUtil#concatenate(Collection, String)}
     * @see {@link CollectionUtil#concatenate(Object[], String)}
     * @see {@link CollectionUtil#concatenate(Iterator, String)}
     */
    @Test
    public
    void
    test_concatenate()
    throws Exception
    {
        final String separator = "$";

        final Object[][] TEST_CASES = new Object[][] {
                new Object[] {},
                new Object[] { null },
                new Object[] { "hello" },
                new Object[] { null, "hello" },
                new Object[] { "Tizen", "Test" },
        };

        assertEquals( StringUtil.NULL_STRING,
                CollectionUtil.concatenate( (Object[])null, separator ) );
        assertEquals( StringUtil.NULL_STRING,
                CollectionUtil.concatenate( (Collection<Object>)null, separator) );
        assertEquals( StringUtil.NULL_STRING,
                CollectionUtil.concatenate( (Iterator<Object>)null, separator) );

        int count = 0;
        for ( final Object[] TEST_CASE : TEST_CASES )
        {
            count++;
            final Object[] array = TEST_CASE;
            final Vector<Object> list = new Vector<Object>( CollectionUtil.asList( array ) );
            Iterator<Object> iter = CollectionUtil.iterator( list.elements() );

            switch (count) {
                case 1:
                    assertEquals( "{}",
                            CollectionUtil.concatenate( array, separator ) );
                    assertEquals( "{}",
                            CollectionUtil.concatenate( list, separator ) );
                    assertEquals( "{}",
                            CollectionUtil.concatenate( iter, separator ) );
                    break;
                case 2:
                    assertEquals( StringUtil.NULL_STRING,
                            CollectionUtil.concatenate( array, separator ) );
                    assertEquals( StringUtil.NULL_STRING,
                            CollectionUtil.concatenate( list, separator ) );
                    assertEquals( StringUtil.NULL_STRING,
                            CollectionUtil.concatenate( iter, separator ) );
                    break;
                case 3:
                    assertEquals( "hello",
                            CollectionUtil.concatenate( array, separator ) );
                    assertEquals( "hello",
                            CollectionUtil.concatenate( list, separator ) );
                    assertEquals( "hello",
                            CollectionUtil.concatenate( iter, separator ) );
                    break;
                case 4:
                    assertEquals( StringUtil.NULL_STRING+separator+"hello",
                            CollectionUtil.concatenate( array, separator ) );
                    assertEquals( StringUtil.NULL_STRING+separator+"hello",
                            CollectionUtil.concatenate( list, separator ) );
                    assertEquals( StringUtil.NULL_STRING+separator+"hello",
                            CollectionUtil.concatenate( iter, separator ) );
                    break;
                case 5:
                    assertEquals( "Tizen"+separator+"Test",
                            CollectionUtil.concatenate( array, separator ) );
                    assertEquals( "Tizen"+separator+"Test",
                            CollectionUtil.concatenate( list, separator ) );
                    assertEquals( "Tizen"+separator+"Test",
                            CollectionUtil.concatenate( iter, separator ) );

                    iter = CollectionUtil.iterator( list.elements() );
                    assertEquals( "Tizen"+null+"Test",
                            CollectionUtil.concatenate( array, null ) );
                    assertEquals( "Tizen"+null+"Test",
                            CollectionUtil.concatenate( list, null ) );
                    assertEquals( "Tizen"+null+"Test",
                            CollectionUtil.concatenate( iter, null ) );

                    iter = CollectionUtil.iterator( list.elements() );
                    assertEquals( "Tizen"+StringUtil.EMPTY_STRING+"Test",
                            CollectionUtil.concatenate( array, StringUtil.EMPTY_STRING ) );
                    assertEquals( "Tizen"+StringUtil.EMPTY_STRING+"Test",
                            CollectionUtil.concatenate( list, StringUtil.EMPTY_STRING ) );
                    assertEquals( "Tizen"+StringUtil.EMPTY_STRING+"Test",
                            CollectionUtil.concatenate( iter, StringUtil.EMPTY_STRING ) );
                    break;
                default:
                    break;
            }

        }
    }

    /**
     * Test {@link CollectionUtil#hashCode(Object)}
     * and {@link CollectionUtil#hashCode(boolean[])}
     * and {@link CollectionUtil#hashCode(int[])}
     * 
     * @throws Exception is case of failure in test
     * 
     * @see {@link CollectionUtil#hashCode(Object)}
     * @see {@link CollectionUtil#hashCode(boolean[])}
     * @see {@link CollectionUtil#hashCode(int[])}
     */
    @Test
    public
    void
    test_hashCode()
    throws Exception
    {
        Object obj = (Object)new ArrayList<String>( Arrays.asList(
                "hello", "Hello", "World", null, "Test"
                ) );
        boolean [] booleanArray = { true, false };
        boolean [] booleanArray2 = {};
        int [] intArray = { 1, 2, 3, 4, 5 };
        int [] intArray2 = {};

        // hashCode(Object)
        assertEquals( 0, CollectionUtil.hashCode( (Object)null ) );
        assertTrue( 0 != CollectionUtil.hashCode(obj) );
        assertTrue( 0 != CollectionUtil.hashCode( (Object)booleanArray ) );
        assertTrue( 0 != CollectionUtil.hashCode( (Object)booleanArray2) );
        assertTrue( 0 != CollectionUtil.hashCode( (Object)intArray) );
        assertTrue( 0 != CollectionUtil.hashCode( (Object)intArray2) );
        assertEquals( CollectionUtil.hashCode(obj), CollectionUtil.hashCode(obj) );

        // hashCode(boolean[])
        assertEquals( 0, CollectionUtil.hashCode( (boolean[])null ) );
        assertTrue( 0 != CollectionUtil.hashCode( booleanArray ) );
        assertTrue( 0 != CollectionUtil.hashCode( booleanArray2) );
        assertTrue( CollectionUtil.hashCode(booleanArray) != CollectionUtil.hashCode(booleanArray2) );
        assertEquals( CollectionUtil.hashCode(booleanArray), CollectionUtil.hashCode(booleanArray) );

        // hashCode(int[])
        assertEquals( 0, CollectionUtil.hashCode( (int[])null ) );
        assertTrue( 0 != CollectionUtil.hashCode( intArray ) );
        assertTrue( 0 != CollectionUtil.hashCode( intArray2) );
        assertTrue( CollectionUtil.hashCode(intArray) != CollectionUtil.hashCode(intArray2) );
        assertEquals( CollectionUtil.hashCode(intArray),  CollectionUtil.hashCode(intArray));
    }

    /**
     * Test {@link CollectionUtil#equals(Collection...)}
     * and {@link CollectionUtil#equals(Object[]...)}
     * 
     * @throws Exception is case of failure in test
     * 
     * @see {@link CollectionUtil#equals(Collection...)}
     * @see {@link CollectionUtil#equals(Object[]...)}
     */
    @SuppressWarnings("unchecked")
    @Test
    public
    void
    test_equals()
    throws Exception
    {
        final Object obj = (Object)new ArrayList<String>( Arrays.asList(
                "hello", "Hello", "World", null, "Test"
                ) );
        final Object obj2 = (Object)Arrays.asList(
                "hello", "Hello", "World", null, "Test"
                );
        final Boolean [] booleanArray = { true, false };
        final Integer [] intArray = { 1, 2, 3, 4, 5 };
        final Integer [] intArray2 = {};
        final Object [] objectArray = { 1, 2, 3, 4, 5 };
        final Object [] objectArray2 = { 1, 2, 3, 4, 5 };
        final Vector<Object> list = new Vector<Object>( CollectionUtil.asList(objectArray) );

        assertTrue( CollectionUtil.equals( (Object[])null, null) );
        assertTrue( CollectionUtil.equals( (Collection<Object>)null, null) );

        // equals(Object[]...)
        assertTrue( CollectionUtil.equals( objectArray, objectArray2 ) );
        assertTrue( CollectionUtil.equals( objectArray, (Object[])intArray ) );
        assertFalse( CollectionUtil.equals( objectArray, (Object[])intArray2 ) );
        assertFalse( CollectionUtil.equals( objectArray, (Object[])booleanArray ) );

        // equals(COllection...)
        assertTrue( CollectionUtil.equals( (Collection<?>)obj, (Collection<?>)obj2 ) );
        assertFalse( CollectionUtil.equals( (Collection<?>)obj, (Collection<?>)list ) );
    }

    /**
     * Test {@link CollectionUtil#swap(Object[], int, int)}
     * 
     * @throws Exception is case of failure in test
     * 
     * @see {@link CollectionUtil#swap(Object[], int, int)}
     */
    @Test
    public
    void
    test_swap()
    throws Exception
    {
        final Object [] objectArray = { 1, 2, 3, 4, 5 };

        CollectionUtil.swap( objectArray, 0, 1 );
        assertEquals( 2, objectArray[0] );
        assertEquals( 1, objectArray[1] );

        CollectionUtil.swap( objectArray, 1, 0 );
        assertEquals( 1, objectArray[0] );
        assertEquals( 2, objectArray[1] );

        CollectionUtil.swap( objectArray, 0, 0 );
        assertEquals( 1, objectArray[0] );
        assertEquals( 2, objectArray[1] );

        try {
            CollectionUtil.swap( objectArray, 0, 5 );
            fail( "swap must throw exception" );
        } catch ( final ArrayIndexOutOfBoundsException e) {
            LoggerFactory.getLogger( getClass() ).info( "Test case successful" );
        }

        try {
            CollectionUtil.swap( objectArray, -1, 0 );
            fail( "swap must throw exception" );
        } catch ( final ArrayIndexOutOfBoundsException e) {
            LoggerFactory.getLogger( getClass() ).info( "Test case successful" );
        }

        try {
            CollectionUtil.swap( null, 0, 1 );
            fail( "swap must throw exception" );
        } catch ( final NullPointerException e) {
            LoggerFactory.getLogger( getClass() ).info( "Test case successful" );
        }
    }
    
    /**
     * Test {@link CollectionUtil#isAvailableGenericTypeForCollection(Collection, Class)}
     * 
     * @throws Exception is case of failure in test
     * 
     * @see {@link CollectionUtil#resolveSetAsList(Set)}
     */
    @Test
    public
    void
    test_isAvailableGenericTypeForCollection() {
        ArrayList<Object> testList = new ArrayList<Object>();
        
        int result = CollectionUtil.isAvailableGenericTypeForCollection(testList, Object.class);
        assertTrue(result == 0);
        
        testList.add(null);
        result = CollectionUtil.isAvailableGenericTypeForCollection(testList, Object.class);
        assertTrue(result == 0);
        
        testList.add("empty");
        result = CollectionUtil.isAvailableGenericTypeForCollection(testList, File.class);
        assertTrue(result == -1);
        
        result = CollectionUtil.isAvailableGenericTypeForCollection(testList, Object.class);
        assertTrue(result == -1);
        
        result = CollectionUtil.isAvailableGenericTypeForCollection(testList, String.class);
        assertTrue(result == 1);
        
        testList.add(new Object());
        result = CollectionUtil.isAvailableGenericTypeForCollection(testList, Object.class);
        assertTrue(result == 1);
        
        testList.remove(0);
        result = CollectionUtil.isAvailableGenericTypeForCollection(testList, String.class);
        assertTrue(result == 1);
    }
    
    
    
    /**
     * Test {@link CollectionUtil#resolveSetAsList(Set)}
     * 
     * @throws Exception is case of failure in test
     * 
     * @see {@link CollectionUtil#resolveSetAsList(Set)}
     */
    @Test
    public
    void
    test_resolveSetAsList() {
        LinkedHashSet<String> set = new LinkedHashSet<String>();
        String[] stringArray= {"one", "two", "three", "four", "five"};
        
        for(String string: stringArray) {
            set.add(string);
        }
        
        List<String> result = CollectionUtil.resolveSetAsList(set);
        
        for(int i=0; i<stringArray.length; i++) {
            assertTrue( (i+1) + "th test case: input - " + stringArray[i] , result.get(i).equals(stringArray[i]));
        }
    }
}

