/*
*  Common
*
* Copyright (c) 2000 - 2011 Samsung Electronics Co., Ltd. All rights reserved.
*
* Contact: 
* BonYong Lee <bonyong.lee@samsung.com>
* 
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
* http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
* Contributors:
* - S-Core Co., Ltd
*
*/
package org.tizen.common.util;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;

import java.io.File;

import org.junit.Test;


/**
 * FilenameUtilTest
 *
 * Test case for {@link FilenameUtil}
 * 
 * @author BonYong Lee{@literal <bonyong.lee@samsung.com>} (S-Core)
 * 
 * @see FilenameUtil
 */
public class
FilenameUtilTest
{
	/**
	 * Tests {@link FilenameUtil#getCanonicalFragments(String)}.
	 * 
	 * @throws Exception in case of failure in test
	 * 
	 * @see FilenameUtil#getCanonicalFragments(String)
	 */
	@Test
	public
	void
	test_getCanonicalFragments()
	throws Exception
	{
		final Object[][] TEST_CASES = new Object[][] {
			new Object[] { "/aaa/bbb/ccc", new String[] { "aaa", "bbb", "ccc" } },
			new Object[] { "/aaa/'\\bbb'/ccc", new String[] { "aaa", "\\bbb", "ccc" } },
			new Object[] { File.separator + "aaa" + File.separator + "bbb" +  File.separator + "ccc", new String[] { "aaa", "bbb", "ccc" } },
		};
		
		int iTest = 0;
		for ( final Object[] TEST_CASE : TEST_CASES )
		{
			final String input = (String) TEST_CASE[0];
			final String[] expected = (String[]) TEST_CASE[1];
			
			assertArrayEquals(
				++iTest + " th test failed. input :<" + input + ">",
				expected,
				FilenameUtil.getCanonicalFragments( input )
			);
		}
	}
	
    /**
     * Tests {@link FilenameUtil#getCanonicalForm(String)}.
     * 
     * @throws Exception in case of failure in test
     * 
     * @see FilenameUtil#getCanonicalForm(String)
     */
	@Test
	public void test_getCanonicalForm() throws Exception
    {
        final Object[][] TEST_CASES = new Object[][] {
            new Object[] { "/aaa/bbb/.", "/aaa/bbb" },
            new Object[] { "/aaa/bbb/ccc/..", "/aaa/bbb" },
            new Object[] { File.separator + "aaa" + File.separator + "bbb" +  File.separator + "ccc", "/aaa/bbb/ccc" },
        };
        
        int iTest = 0;
        for ( final Object[] TEST_CASE : TEST_CASES )
        {
            ++iTest;
            final String input = (String) TEST_CASE[0];
            final String expected = (String) TEST_CASE[1];
            
            assertEquals(
                iTest + " th test failed. input :<" + input + ">",
                expected,
                FilenameUtil.getCanonicalForm( input )
            );
        }
    }
	
	/**
	 * Tests {@link FilenameUtil#addTailingPath(String, String)}.
	 * 
	 * @throws Exception in case of failure in test
	 * 
	 * @see FilenameUtil#addTailingPath(String, String)
	 */
	@Test
	public
	void
	test_addTailingPath()
	throws Exception
	{
		final Object[][] TEST_CASES = new Object[][] {
			new Object[] { "/aaa/bbb/ccc", "aaa", "/aaa/bbb/ccc/aaa" },
			new Object[] { "/aaa/bbb/ccc/", "aaa", "/aaa/bbb/ccc/aaa" },
			new Object[] { "/aaa/bbb/ccc/", "/aaa", "/aaa/bbb/ccc/aaa" },
			new Object[] { "/home/bylee/tizen_sdk/tools/ide", "conf/profiles.xml", "/home/bylee/tizen_sdk/tools/ide/conf/profiles.xml" },
			new Object[] { "/home/bylee/tizen_sdk/tools/ide/", "conf/profiles.xml", "/home/bylee/tizen_sdk/tools/ide/conf/profiles.xml" },
			new Object[] { "/home/bylee/tizen_sdk/tools/ide/", "/conf/profiles.xml", "/home/bylee/tizen_sdk/tools/ide/conf/profiles.xml" },
			new Object[] { "C:\test", "profiles.xml", "C:\test/profiles.xml" }
		};
		
		int iTest = 0;
		for ( final Object[] TEST_CASE : TEST_CASES )
		{
			++iTest;
			final String path1 = (String) TEST_CASE[0];
			final String path2 = (String) TEST_CASE[1];
			final String expected = (String) TEST_CASE[2];
			
			assertEquals(
				iTest + " th test failed. input :<" + path1 + "," + path2 + ">",
				expected,
				FilenameUtil.addTailingPath( path1, path2 )
			);
		}
	}
	
	/**
	 * Tests {@link FilenameUtil#removeTailingPath(String, int)}.
	 * 
	 * @throws Exception in case of failure in test
	 * 
	 * @see FilenameUtil#removeTailingPath(String, int)
	 */
	@Test
	public
	void
	test_removeTailingPath()
	throws Exception
	{
		final Object[][] TEST_CASES = new Object[][] {
			new Object[] { "/aaa/bbb/ccc", 1, "/aaa/bbb" },
			new Object[] { File.separator + "aaa" + File.separator + "bbb" +  File.separator + "ccc", 1, "/aaa/bbb" },
		};
		
		int iTest = 0;
		for ( final Object[] TEST_CASE : TEST_CASES )
		{
			++iTest;
			final String input = (String) TEST_CASE[0];
			final int nRemove = (Integer) TEST_CASE[1];
			final String expected = (String) TEST_CASE[2];
			
			assertEquals(
				iTest + " th test failed. input :<" + input + ">",
				expected,
				FilenameUtil.removeTailingPath( input, nRemove )
			);
		}
	}

	/**
	 * Tests {@link FilenameUtil#getTailingPath(String, int)}.
	 * 
	 * @throws Exception in case of failure in test
	 * 
	 * @see FilenameUtil#getTailingPath(String, int)
	 */
	@Test
	public
	void
	test_getTailingPath()
	throws Exception
	{
		final Object[][] TEST_CASES = new Object[][] {
			new Object[] { "/aaa/bbb/ccc", 1, "ccc" },
			new Object[] { "/aaa/bbb/ccc/", 1, "ccc" },
			new Object[] { "/aaa/bbb/ccc", 2, "bbb/ccc" },
			new Object[] { File.separator + "aaa" + File.separator + "bbb" +  File.separator + "ccc", 1, "ccc" },
		};
		
		int iTest = 0;
		for ( final Object[] TEST_CASE : TEST_CASES )
		{
			++iTest;
			final String input = (String) TEST_CASE[0];
			final int nRemove = (Integer) TEST_CASE[1];
			final String expected = (String) TEST_CASE[2];
			
			assertEquals(
				iTest + " th test failed. input :<" + input + ">",
				expected,
				FilenameUtil.getTailingPath( input, nRemove )
			);
		}
	}
	
	/**
	 * Tests {@link FilenameUtil#getFilename(String)}.
	 * 
	 * @throws Exception in case of failure in test
	 * 
	 * @see FilenameUtil#getFilename(String)
	 */
	@Test
	public
	void
	test_getFilename()
	throws Exception
	{
		final Object[][] TEST_CASES = new Object[][] {
			new Object[] { "/aaa/bbb/ccc", "ccc" },
			new Object[] { "/aaa/bbb/ccc.jar", "ccc.jar" },
			new Object[] { File.separator + "aaa" + File.separator + "bbb" +  File.separator + "ccc", "ccc" },
		};
		
		int iTest = 0;
		for ( final Object[] TEST_CASE : TEST_CASES )
		{
			++iTest;
			final String input = (String) TEST_CASE[0];
			final String expected = (String) TEST_CASE[1];
			
			assertEquals(
				iTest + " th test failed. input :<" + input + ">",
				expected,
				FilenameUtil.getFilename( input )
			);
		}
	}
	
	/**
	 * Tests {@link FilenameUtil#getExtension(String)}.
	 * 
	 * @throws Exception in case of failure in test
	 * 
	 * @see FilenameUtil#getExtension(String)
	 */
	@Test
	public
	void
	test_getExtension()
	throws Exception
	{
		final Object[][] TEST_CASES = new Object[][] {
			new Object[] { "test.txt", "txt" },
			new Object[] { "ccc.jar", "jar" },
			new Object[] { ".aaa", "aaa" },
			new Object[] { "aaa", "" },
		};
		
		int iTest = 0;
		for ( final Object[] TEST_CASE : TEST_CASES )
		{
			++iTest;
			final String input = (String) TEST_CASE[0];
			final String expected = (String) TEST_CASE[1];
			
			assertEquals(
				iTest + " th test failed. input :<" + input + ">",
				expected,
				FilenameUtil.getExtension( input )
			);
		}
	}
    /** 
     * Tests {@link FilenameUtil#getRelativePath(String, String)}.
     * 
     * @see FilenameUtil#getRelativePath(String, String)
     */
    @Test
    public void test_getRelativePath() {
        String cwd = "/a/b/c/d";
        String[][] test_list = {
                //input             expected
                {"/a/b/c/d",        ""},
                {"/a/b/c/d/e",      "e"},
                {"/a/b/d/e/f",      "../../d/e/f"},
                {"/a/b/d",          "../../d"},
                {"/b/d/e",          "../../../../b/d/e"}
        };

        for (int i = 0 ; i<test_list.length; i++) {
            String[] test = test_list[i];

            String actual = FilenameUtil.getRelativePath(cwd, test[0]);
            String expected = test[1];

            assertEquals(expected, actual);
        }
    }

    /** 
     * Tests {@link FilenameUtil#getRelativePathUsingURI(String, String)}.
     * 
     * @see FilenameUtil#getRelativePathUsingURI(String, String)
     */
    @Test
    public void test_getRelativePath2() {
        String[][] test_list = {
                // cwd, input             expected
                {"/a/b/c/d", "/a/b/c/d",        ""},
                {"/a/b/c/d", "/a/b/c/d/e",      "e"},
                {"/a/b/c/d", "/a/b/d/e/f",      "../../d/e/f"},
                {"/a/'b/c/d", "/a/'b/d/e/f",      "../../d/e/f"},
                {"/a/b/c/d", "/a/b/d/e'/f",      "../../d/e'/f"},
                {"/a/b/c/d", "/a/b/d",          "../../d"},
                {"/a/b/c/d", "/b/d/e",          "../../../../b/d/e"},
                {"/a/b/c'e/d'\"r", "/a/b/d'\"r",          "../../d'\"r"},
                {"C:\\Windows\\Speech\\Common", "C:\\Windows\\Boot\\Fonts\\chs_boot.ttf", "../../Boot/Fonts/chs_boot.ttf"},
        };

        for (int i = 0 ; i<test_list.length; i++) {
            String[] test = test_list[i];
            
            String actual = FilenameUtil.getRelativePath2(test[0], test[1]);
            String expected = test[2];
            System.out.printf( "%s, %s, %s, %s\n", test[0], test[1], test[2], actual );
            assertEquals(expected, actual);
        }
    }

    
    /** 
     * Tests {@link FilenameUtil#hasExtension(String, String)}.
     * 
     * @see FilenameUtil#hasExtension(String, String)
     */
    @Test
    public void test_hasExtension() {
        Object[][] test_list = {
                //input1            input2          expected
                {"abc.exe",         "exe",          true},
                {"abc",             "",             true},
                {"",                "",             false},
                {"",                "exe",          false},
                {"abc.exe",         "bat",          false},
                {"abc",             "exe",          false},
                {"abc.exe.exe1",    "exe1",         true},
                {"abc.",            "",             true},
                {"abc..",           "",             true},
                {"/home/abc.txt",   "txt",          true}
        };

        for (Object[] test : test_list) {
            boolean actual = FilenameUtil.hasExtension((String)test[0], (String)test[1]);
            boolean expected = (Boolean) test[2];

            assertEquals(expected, actual);
        }
    }
}
