/*
*  Common
*
* Copyright (c) 2000 - 2011 Samsung Electronics Co., Ltd. All rights reserved.
*
* Contact: 
* BonYong Lee <bonyong.lee@samsung.com>
* 
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
* http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
* Contributors:
* - S-Core Co., Ltd
*
*/
package org.tizen.common.util;

import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import java.awt.image.BufferedImage;
import java.io.InputStream;
import java.net.URL;

import org.eclipse.core.runtime.Plugin;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.ImageData;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.mockito.Mockito;
import org.osgi.framework.Bundle;
import org.tizen.common.SurrogateWithArgument;


/**
 * ImageUtilTest
 *
 * Test case for {@link ImageUtil}
 * 
 * LookAtMe : "SetUp" & "TearDown" pattern for test
 * 
 * @author BonYong Lee{@literal <bonyong.lee@samsung.com>} (S-Core)
 * 
 * @see ImageUtil
 */
public class ImageUtilTest {
    /**
     * Property key for URL handler
     */
    protected static final String PROP_PKG = "java.protocol.handler.pkgs";

    protected static final String TEST_PATH = ImageUtilTest.class.getSimpleName() + ".png";
    protected static final String TEST_URL = "cp:///" + TEST_PATH;
    protected static final String JPG_TEST_PATH = ImageUtilTest.class.getSimpleName() + ".jpg";
    protected static final String JPG_TEST_URL = "cp:///" + JPG_TEST_PATH;

    /**
     * Old property value for URL handler
     */
    protected String oldConfig;

    /**
     * Set up URL handler before test
     */
    @SuppressWarnings("unchecked")
    @Before
    public void setUp() {
        oldConfig = System.getProperty( PROP_PKG );
        System.setProperty( PROP_PKG, "org.tizen.common.util.url" );

        ImageUtil.platformSurrogate = mock(SurrogateWithArgument.class);
        ImageUtil.pluginSurrogate = mock(SurrogateWithArgument.class);

    }

    /**
     * Restore URL handler after test
     */
    @After
    public void tearDown() {
        ImageUtil.pluginSurrogate = null;
        ImageUtil.platformSurrogate = null;

        if ( null == oldConfig ) {
            System.clearProperty( PROP_PKG );
        } else {
            System.setProperty( PROP_PKG, oldConfig );
        }
    }

    /**
     * Test {@link ImageUtil#getImageDescriptor(Bundle, String)}
     * 
     * @throws Exception in case of failure in test
     * 
     * @see ImageUtil#getImageDescriptor(Bundle, String)
     */
    @Test
    public void test_getImageDescriptor1() throws Exception {
        final Bundle bundleMockup = mock( Bundle.class );
        final URL virtualPathUrl = new URL( TEST_URL );

        when( bundleMockup.getEntry( TEST_PATH ) ).thenReturn( virtualPathUrl );
        final ImageDescriptor imgDes = ImageUtil.getImageDescriptor( bundleMockup, TEST_PATH );

        Assert.assertNotNull(imgDes.createImage(false));
    }

    /**
     * Test {@link ImageUtil#getImageDescriptor(String, String)}
     * 
     * @throws Exception in case of failure in test
     * 
     * @see ImageUtil#getImageDescriptor(String, String)
     */
    @Test
    public void test_getImageDescriptor2() throws Exception {
        final Bundle bundleMockup = mock( Bundle.class );
        final URL virtualPathUrl = new URL( TEST_URL );
        when( bundleMockup.getEntry( TEST_PATH ) ).thenReturn( virtualPathUrl );
        final String pluginID = "testID";
        when(ImageUtil.platformSurrogate.getAdapter(pluginID)).thenReturn(bundleMockup);
        final ImageDescriptor imgDes = ImageUtil.getImageDescriptor(pluginID, TEST_PATH);

        final Image img = imgDes.createImage(false);

        Assert.assertNotNull(img);
    }

    /**
     * Test {@link ImageUtil#getImageDescriptor(Plugin, String)}
     * 
     * @throws Exception in case of failure in test
     * 
     * @see ImageUtil#getImageDescriptor(Plugin, String)
     */
    @Test
    public void test_getImageDescriptor3() throws Exception {
        final Bundle bundleMockup = mock( Bundle.class );
        final URL virtualPathUrl = new URL( TEST_URL );
        when( bundleMockup.getEntry( TEST_PATH ) ).thenReturn( virtualPathUrl );
        final Plugin pluginMockup = mock(Plugin.class);
        when(ImageUtil.pluginSurrogate.getAdapter(pluginMockup)).thenReturn(bundleMockup);
        final ImageDescriptor imgDes = ImageUtil.getImageDescriptor(pluginMockup, TEST_PATH);
        final Image img = imgDes.createImage(false);

        Assert.assertNotNull(img);
    }

    /**
     * Test {@link ImageUtil#getImage(ImageDescriptor)}
     * 
     * @throws Exception in case of failure in test
     * 
     * @see ImageUtil#getImage(ImageDescriptor)
     */
    @Test
    public void test_getImage1() throws Exception {
        final ImageDescriptor imgMock = mock(ImageDescriptor.class);
        ImageUtil.getImage(imgMock);
        Mockito.verify(imgMock).createImage();
    }

    /**
     * Test {@link ImageUtil#getImageData(ImageDescriptor)}
     * 
     * @throws Exception in case of failure in test
     * 
     * @see ImageUtil#getImageData(ImageDescriptor)
     */
    @Test
    public void test_getImageData1() throws Exception {
        final ImageDescriptor imgMock = mock(ImageDescriptor.class);

        ImageUtil.getImageData(imgMock);
        Mockito.verify(imgMock).getImageData();
    }

    /**
     * Test {@link ImageUtil#getImage(Plugin, String)}
     * 
     * @throws Exception in case of failure in test
     * 
     * @see ImageUtil#getImage(Plugin, String)
     */
    @Test
    public void test_getImage2() throws Exception {
        final Bundle bundleMockup = mock( Bundle.class );
        final URL virtualPathUrl = new URL( TEST_URL );
        when( bundleMockup.getEntry( TEST_PATH ) ).thenReturn( virtualPathUrl );
        final Plugin pluginMockup = mock(Plugin.class);
        when(ImageUtil.pluginSurrogate.getAdapter(pluginMockup)).thenReturn(bundleMockup);

        Image img = ImageUtil.getImage(pluginMockup, TEST_PATH);
        Assert.assertNotNull(img);
    }

    /**
     * Test {@link ImageUtil#getImageData(Plugin, String)}
     * 
     * @throws Exception in case of failure in test
     * 
     * @see ImageUtil#getImageData(Plugin, String)
     */
    @Test
    public void test_getImageData2() throws Exception {
        final Bundle bundleMockup = mock( Bundle.class );
        final URL virtualPathUrl = new URL( TEST_URL );
        when( bundleMockup.getEntry( TEST_PATH ) ).thenReturn( virtualPathUrl );
        final Plugin pluginMockup = mock(Plugin.class);
        when(ImageUtil.pluginSurrogate.getAdapter(pluginMockup)).thenReturn(bundleMockup);

        ImageData imgData = ImageUtil.getImageData(pluginMockup, TEST_PATH);
        Assert.assertNotNull( imgData );
    }

    /**
     * Test {@link ImageUtil#getImage(String, String)}
     * 
     * @throws Exception in case of failure in test
     * 
     * @see ImageUtil#getImage(String, String)
     */
    @Test
    public void test_getImage3() throws Exception {
        final String pluginID = "testID";
        final Bundle bundleMockup = mock( Bundle.class );
        final URL virtualPathUrl = new URL( TEST_URL );
        when( bundleMockup.getEntry( TEST_PATH ) ).thenReturn( virtualPathUrl );
        when(ImageUtil.platformSurrogate.getAdapter(pluginID)).thenReturn(bundleMockup);

        Image img = ImageUtil.getImage(pluginID, TEST_PATH);
        Assert.assertNotNull(img);
    }

    /**
     * Test {@link ImageUtil#getImageData(String, String)}
     * 
     * @throws Exception in case of failure in test
     * 
     * @see ImageUtil#getImageData(String, String)
     */
    @Test
    public void test_getImageData3() throws Exception {
        final String pluginID = "testID";
        final Bundle bundleMockup = mock( Bundle.class );
        final URL virtualPathUrl = new URL( TEST_URL );
        when( bundleMockup.getEntry( TEST_PATH ) ).thenReturn( virtualPathUrl );
        when(ImageUtil.platformSurrogate.getAdapter(pluginID)).thenReturn(bundleMockup);

        ImageData imgData = ImageUtil.getImageData(pluginID, TEST_PATH);
        Assert.assertNotNull( imgData );
    }

    /**
     * Test {@link ImageUtil#getBufferedImage(InputStream)}
     * 
     * @throws Exception in case of failure in test
     * 
     * @see ImageUtil#getBufferedImage(InputStream)
     */
    @Test
    public void test_getBufferedImage() throws Exception {
        final URL virtualJPGPathUrl = new URL(JPG_TEST_URL);
        InputStream input = virtualJPGPathUrl.openStream();
        try {
            BufferedImage image = ImageUtil.getBufferedImage(input);
            if (image != null) {
                Assert.assertNotNull(image);
                image.flush();
            }
        } finally {
            IOUtil.tryClose(input);
        }

        final URL virtualPNGPathUrl = new URL(TEST_URL);
        input = virtualPNGPathUrl.openStream();
        try {
            BufferedImage image = ImageUtil.getBufferedImage(input);
            if (image != null) {
                Assert.assertNotNull(image);
                image.flush();
            }
        } finally {
            IOUtil.tryClose(input);
        }
    }

}
