/*
*  Common
*
* Copyright (c) 2000 - 2011 Samsung Electronics Co., Ltd. All rights reserved.
*
* Contact:
* ChangHyun Lee <changhyun1.lee@samsung.com>
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
* http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
* Contributors:
* - S-Core Co., Ltd
*
*/

package org.tizen.common.util;

import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;

import org.junit.Test;

/**
 * ValidationUtilTest.
 *
 * Test case for {@link ValidationUtil}
 *
 * @author Changhyun Lee{@literal <changhyun1.lee@samsung.com>} (S-Core)
 *
 * @see ValidationUtil
 *
 */
public class ValidationUtilTest {

    /**
     * Test {@link ValidationUtil#checkForEmail(String value)}
     *
     * @throws Exception in case of failure in test
     *
     * @see {@link ValidationUtil#checkForEmail(String value)}
     */
    @Test
    public void test_checkForEmail() throws Exception {
        final String[] validedEmail = {
                "changhyun1.lee@samsung.com",
                "12345@exmaple.com",
                "abcde@exmaple.com",
                "!a!b#c%d@example.com",
                "_abcd@example.com",
                "niceandsimple@example.com",
                "simplewithsymbol@example.com",
                "less.common@example.com",
                "a.little.more.unusual@dept.example.com"
            };
        for (String email : validedEmail) {
            assertTrue(ValidationUtil.checkForEmail(email));
        }

        final String[] InvalidedEmail = {
                "changhyun1.lee@samsung",
                "changhyun1.leesamsung.com",
                "changhyun1.lee@samsung.com/",
                "changhyun1.lee@samsung.com!",
                "@samsung.com",
                "12345!exmaple.com",
                "!a!b#c%d@example.co!m",
                "_abcd@example_com",
                "user@[IPv6:2001:db8:1ff::a0b:dbd0]",
                "\"much.more unusual\"@example.com",
                "\"very.unusual.@.unusual.com\"@example.com",
                "\"very.(),:;<>[]\".VERY.\"very@\\ \"very\".unusual\"@strange.example.com",
                "0@a",
                "\"!#$%&'*+-/=?^_`{}|~@example.org",
                "\"()<>[]:,;@\\\"!#$%&'*+-/=?^_`{}| ~  ? ^_`{}|~.a\"@example.org",
                "\"\"@example.org",
                "postbox@com", // top-level domains are invalid hostnames
                "Abc.example.com", // an @ character must separate the local and domain parts
                "Abc.@example.com", // character dot(.) is last in local part
                "Abc..123@example.com", // character dot(.) is double
                "A@b@c@example.com ", // only one @ is allowed outside quotation marks
                "a\"b(c)d,e:f;g<h>i[j\\k]l@example.com", // none of the special characters in this local part is allowed outside quotation marks
                "just\"not\"right@example.com", // quoted strings must be dot separated, or the only element making up the local-part
                "this is\"not\\allowed@example.com", // spaces, quotes, and backslashes may only exist when within quoted strings and preceded by a slash
                "this\\ still\"not\\allowed@example.com" // even if escaped (preceded by a backslash), spaces, quotes, and backslashes must still be contained by quotes
            };
        for (String email : InvalidedEmail) {
            assertFalse(ValidationUtil.checkForEmail(email));
        }
    }

    /**
     * Test {@link ValidationUtil#checkForIRI(String value)}
     *
     * @throws Exception in case of failure in test
     *
     * @see {@link ValidationUtil#checkForIRI(String value)}
     */
    @Test
    public void test_checkForIRI() throws Exception {
        final String[] validedIRIs = {"http://tizen.org",
                "http://www.tizen.org",
                "http://tizen.org/",
                "http://tizen.org/id",
                "http://myapp",
                "http://myapp/",
                "http://bonyong.lee:test@myapp:4000/test.dd/aaa#test",
                "http://example.org/about/mother.fr",
                "http://example.org/contact.nl",
                "http://example.org/contact?lang=nl",
                "http://example.org/nl/contact",
                "http://example.org/nl/Contact",
                "http://example.org/nl/CONTACT",
                "cxd:dsf",
                "ftp://fds",
                "irc://fdf"
            };
        for (String url : validedIRIs) {
            assertTrue(ValidationUtil.checkForIRI(url));
        }
    }

    /**
     * Test {@link ValidationUtil#checkForURL(String value)}
     *
     * @throws Exception in case of failure in test
     *
     * @see {@link ValidationUtil#checkForURL(String value)}
     */
    @Test
    public void test_checkForURL() throws Exception {
        final String[] validedURL = {"http://tizen.org", "http://www.tizen.org", "http://tizen.org/", "http://tizen.org/id", "http://myapp"};
        for (String url : validedURL) {
            assertTrue(ValidationUtil.checkForURL(url));
        }

        final String[] invalidedURL = {"http:/tizen.org", "tizen.org", "www.tizen.org", "http://tizen.org/!", "http://tizen.org/id^", "http://tizen.org/id*"};
        for (String url : invalidedURL) {
            assertFalse(ValidationUtil.checkForURL(url));
        }
    }

    /**
     * Test {@link ValidationUtil#checkForFileExtension(String value, String[] extensions)}
     *
     * @throws Exception in case of failure in test
     *
     * @see {@link ValidationUtil#checkForFileExtension(String value, String[] extensions)}
     */
    @Test
    public void test_checkForFileExtension() throws Exception {
        final String[] extensions = {".html", ".js", ".css"};
        assertTrue(ValidationUtil.checkForFileExtension("file.html", extensions));
        assertTrue(ValidationUtil.checkForFileExtension("123.html", extensions));
        assertTrue(ValidationUtil.checkForFileExtension("file.js", extensions));
        assertTrue(ValidationUtil.checkForFileExtension("123.js", extensions));
        assertTrue(ValidationUtil.checkForFileExtension("file.css", extensions));
        assertTrue(ValidationUtil.checkForFileExtension("123.css", extensions));

        assertFalse(ValidationUtil.checkForFileExtension("file.html1", extensions));
        assertFalse(ValidationUtil.checkForFileExtension("123.html!", extensions));
        assertFalse(ValidationUtil.checkForFileExtension("file.jshtml", extensions));
        assertFalse(ValidationUtil.checkForFileExtension("123.js2", extensions));
        assertFalse(ValidationUtil.checkForFileExtension("file.aaacss", extensions));
        assertFalse(ValidationUtil.checkForFileExtension("123.111css", extensions));
    }

    /**
     * Test {@link ValidationUtil#checkForVersion(String value)}
     *
     * @throws Exception in case of failure in test
     *
     * @see {@link ValidationUtil#checkForVersion(String value)}
     */
    @Test
    public void test_checkForVersion() throws Exception {
        final String[] validedVersions = { "0.0.0", "0.0.1", "0.1.0", "1.0.0", "1.0.1", "1.1.1", "0.0.00001", "000.000.00001", "000.001.00000", "001.000.00000", "001.001.00001", "255.0.0", "0.255.0", "0.0.65535", "255.255.65535" };
        for ( String version : validedVersions ) {
            assertTrue( "Failed for \"" + version + "\"", ValidationUtil.checkForVersion( version ) );
        }

        final String[] invalidedVersions = { "", "256.256.65536", "0", "0.0.", "1", "256.0.0", "0.256.0", "0.0.65536", "0000.000.00001", "000.0000.00001", "000.000.000001" };
        for ( String version : invalidedVersions ) {
            assertFalse( "Failed for \"" + version + "\"", ValidationUtil.checkForVersion( version ) );
        }
    }

    /**
     * Test {@link ValidationUtil#checkForDynamicBoxId(String value)}
     *
     * @throws Exception in case of failure in test
     *
     * @see {@link ValidationUtil#checkForDynamicBoxId(String value)}
     */
    @Test
    public void test_checkForDynamicBoxId() throws Exception {
        final String[] validedIDS = { "1wfV7ZwO6r.test.test", "1wfV7ZwO6r.test.111", "1wfV7ZwO6r.test.test2222" };
        for ( String id : validedIDS ) {
            assertTrue( "Failed for \"" + id + "\"", ValidationUtil.checkForDynamicBoxId( id ) );
        }

        final String[] invalidedIDS = { "", "1wfV7ZwO6r.test.-1", "1wfV7ZwO6r.test.!test", "1wfV7ZwO6r.test.test test", "1wfV7ZwO6r.test.test!#", "1wfV7ZwO6r." };
        for ( String id : invalidedIDS ) {
            assertFalse( "Failed for \"" + id + "\"", ValidationUtil.checkForDynamicBoxId( id ) );
        }
    }

    /**
     * Test {@link ValidationUtil#checkForServiceId(String value)}
     *
     * @throws Exception in case of failure in test
     *
     * @see {@link ValidationUtil#checkForServiceId(String value)}
     */
    @Test
    public void test_checkForServiceId() throws Exception {
        final String[] validedServiceIds = { "1234567890.12", "efghiZKlmn.abc", "abcd123DFz.test" };
        for ( String id : validedServiceIds ) {
            assertTrue("Failed for \"" + id + "\"", ValidationUtil.checkForServiceId( id ) );
        }
        final String[] invalidedIds = {"123456789.12.test", "123abcdef.test", "abcde12.test", "abcdefghijklmn.test", "test.test.test." };
        for ( String id : invalidedIds ) {
            assertFalse("Failed for \"" + id + "\"", ValidationUtil.checkForServiceId( id ) );
        }
    }


    /**
     * Test {@link ValidationUtil#checkForServiceContentFileExtension(String value)}
     *
     * @throws Exception in case of failure in test
     *
     * @see {@link ValidationUtil#checkForServiceContentFileExtension(String value)}
     */
    @Test
    public void test_checkForServiceContentFileExtension() throws Exception {
        assertTrue(ValidationUtil.checkForServiceContentFileExtension("file.js"));
        assertTrue(ValidationUtil.checkForServiceContentFileExtension("1file2.js"));
        assertTrue(ValidationUtil.checkForServiceContentFileExtension("123.js"));

        assertFalse(ValidationUtil.checkForServiceContentFileExtension("abc.js1"));
        assertFalse(ValidationUtil.checkForServiceContentFileExtension("123.aajs"));
        assertFalse(ValidationUtil.checkForServiceContentFileExtension("abc.11js"));
    }


    /**
     * Test {@link ValidationUtil#checkForFileURI(String value)}
     *
     * @throws Exception in case of failure in test
     *
     * @see {@link ValidationUtil#checkForFileURI(String value)}
     */
    @Test
    public void test_checkForFileURI() throws Exception {
        final String[] validedFileURLs = { "icon", "icon~(1)", "icon_1", "icon~!@$&*()_+=;'.png", "~!@$&*()_+=;'." };
        for ( String id : validedFileURLs ) {
            assertTrue( "Failed for \"" + id + "\"", ValidationUtil.checkForFileURI( id ) );
        }

        final String[] invalidedFileURLs = { "icon:", "icon/icon", "icon/", "icon\"", "/\":`#%^<>", "http://my.com/icon.png", "file://myfile.png" };
        for ( String id : invalidedFileURLs ) {
            assertFalse( "Failed for \"" + id + "\"", ValidationUtil.checkForFileURI( id ) );
        }
    }

    /**
     * Test {@link ValidationUtil#checkForAccessOrigin(String orgin)}
     *
     * @throws Exception in case of failure in test
     *
     * @see {@link ValidationUtil#checkForAccessOrigin(String origin)}
     */
    @Test
    public void test_checkForAccessOrigin() throws Exception {
        final String[] validedOrigins = {"http://tizen.org", "http://www.tizen.org", "http://tizen.org/", "http://tizen.org/id", "http://myapp", "*"};
        for (String origin : validedOrigins) {
            assertTrue(ValidationUtil.checkForAccessOrigin(origin));
        }

        final String[] invalidedOrigins = {"http:/tizen.org", "tizen.org", "www.tizen.org", "http://tizen.org/!", "http://tizen.org/id^", "http://tizen.org/id*", "*.*"};
        for (String origin : invalidedOrigins) {
            assertFalse(ValidationUtil.checkForAccessOrigin(origin));
        }
    }

    /**
     * Test {@link ValidationUtil#checkForPrivilege(String value)}
     *
     * @throws Exception in case of failure in test
     *
     * @see {@link ValidationUtil#checkForPrivilege(String value)}
     */
    @Test
    public void test_checkForPrivilege() throws Exception {
        final String[] validedPrivileges = {
                "",
                "http://tizen.org/privilege/account.read",
                "http://tizen.org/privilege/packagemanager.install",
                "http://tizen.org/privilege/unlimitedstorage",
                ".-_"
        };
        for ( String privilege : validedPrivileges ) {
            assertTrue( "Failed for \"" + privilege + "\"", ValidationUtil.checkForPrivilege( privilege ) );
        }

        final String[] invalidedPrivileges = {
                "http://tizen.org/privilege/account.read~",
                "http://tizen.org/privilege/packagemanage!r.install",
                "http://tizen.org/privilege/unlimitedstor@age",
                "!",
                "@",
                "#",
                "$",
                "%",
                "^",
                "&",
                "*",
                "(",
                ")",
                "+",
                "`",
                "~",
                "|",
                "\\",
                "?",
                "{",
                "}",
                "[",
                "]"
        };
        for ( String privilege : invalidedPrivileges ) {
            assertFalse( "Failed for \"" + privilege + "\"", ValidationUtil.checkForPrivilege( privilege ) );
        }
    }
}
