/*
 * Common
 *
 * Copyright (c) 2000 - 2011 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Contact: 
 * BonYong Lee <bonyong.lee@samsung.com>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Contributors:
 * - S-Core Co., Ltd
 *
 */
package org.tizen.common;

import static java.lang.System.currentTimeMillis;
import static org.tizen.common.util.StringUtil.nvl;
import static org.tizen.sdblib.util.ThreadUtil.trySleep;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.OutputStreamWriter;
import java.io.StringReader;
import java.lang.annotation.Annotation;
import java.lang.reflect.Array;
import java.lang.reflect.Field;
import java.text.MessageFormat;
import java.text.ParseException;

import org.tizen.sdblib.daemon.ServerState;
import org.tizen.sdblib.util.Log;
import org.tizen.sdblib.util.StreamGobbler;

/**
 * CommonPluginTest
 *
 * Test case for {@link CommonPlugin}
 * 
 * @author BonYong Lee{@literal <bonyong.lee@samsung.com>} (S-Core)
 * 
 * @see CommonPlugin
 */
public class
Shell
implements IShell
{
    /**
     * Timeout for execution
     */
    protected static final long TIMEOUT = 100000;   // 100 sec
    
    /**
     * Starting entry method
     * 
     * @param cmd command to execute
     * @return {@link IShell}
     * 
     * @throw Exception If execution fail
     */
    public static
    IShell
    run(
        final String cmd
    )
    throws Exception
    {
        return new Shell( cmd );
    }
    
    /**
     * Starting entry method and wait until process is terminated
     * 
     * @param cmd command to execute
     * 
     * @throw Exception If execution fail
     */
    public static
    void
    runAndWait(
        final String cmd
    )
    throws Exception
    {
        Shell shell = new Shell( cmd );
        shell.process.waitFor();
    }

    /**
     * process
     */
    final Process process;
    
    /**
     * execution starting time
     */
    final long startTime;
    
    /**
     * Input stream dispenser
     */
    final StreamGobbler in;
    
    /**
     * Error stream dispenser
     */
    final StreamGobbler error;
    
    /**
     * Writer for writing data to the process
     */
    final BufferedWriter out;
    
    /**
     * Checker for waitFor is done
     */
    boolean waitForFin;
    
    /**
     * Constructor with command and boot the input and error stream
     * 
     * @param cmd command to execution
     * 
     * @throws Exception If execution fail
     */
    public Shell(
        final String cmd
    )
    throws Exception
    {
        process = Runtime.getRuntime().exec( cmd );
        startTime = System.currentTimeMillis();
        in = new StreamGobbler( process.getInputStream() );
        error = new StreamGobbler( process.getErrorStream() );
        out = new BufferedWriter(new OutputStreamWriter(process.getOutputStream()));
        waitForFin = false;
        
        in.boot();
        error.boot();
        
        new Thread(new Runnable() {
            
            @Override
            public void run() {
                try {
                    process.waitFor();
                    waitForFin = true;
                } catch (InterruptedException e) {
                    Log.e("Exception occurred", e);
                }
            }
        }).start();
    }
    
    /** (non-Javadoc)
     * @see org.tizen.common.IShell#write(String, long)
     */
    public
    void
    write(
       final String data,
       final long delay
    )
    throws Exception
    {
        if(delay > 0) {
            Thread.sleep(delay);
        }
        out.append(data);
        out.newLine();
        out.flush();
    }
    
    /** (non-Javadoc)
     * @see org.tizen.common.IShell#expect(Object, long)
     */
    public <T>
    T
    expect(
        final T instance,
        long delay
    )
    throws Exception
    {
        trySleep(delay);
        process.destroy();
        final String result = in.getResult();
        
        return convert(instance, result);
    }
    
    /** (non-Javadoc)
     * @see org.tizen.common.IShell#expect(java.lang.Object)
     */
    public <T>
    T
    expect(
        final T instance
    )
    throws Exception
    {
        return expect(instance, false);
    }
    
    /** (non-Javadoc)
     * @see org.tizen.common.IShell#expect(java.lang.Object)
     */
    public <T>
    T
    expect(
        final T instance,
        boolean standardError
    )
    throws Exception
    {
        // Check done
        while ( TIMEOUT > currentTimeMillis() - startTime && ( !in.isState( ServerState.Terminated ) ) && !waitForFin)
        {
            trySleep( 200 );
        }

        if( !in.isState(ServerState.Terminated) && !waitForFin) {
            return instance;
        }

        // wait for result buffer being fully filled.
        while(!in.isState(ServerState.Terminated ))
        {
            trySleep (200);
        }

        String result = "";
        if(standardError) {
            result = error.getResult();
        }
        else {
            result = in.getResult();
        }
        return convert(instance, result);
    }
    
    private <T>
    T
    convert (
        final T instance, 
        String result
    ) 
    throws Exception {
        final BufferedReader reader = new BufferedReader( new StringReader( result ) );
        
        String line = null;

        final Class<?> clazz = instance.getClass();
        while ( null != ( line = reader.readLine() ) )
        {
            final Field[] fields = clazz.getDeclaredFields();
            for ( final Field f : fields )
            {
                Annotation annotation = f.getAnnotation( Token.class );
                if ( null != annotation )
                {
                    addValue( instance, f, process( line, (Token) annotation ) );
                }
                
                annotation = f.getAnnotation( Pattern.class );
                if ( null != annotation )
                {
                    addValue( instance, f, process( line, (Pattern) annotation ) );
                }
                
            }
        }
        
        return instance;
    }
    
    
    /**
     * Set or add value to instance's field
     * 
     * @param instance object instance
     * @param field object field
     * @param value value to set
     * 
     * @throws IllegalArgumentException
     * @throws IllegalAccessException
     */
    protected
    void
    addValue(
        final Object instance,
        final Field field,
        final String value
    )
    throws IllegalArgumentException, IllegalAccessException
    {
        if ( null == value )
        {
            return ;
        }
        field.setAccessible( true );
        if ( field.getType().isArray() )
        {
            Object old = field.get( instance );
            final int length = (null==old)?0:Array.getLength( old );
            Object array = Array.newInstance( field.getType().getComponentType(), length + 1 );
            if ( null != old )
            {
                System.arraycopy( old, 0, array, 0, length );
            }
            Array.set( array, length, value );
            field.set(instance, array);
        }
        else
        {
            field.set( instance, value );
        }
    }
    
    /**
     * Process line with Token annotation
     * 
     * @param line string to parse
     * @param token token annotation
     * 
     * @return result to be parsed
     */
    protected
    String
    process(
        final String line,
        final Token token
    )
    {
        final String from = token.from();
        final String to = token.to();

        final int fromIndex = line.indexOf( nvl( from ) );
        final int toIndex = line.lastIndexOf( nvl( to ) );

        if ( fromIndex < 0 || toIndex < 0 )
        {
            return null;
        }
        
        String substring = line.substring( fromIndex + from.length(), toIndex );
        if ( token.trim() )
        {
            substring = substring.trim();
        }
        
        return substring;
    }

    /**
     * @param line string to parse
     * @param pattern pattern annotation
     * 
     * @return result to be parsed
     */
    protected
    String
    process(
        final String line,
        final Pattern pattern
    )
    {
        try
        {
            final Object[] args = new MessageFormat( pattern.pattern() ).parse( line );
            if ( 0 <= pattern.index() && pattern.index() < args.length )
            {
                final String ret = (String) args[ pattern.index() ];
                return (pattern.trim())?(ret.trim()):(ret);
            }
            return null;
        }
        catch ( final ParseException e)
        {
            return null;
        }
    }

}
