/*
 * 
 *
 * Copyright (c) 2000 - 2011 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Contact: 
 * BonYong Lee <bonyong.lee@samsung.com>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Contributors:
 * - S-Core Co., Ltd
 *
 */
package org.tizen.common;

import java.util.ArrayList;
import java.util.List;

import org.junit.internal.builders.JUnit4Builder;
import org.junit.runner.Description;
import org.junit.runner.Runner;
import org.junit.runner.notification.RunNotifier;
import org.junit.runners.ParentRunner;
import org.junit.runners.model.InitializationError;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.tizen.common.classloader.ClassSource;
import org.tizen.common.classloader.DirectoryClassSource;
import org.tizen.common.classloader.JarClassSource;

/**
 * <p>
 * UnsafeRunner.
 * 
 * JUnit extensible {@link Runner}
 * </p>
 * 
 * @author BonYong Lee{@literal <bonyong.lee@samsung.com>} (S-Core)
 */
public class
UnsafeRunner
extends ParentRunner<Runner>
{
	/**
	 * Logger for this instance
	 */
	protected final Logger logger = LoggerFactory.getLogger( getClass() );

	/**
	 * Classloader to convert test case
	 */
	protected ClassLoader loader;

	/**
	 * Test case name
	 */
	protected final String className;

	/**
	 * Constructor with test case {@link Class}
	 * 
	 * @param testClass test case {@link Class}
	 * 
	 * @throws InitializationError If <code>testClass</code> is invalid test case
	 */
	public
	UnsafeRunner(
		final Class<?> testClass
	)
	throws InitializationError
	{
		super( testClass );
		logger.trace( "Class :{}", testClass );

		this.className = testClass.getName();

		ClassSource[] sources = getSources( testClass );

		loader = new UnsafeClassLoader( getClass().getClassLoader(), sources );
	}

	/**
	 * Extrace {@link ClassSource} from test case's annotation
	 * 
	 * @param testClass test case
	 * 
	 * @return {@link ClassSource}s
	 */
	protected
	ClassSource[] getSources(
		final Class<?> testClass
	)
	{
		final Location annotation =
			testClass.getAnnotation( Location.class );
		if ( null == annotation )
		{
			return new ClassSource[0];
		}

		final String[] locations = annotation.value();

		final ArrayList<ClassSource> sources = new ArrayList<ClassSource>();

		for ( final String location : locations )
		{
			if ( location.endsWith( ".jar" ) )
			{
				sources.add( new JarClassSource( location ) );
			}
			else if ( location.endsWith( "/" ) )
			{
				sources.add( new DirectoryClassSource( location ) );
			}
		}

		return sources.toArray( new ClassSource[0] );
	}

	/* (non-Javadoc)
	 * @see org.junit.runners.ParentRunner#getChildren()
	 */
	@Override
	protected
	List<Runner>
	getChildren()
	{
		final ArrayList<Runner> runners = new ArrayList<Runner>();
		try {
			final Class<?> clazz = loader.loadClass( className );
			logger.trace( "Class :{}, Loader :{}", clazz, clazz.getClassLoader() );
			final JUnit4Builder builder = new JUnit4Builder();
			runners.add( builder.runnerForClass( clazz ) );
		} catch (ClassNotFoundException e) {
			e.printStackTrace();
		} catch (Throwable e) {
			e.printStackTrace();
		}

		logger.debug( "Runners :{}", runners );

		return runners;
	}

	/* (non-Javadoc)
	 * @see org.junit.runners.ParentRunner#describeChild(java.lang.Object)
	 */
	@Override
	protected
	Description
	describeChild(
		final Runner runner
	)
	{
		return runner.getDescription();
	}

	/* (non-Javadoc)
	 * @see org.junit.runners.ParentRunner#runChild(java.lang.Object, org.junit.runner.notification.RunNotifier)
	 */
	@Override
	protected
	void
	runChild(
		final Runner runner,
		final RunNotifier notifier
	)
	{
		logger.trace( "Runner :{}", runner );
		runner.run( notifier );
	}
}
