/*
 * Common
 *
 * Copyright (c) 2000 - 2011 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Contact:
 * Ho Namkoong <ho.namkoong@samsung.com> 
 * BonYong Lee <bonyong.lee@samsung.com>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Contributors:
 * - S-Core Co., Ltd
 *
 */

package org.tizen.common.sdb;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.*;
import static org.tizen.common.Shell.*;
import static org.tizen.common.sdb.SdbTestUtil.*;

import java.text.MessageFormat;

import org.junit.Test;
import org.tizen.common.Pattern;

/**
 * <p>
 * ConnectTest
 * 
 * Test case for sdb connect, disconnect
 *
 * </p>
 * 
 * @author Ho Namkoong{@literal <ho.namkoong@samsung.com>} (S-Core)
 */
public class ConnectTest {
    
    //connect and disconnect works async
    public static final int CONNECT_SLEEP = 1000;
    
    /**
     * Test {sdb connect}
     * 
     * @author Ho Namkoong{@literal <ho.namkoong@samsung.com>} (S-Core)
     * 
     * command
     * 1. sdb disconnect
     * 2. sdb connect 127.0.0.1:{port + 1}
     * 3. sdb devices
     * 4. sdb disconnect 127.0.0.1:{port + 1}
     * expected result
     * 127.0.0.1:{port + 1}   device  {device name}
     */
    @Test
    public void test_connect() throws Exception {
        
        class ConnectResult {
            @Pattern( pattern = LOOPBACK_IP + ":{0}\t{1}", index = 0 ) public String[] tcpPortNums;
        }
        
        try {
            int port = getPorts()[0];
            runAndWait(SDB_PATH + " disconnect");
            Thread.sleep(CONNECT_SLEEP);
            runAndWait( SDB_PATH + " connect " + LOOPBACK_IP + ":" + port);
            final ConnectResult result = run( SDB_PATH + " devices").expect( new ConnectResult() );
            assertNotNull("tcp port number of sdb devices for connect is null", result.tcpPortNums);
            int tcpPort = new Integer(result.tcpPortNums[0]);
            assertTrue("More than one devices are connected", result.tcpPortNums.length == 1);
            assertEquals(MessageFormat.format("Device port: \'{0}\' and connected port: \'{1}\' are not matched", port, result.tcpPortNums[0]), 
                    port, tcpPort);
        }
        finally {
            runAndWait( SDB_PATH + " disconnect");
            Thread.sleep(CONNECT_SLEEP);
        }
    }
    
    /**
     * Test {sdb connect}
     * 
     * @author Ho Namkoong{@literal <ho.namkoong@samsung.com>} (S-Core)
     * 
     * command
     * 1. sdb connect 127.0.0.1:{port + 1}
     * 2. sdb disconnect 127.0.0.1:{port + 1}
     * 3. sdb devices
     * expected result
     * there is no string like: 127.0.0.1:{port + 1}   device
     */
    @Test
    public void test_disconnect() throws Exception {
        
        try {
            runAndWait( SDB_PATH + " disconnect");
            Thread.sleep(CONNECT_SLEEP);
            int ports[] = getPorts();
            runAndWait( SDB_PATH + " connect " + LOOPBACK_IP + ":" + ports[0]);
            Thread.sleep(CONNECT_SLEEP);
            runAndWait( SDB_PATH + " disconnect " + LOOPBACK_IP + ":" + ports[0]);
            Thread.sleep(CONNECT_SLEEP);
            String[] results = SdbTestUtil.runAndGetWholeString(SDB_PATH + " devices", true);
            for(String result: results) {
                assertFalse(MessageFormat.format("fail to disconnect {0}", result), result.contains(LOOPBACK_IP));
            }
        }
        finally {
            runAndWait( SDB_PATH + " disconnect");
            Thread.sleep(CONNECT_SLEEP);
        }
    }
    
}
