/*
*  Common
*
* Copyright (c) 2000 - 2011 Samsung Electronics Co., Ltd. All rights reserved.
*
* Contact: 
* BonYong Lee <bonyong.lee@samsung.com>
* Gun Kim <gune.kim@samsung.com>
* 
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
* http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
* Contributors:
* - S-Core Co., Ltd
*
*/
package org.tizen.common.util;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;
import static org.tizen.common.util.ArrayUtil.safe;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.ConcurrentModificationException;
import java.util.Iterator;

import org.junit.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * ArrayUtilsTest
 *
 * Test case for {@link ArrayUtil}
 * 
 * @author BonYong Lee{@literal <bonyong.lee@samsung.com>} (S-Core)
 * 
 * @see ArrayUtil
 */
public class ArrayUtilTest
{
    protected Logger logger = LoggerFactory.getLogger( getClass() );

    public static int arrayIndex = 0;
    public static boolean successTest = true;

    /**
     * Test {@link ArrayUtil#pickupFirst(Collection)}
     * 
     * @throws Exception in case of failure in test
     * 
     * @see ArrayUtil#pickupFirst(Collection)
     */
    @Test
    public void
    test_pickupFirst()
            throws Exception
            {
        final Object[][] TEST_CASES = new Object[][] {
                new Object[] { null, null },
                new Object[] { new Object[] { null }, null },
                new Object[] { new Object[] { "hello", null }, "hello" },
                new Object[] { new Object[] { null, "hello" }, null },
                new Object[] { new Object[] { "A", "B" }, "A" },
        };

        int iTestCase = 0;
        for ( final Object[] TEST_CASE : TEST_CASES )
        {
            ++iTestCase;
            final Object[] input = (Object[]) TEST_CASE[0];
            final Object expected = (Object) TEST_CASE[1];

            assertEquals( iTestCase + " th test case", expected, ArrayUtil.pickupFirst( input ) );
        }
            }

    /**
     * Test {@link ArrayUtil#pickupLast(Collection)}
     * 
     * @throws Exception in case of failure in test
     * 
     * @see ArrayUtil#pickupLast(Collection)
     */
    @Test
    public void
    test_pickupLast()
            throws Exception
            {
        final Object[][] TEST_CASES = new Object[][] {
                new Object[] { null, null },
                new Object[] { new Object[] { null }, null },
                new Object[] { new Object[] { "hello", null }, null },
                new Object[] { new Object[] { null, "hello" }, "hello" },
                new Object[] { new Object[] { "A", "B" }, "B" },
        };

        int iTestCase = 0;
        for ( final Object[] TEST_CASE : TEST_CASES )
        {
            ++iTestCase;
            final Object[] input = (Object[]) TEST_CASE[0];
            final Object expected = (Object) TEST_CASE[1];

            assertEquals( iTestCase + " th test case", expected, ArrayUtil.pickupLast( input ) );
        }
            }


    /**
     * Test {@link ArrayUtil#iterator(Object[])}
     * 
     * @throws Exception in case of failure in test
     * 
     * @see ArrayUtil#iterator(Object[])
     */
    @Test
    public void test_iterator() throws Exception
    {
        final Object[][] TEST_CASES = new Object[][] {
                new Object[] { null },
                new Object[] { "hello" },
                new Object[] { null, "hello" },
                new Object[] { "A", "B" },
        };

        int iTestCase = 0;
        for ( final Object[] TEST_CASE : TEST_CASES )
        {
            ++iTestCase;
            final Object[] array = TEST_CASE;
            final ArrayList<Object> list = new ArrayList<Object>( Arrays.asList( array ) );
            final Iterator<Object> iter = list.iterator();
            final Iterator<Object> testTarget = ArrayUtil.iterator( array );

            int iElement = 0;
            while ( iter.hasNext() && testTarget.hasNext() )
            {
                assertEquals( iTestCase + " th test case :" + (++iElement) + " th element", iter.next(), testTarget.next() );
            }
            assertEquals( iTestCase + "th test case's last assert :", iter.hasNext(), testTarget.hasNext() );
        }

        try
        {
            final Object[] array = new Object[] { "a", "b", "c" };
            final Iterator<Object> iter = ArrayUtil.iterator( array );
            assertTrue( iter.hasNext() );
            assertEquals( array[0], iter.next() );

            array[1] = "d";

            assertTrue( iter.hasNext() );
            iter.next();
            fail( "ConcurrentModificatoinException Test fail" );

        } catch ( ConcurrentModificationException e )
        {
            // Test Success
            LoggerFactory.getLogger( getClass() ).info( "Test case successfult" );
        }
    }

    /**
     * Test {@link ArrayUtil#iterate(Object[], IteratingRunner, boolean)}
     * 
     * @throws Exception in case of failure in test
     * 
     * @see {@link ArrayUtil#iterate(Object[], IteratingRunner, boolean)}
     */
    @Test
    public void test_iterator_with_iteratingRunner() {
        final Object[][] TEST_CASES = new Object[][] {
                null,
                new Object[] { null },
                new Object[] { "hello" },
                new Object[] { null, "hello" },
                new Object[] { "A", "B" },
        };
        for ( final Object[] TEST_CASE : TEST_CASES )
        {
            final Object[] array = TEST_CASE;

            try {
                ArrayUtil.iterate(array, null, false);
            } catch (InvocationTargetException e1) {
                fail("Failed iterate(Collection, null, false) test");
            }

            arrayIndex = 0;
            successTest = true;
            try {
                ArrayUtil.iterate(array,
                        new IteratingRunner<Object>() {
                    @Override
                    public void run(Object arg) {
                        if ( successTest && arg != array[arrayIndex++] ) {
                            successTest = false;
                        }
                    }
                },
                true);
                if ( !successTest ) {
                    fail("Failed iterate(Collection, IteratingRunner, true) test");
                }
            } catch (InvocationTargetException e) {
                fail("Failed iterate(Collection, IteratingRunner, true) test");
            }

            arrayIndex = 0;
            successTest = true;
            try {
                ArrayUtil.iterate(array,
                        new IteratingRunner<Object>() {
                    @Override
                    public void run(Object arg) {
                        if ( arg == null ) {
                            successTest = false;
                            return;
                        }
                        if ( successTest ) {
                            for ( int i = arrayIndex ; i < array.length ; i++ ) {
                                if( array[i] == null ) {
                                    continue;
                                }
                                if ( arg != array[i] ) {
                                    successTest = false;
                                    return;
                                }
                                else {
                                    arrayIndex = ++i;
                                }
                            }
                        }
                    }
                },
                false);
                if ( !successTest ) {
                    fail("Failed iterate(Collection, IteratingRunner, false) test");
                }
            } catch (InvocationTargetException e) {
                fail("Failed iterate(Collection, IteratingRunner, false) test");
            }
        }
    }


    /**
     * Test {@link ArrayUtil#convertToWrapper(char[])}
     * 
     * @throws Exception in case of failure in test
     * 
     * @see ArrayUtil#convertToWrapper(char[])
     */
    @Test
    public void test_convertToWrapper() throws Exception
    {
        final char[] input = "hello".toCharArray();
        final Character[] ret = ArrayUtil.convertToWrapper( input );

        assertEquals( input.length, ret.length );
    }

    /**
     * Test {@link ArrayUtil#filter(Object[], IteratingAcceptor, boolean)}
     * 
     * @throws Exception in case of failure in test
     * 
     * @see ArrayUtil#filter(Object[], IteratingAcceptor, boolean)
     */
    @Test
    public void test_filter() throws Exception
    {
        final Character[] input = ArrayUtil.convertToWrapper( "hello".toCharArray() );

        assertTrue( Arrays.equals(
                new Character[] { 'l', 'l' },
                ArrayUtil.filter( input, new IteratingAcceptor<Character>() {
                        public boolean accept( Character arg)
                        {
                            return ( 'l' == arg );
                        }
                }, true )
        ) );
    }

    @Test
    public void test_toObjectArray() {
        final Object[][] inputArray = new Object[][]{
                null,
                new String[]{"hello", "test", "java"},
                new Integer[]{10, 20, 30},
                {10,20,30},
                {true, false, true},
                {true, 10, false},
                new Object[]{"hello", 10, 5},
                new Object[]{5, "java", 5},
                new Object[]{null, "hello", 10},
                new Object[]{null, null, null},
                new Object[]{ }
        };

        final int inputInt[] = {10,20,30};
        final boolean inputBool[] = {true, false, false};

        Object[] actualData = null;

        for ( Object[] input : inputArray ) {
            actualData = ArrayUtil.toObjectArray(input);
            if ( (input == null || input.length == 0) ) {
                if ( actualData.length != 0 ) {
                    fail("Failed toObjectArray(object) test");
                }
            }
            else {
                if ( input != actualData) {
                    fail("Failed toObjectArray(object) test");
                }
            }
        }

        actualData = ArrayUtil.toObjectArray(inputInt);
        if ( !(actualData instanceof Integer[]) ) {
            fail("Failed toObjectArray(int[]) test");
        }
        actualData = ArrayUtil.toObjectArray(inputBool);
        if ( !(actualData instanceof Boolean[]) ) {
            fail("Failed toObjectArray(boolean[]) test");
        }
    }

    @Test
    public void test_size_primitive(){
        final boolean[][] inputBool = {
                null,
                {false},
                {true, true},
        };
        final byte[][] inputByte = {
                null,
                {0x00},
                {0x1a, 0x10},
        };
        final char[][] inputChar = {
                null,
                {'a'},
                {'a', 'b'}
        };
        final short[][] inputShort = {
                null,
                {1},
                {10, 20}
        };
        final int[][] inputInt = {
                null,
                {100},
                {200, 300}
        };
        final long[][] inputLong = {
                null,
                {100l},
                {200l, 300l}
        };
        final float[][] inputFloat = {
                null,
                {100f},
                {200f, 300f}
        };
        final double[][] inputDouble = {
                null,
                {100d},
                {200d, 300d}
        };
        int size = 0;

        for ( boolean[] input : inputBool ) {
            size = ArrayUtil.size(input);
            if ( input == null ) {
                if ( size != 0 ) {
                    fail("Failed size(boolean[]) test");
                }
            }
            else {
                assertEquals(input.length, ArrayUtil.size(input));
            }
        }

        for ( byte[] input : inputByte ) {
            size = ArrayUtil.size(input);
            if ( input == null ) {
                if ( size != 0 ) {
                    fail("Failed size(byte[]) test");
                }
            }
            else {
                assertEquals(input.length, ArrayUtil.size(input));
            }
        }

        for ( char[] input : inputChar ) {
            size = ArrayUtil.size(input);
            if ( input == null ) {
                if ( size != 0 ) {
                    fail("Failed size(char[]) test");
                }
            }
            else {
                assertEquals(input.length, ArrayUtil.size(input));
            }
        }

        for ( short[] input : inputShort ) {
            size = ArrayUtil.size(input);
            if ( input == null ) {
                if ( size != 0 ) {
                    fail("Failed size(short[]) test");
                }
            }
            else {
                assertEquals(input.length, ArrayUtil.size(input));
            }
        }

        for ( int[] input : inputInt ) {
            size = ArrayUtil.size(input);
            if ( input == null ) {
                if ( size != 0 ) {
                    fail("Failed size(int[]) test");
                }
            }
            else {
                assertEquals(input.length, ArrayUtil.size(input));
            }
        }

        for ( long[] input : inputLong ) {
            size = ArrayUtil.size(input);
            if ( input == null ) {
                if ( size != 0 ) {
                    fail("Failed size(long[]) test");
                }
            }
            else {
                assertEquals(input.length, ArrayUtil.size(input));
            }
        }

        for ( float[] input : inputFloat ) {
            size = ArrayUtil.size(input);
            if ( input == null ) {
                if ( size != 0 ) {
                    fail("Failed size(float[]) test");
                }
            }
            else {
                assertEquals(input.length, ArrayUtil.size(input));
            }
        }

        for ( double[] input : inputDouble ) {
            size = ArrayUtil.size(input);
            if ( input == null ) {
                if ( size != 0 ) {
                    fail("Failed size(double[]) test");
                }
            }
            else {
                assertEquals(input.length, ArrayUtil.size(input));
            }
        }
    }

    @Test
    public void test_size() {
        final Object[][] inputArray = new Object[][]{
                null,
                new String[]{"hello", "test", "java"},
                new Integer[]{10, 20, 30},
                {10,20,30},
                {true, false, true},
                {true, 10, false},
                new Object[]{"hello", 10, 5},
                new Object[]{5, "java", 5},
                new Object[]{null, "hello", 10},
                new Object[]{null, null, null},
                new Object[]{ }
        };
        final int inputInt[] = {10,20,30};
        final boolean inputBool[] = {true, false, false};

        int size = 0;
        int size2 = 0;
        for ( Object[] input : inputArray ) {
            size = ArrayUtil.size(input);
            if ( input == null || input.length == 0 ) {
                if (size != 0) {
                    fail("Failed size(T[]) test");
                }
            }
            else {
                assertEquals(input.length, size);
            }

            size2 = ArrayUtil.size((Object)input);
            if ( input == null || input.length == 0 ) {
                if (size2 != 0) {
                    fail("Failed size(T[]) test");
                }
            }
            else {
                assertEquals(input.length, size2);
            }
        }

        assertEquals("Failed size(Object) test", inputInt.length, ArrayUtil.size((Object)inputInt));
        assertEquals("Failed size(Object) test", inputInt.length, ArrayUtil.size((Object)inputBool));
    }

    @Test
    public void test_add() {
        final Object[][] inputArray = new Object[][]{
                null,
                new String[]{"hello", "test", "java"},
                new Integer[]{10, 20, 30},
                {10,20,30},
                {true, false, true},
                {true, 10, false},
                new Object[]{"hello", 10, 5},
                new Object[]{5, "java", 5},
                new Object[]{null, "hello", 10},
                new Object[]{null, null, null},
                new Object[]{ }
        };
        final Object[] inputAddArg = new Object[]{
                null,
                "hello",
                10,
                true
        };

        for ( Object[] input : inputArray ) {
            for ( Object arg : inputAddArg ) {
                Object[] actualData = null;
                try {
                    actualData = ArrayUtil.add(input, arg);
                } catch (ArrayStoreException e) {
                    if ( input == null ) {
                        fail("Failed add(T[], Object) test");
                    }
                    if ( arg.getClass().getName().equals(input[0].getClass().getName()) ) {
                        fail("Failed add(T[], Object) test");
                    }
                    continue;
                }
                if ( input == null || input.length == 0 ) {
                    assertEquals(1, actualData.length);
                    assertEquals(arg, actualData[0]);
                }
                else {
                    assertEquals(input.length + 1 , actualData.length);
                    assertEquals(arg, actualData[input.length]);
                }
            }
        }
    }

    @Test
    public void test_remove() {

        assertArrayEquals( new String[] { "a", "b", "c" }, ArrayUtil.remove( new String[] { "a", "b", "c" }, -1 ) );
        assertArrayEquals( new String[] { "b", "c" }, ArrayUtil.remove( new String[] { "a", "b", "c" }, 0 ) );
        assertArrayEquals( new String[] { "a", "c" }, ArrayUtil.remove( new String[] { "a", "b", "c" }, 1 ) );
        assertArrayEquals( new String[] { "a", "b" }, ArrayUtil.remove( new String[] { "a", "b", "c" }, 2 ) );
        assertArrayEquals( new String[] { "a", "b", "c" }, ArrayUtil.remove( new String[] { "a", "b", "c" }, 3 ) );

        assertArrayEquals( new String[] { "a", "b", "c" }, ArrayUtil.remove( new String[] { "a", "b", "c" }, -1, 0 ) );
        assertArrayEquals( new String[] { "b", "c" }, ArrayUtil.remove( new String[] { "a", "b", "c" }, 0, 1 ) );
        assertArrayEquals( new String[] { "a", "c" }, ArrayUtil.remove( new String[] { "a", "b", "c" }, 1, 2 ) );
        assertArrayEquals( new String[] { "a", "b" }, ArrayUtil.remove( new String[] { "a", "b", "c" }, 2, 3 ) );
        assertArrayEquals( new String[] { "a", "b", "c" }, ArrayUtil.remove( new String[] { "a", "b", "c" }, 3, 4 ) );

        assertArrayEquals( new String[] { "c" }, ArrayUtil.remove( new String[] { "a", "b", "c" }, -1, 2 ) );

        final Object[][] inputArray = new Object[][]{
                null,
                new String[]{"hello", "test", "java", "c", "c++", "junit", "time"},
                new Integer[]{10, 20, 30, 40, 50, 60, 70, 80, 90},
                {10, 20, 30, 40, 50, 60, 70, 80, 90, 100},
                {true, false, true, true, false, true, false},
                {true, 10, false, 20, "test", "java", 30},
                new Object[]{"hello", 10, 5, 20, 30, true},
                new Object[]{5, "java", 5},
                new Object[]{null, "hello", 10},
                new Object[]{null, null, null},
                new Object[]{ }

        };
        
        try
        {
            ArrayUtil.remove( null, 0, 1 );
            fail();
        }
        catch( final IllegalArgumentException e )
        {
            logger.info( "Test case successfult" );
        }

        for ( Object[] input : inputArray ) {
            for ( int i = -1 ; i < 10 ; i++ ) {
                for( int j = -1 ; j < 10 ; j++ ) {
                    int deleteCount = 0;
                    Object[] actualData = null;
                    try {
                        logger.trace( "{}: {}, {}", new Object[] { ObjectUtil.toString( input ), i, j } );
                        actualData = ArrayUtil.remove( input, i, j);
                        i = Math.max( 0, i );
                        if( j <= i) {
                            assertArrayEquals(input , actualData);
                        }
                        else {
                            deleteCount = j - i;
                            for ( int k = 0 ; k < actualData.length ; k++ ) {
                                if ( k >= i ) {
                                    assertEquals(
                                            "K:" + k + ", J:" + j + ", I:" +i + ", Input :" + input[k + deleteCount] + ", Actual :" + actualData[k],
                                            input[k+deleteCount], actualData[k]);
                                }
                                else {
                                    assertEquals(
                                            "K:" + k + ", J:" + j + ", I:" +i + ", Input :" + input[k] + ", Actual :" + actualData[k],
                                            input[k], actualData[k]);
                                }

                            }
                        }
                    }
                    catch ( ArrayIndexOutOfBoundsException e ) {
                        if ( j >= input.length || i >= input.length ) {
                            continue;
                        }
                        else {
                            fail("Failed remove(Object[], i : " + i + " , j : " + j + ") test");
                        }
                    }
                    catch ( IllegalArgumentException e ) {
                        if ( null == input || i >= j ) {
                            continue;
                        }
                        else {
                            e.printStackTrace();
                            fail("Failed remove(Object[], int, int) test");
                        }
                    }
                    catch (NegativeArraySizeException e) {
                        if ( input.length < j - i ) {
                            continue;
                        }
                        else {
                            fail("Failed remove(Object[], int, int) test");
                        }
                    }
                }
            }
        }
    }

    @Test
    public void test_contains() {
        final Object[][] inputArray = new Object[][]{
                null,
                new String[]{"hello", "test", "java"},
                new Integer[]{10, 20, 30},
                {10,20,30},
                {true, false, true},
                {true, 10, false},
                new Object[]{"hello", 10, 5},
                new Object[]{5, "java", 5},
                new Object[]{null, "hello", 10},
                new Object[]{null, null, null},
                new Object[]{ }
        };
        final Object[] inputElement = new Object[]{
                10583,
                "zxcvb",
        };

        boolean actualData = true;

        for ( Object[] input : inputArray ) {
            if ( input == null ) {
                actualData = ArrayUtil.contains(input, "blah");
                assertEquals( false, actualData );
            }
            else {
                for ( Object element : input) {
                    actualData = ArrayUtil.contains(input, element);
                    assertEquals( true, actualData );
                }
            }
            for ( Object element : inputElement ) {
                actualData = ArrayUtil.contains(input, element);
                assertEquals( false, actualData );
            }
        }
    }

    @Test
    public void test_get() {
        final Object[][] inputArray = new Object[][]{
                null,
                new String[]{"hello", "test", "java"},
                new Integer[]{10, 20, 30},
                {10,20,30},
                {true, false, true},
                {true, 10, false},
                new Object[]{"hello", 10, 5},
                new Object[]{5, "java", 5},
                new Object[]{null, "hello", 10},
                new Object[]{null, null, null},
                new Object[]{ }
        };

        for( Object[] input : inputArray) {
            if ( input == null || input.length == 0 ) {
                assertEquals( null, ArrayUtil.get(input, -1) );
                assertEquals( null, ArrayUtil.get(input, 0) );
            }
            else {
                assertEquals(null, ArrayUtil.get(input, -1) );
                assertEquals( input[0], ArrayUtil.get(input, 0) );
                assertEquals(null, ArrayUtil.get(input, input.length + 1) );
            }
        }
    }
    
    public
    void
    test_safe()
    throws Exception
    {
        final String[][] TEST_CASES = new String[][] {
            null,
            new String[] { null, "aaa" },
            new String[] { "hello", "world" },
        };
        
        for ( final String[] TEST_CASE : TEST_CASES )
        {
            assertNotNull( safe( TEST_CASE ) );
            for ( final String value : safe( TEST_CASE ) )
            {
                assertTrue( "value :" + value, true );
            }
        }
        
    }
}
