/*
*  Common
*
* Copyright (c) 2000 - 2011 Samsung Electronics Co., Ltd. All rights reserved.
*
* Contact: 
* Jihoon Song <jihoon80.song@samsung.com>
* 
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
* http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
* Contributors:
* - S-Core Co., Ltd
*
*/

package org.tizen.common.util;

import java.net.URI;
import java.net.URL;

import org.junit.*;
import org.tizen.common.Surrogate;
import org.tizen.common.util.EFSUtil.DialogChoicer;

import org.eclipse.core.filesystem.EFS;
import org.eclipse.core.filesystem.IFileInfo;
import org.eclipse.core.filesystem.IFileStore;
import org.eclipse.core.filesystem.IFileSystem;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.swt.widgets.Shell;

import static org.junit.Assert.*;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import static org.mockito.Mockito.verify;

/**
 * EFSUtilTest.
 * 
 * Helper related to {@link EFSUtil}
 * 
 * @author Jihoon Song{@literal <jihoon80.song@samsung.com>} (S-Core)
 * 
 * @see EFSUtil
 * 
 */
public class EFSUtilTest {
    
    /**
     * Property key for URL handler
     */
    protected static final String PROP_PKG = "java.protocol.handler.pkgs";
    
    protected static final String TEST_PATH = "test";
    protected static final String TEST_URL = "cp:///" + TEST_PATH;
    
    /**
     * Old property value for URL handler
     */
    protected String oldConfig;
    
    
    /**
     * Set up URL handler and FileSystem before test
     */
    @Before
    public void setUp() {
        oldConfig = System.getProperty( PROP_PKG );
        System.setProperty( PROP_PKG, "org.tizen.common.util.url" );
        
        // prevent recreation mock-up when called
        final IFileSystem mockFileSystem = mock(IFileSystem.class);
        
        EFSUtil.setFileSystemSurrogate(new Surrogate<IFileSystem>() {
            @Override
            public IFileSystem getAdapter() {
                return mockFileSystem;
            }
        });
    }
    
    /**
     * Restore URL handler after test
     */
    @After
    public void tearDown() {
        if ( null == oldConfig )
        {
            System.clearProperty( "java.protocol.handler.pkgs" );
        }
        else
        {
            System.setProperty( "java.protocol.handler.pkgs", oldConfig );
        }
    }
    
    /**
     * Test {@link EFSUtil#isExistResource(IFileStore)}
     * 
     * @throws Exception in case of failure in test
     * 
     * @see EFSUtil#isExistResource(IFileStore)
     */
    @Test
    public void test_isExistResource_IFileStore() throws Exception {
        // null check
        IFileStore nullFileStore = null;
        assertFalse( EFSUtil.isExistResource(nullFileStore) );
        
        
        IFileInfo mockFileInfo = mock( IFileInfo.class );
        IFileStore mockFileStore = mock( IFileStore.class );
        when( mockFileStore.fetchInfo() ).thenReturn( mockFileInfo );
        
        // true check
        when( mockFileInfo.exists() ).thenReturn( true );
        assertTrue( EFSUtil.isExistResource(mockFileStore) );
        
        // false check
        when( mockFileInfo.exists() ).thenReturn( false );
        assertFalse( EFSUtil.isExistResource(mockFileStore) );
    }
    
    /**
     * Test {@link EFSUtil#isExistResource(URI)}
     * 
     * @throws Exception in case of failure in test
     * 
     * @see EFSUtil#isExistResource(URI)
     */
    @Test
    public void test_isExistResource_URI() throws Exception {
        // null check
        final URI nullURI = null;
        when( EFSUtil.getFileSystem().getStore(nullURI) ).thenReturn( null );
        assertFalse( EFSUtil.isExistResource(nullURI) );
        
        
        IFileInfo mockFileInfo = mock( IFileInfo.class );
        IFileStore mockFileStore = mock( IFileStore.class );
        when( mockFileStore.fetchInfo() ).thenReturn( mockFileInfo );
        
        final URI virtualURI = new URL(TEST_URL).toURI();
        when( EFSUtil.getFileSystem().getStore(virtualURI) ).thenReturn( mockFileStore );
        
        // true check
        when( mockFileInfo.exists() ).thenReturn( true );
        assertTrue( EFSUtil.isExistResource(virtualURI) );
        
        // false check
        when( mockFileInfo.exists() ).thenReturn( false );
        assertFalse( EFSUtil.isExistResource(virtualURI) );
    }
    
    /**
     * Test {@link EFSUtil#isExistResourceWithDialog(Shell, String, String, DialogChoicer, IFileStore)}
     * 
     * @throws Exception in case of failure in test
     * 
     * @see EFSUtil#isExistResourceWithDialog(Shell, String, String, DialogChoicer, IFileStore)
     */
    @Test
    public void test_isExistResourceWithDialog() throws Exception {
        Shell mockShell = mock( Shell.class );
        IFileStore mockFileStore = mock( IFileStore.class );
        IFileInfo mockFileInfo = mock (IFileInfo.class );
        when(mockFileStore.fetchInfo()).thenReturn(mockFileInfo);
        
        final Object[][] TEST_CASES = new Object[][] { // [TestCase] [Arguments..., ExpectResult]
                new Object[] { mockShell, "Title", "Msg", DialogChoicer.Cancel, mockFileStore, true, DialogChoicer.Cancel },
                new Object[] { mockShell, "Title", "Msg", DialogChoicer.Cancel, mockFileStore, false, DialogChoicer.Cancel },
                new Object[] { mockShell, "Title", "Msg", DialogChoicer.NoToAll, mockFileStore, true, DialogChoicer.NoToAll },
                new Object[] { mockShell, "Title", "Msg", DialogChoicer.NoToAll, mockFileStore, false, DialogChoicer.NoToAll },
                new Object[] { mockShell, "Title", "Msg", DialogChoicer.YesToAll, mockFileStore, true, DialogChoicer.YesToAll },
                new Object[] { mockShell, "Title", "Msg", DialogChoicer.YesToAll, mockFileStore, false, DialogChoicer.YesToAll },
                new Object[] { mockShell, "Title", "Msg", DialogChoicer.Yes, mockFileStore, false, DialogChoicer.Yes },
                new Object[] { mockShell, "Title", "Msg", DialogChoicer.No, mockFileStore, false, DialogChoicer.Yes }
                // except dialog open scenarios
        };
        
        DialogChoicer existResourceWithDialog = null;
        
        for (final Object[] TEST_CASE : TEST_CASES) {
            final Shell shell = (Shell) TEST_CASE[0];
            final String title = (String) TEST_CASE[1];
            final String msg = (String) TEST_CASE[2];
            final DialogChoicer choicer = (DialogChoicer) TEST_CASE[3];
            final IFileStore fileStore = (IFileStore) TEST_CASE[4];
            final boolean exist = (Boolean) TEST_CASE[5];
            final DialogChoicer expected = (DialogChoicer) TEST_CASE[6];
            
            when( mockFileInfo.exists() ).thenReturn(exist);
            existResourceWithDialog = EFSUtil.isExistResourceWithDialog(
                    shell, title, msg, choicer, fileStore);
            assertEquals( shell + ", " + title + ", " + msg + ", " + choicer + ", " +
                    fileStore.fetchInfo().exists() + " : " + expected, existResourceWithDialog, expected );
        }
    }
    
    
    /**
     * Test {@link EFSUtil#copy(URI, URI, int, IProgressMonitor)}
     * 
     * @throws Exception in case of failure in test
     * 
     * @see EFSUtil#copy(URI, URI, int, IProgressMonitor)
     */
    @Test
    public void test_copy() throws Exception {
        final URI virtualURI = new URL(TEST_URL).toURI();
        final IProgressMonitor mockMonitor = mock(IProgressMonitor.class);
        
        IFileStore mockFileStore = mock( IFileStore.class );
        when( EFSUtil.getFileSystem().getStore(virtualURI) ).thenReturn( mockFileStore );
        
        final Object[][] TEST_CASES = new Object[][] { // [TestCase] [Arguments..., ExpectResult]
                new Object[] { virtualURI, virtualURI, EFS.OVERWRITE, mockMonitor },
                new Object[] { virtualURI, virtualURI, EFS.SHALLOW, mockMonitor },
                new Object[] { virtualURI, virtualURI, EFS.OVERWRITE | EFS.SHALLOW, mockMonitor }
        };
        
        for (final Object[] TEST_CASE : TEST_CASES) {
            final URI source = (URI) TEST_CASE[0];
            final URI destination = (URI) TEST_CASE[1];
            final int options = (Integer) TEST_CASE[2];
            final IProgressMonitor monitor = (IProgressMonitor) TEST_CASE[3];
            
            EFSUtil.copy(source, destination, options, monitor);
            
            verify( mockFileStore ).copy(mockFileStore, options, mockMonitor);
        }
    }
}
