/*
*  Common
*
* Copyright (c) 2000 - 2011 Samsung Electronics Co., Ltd. All rights reserved.
*
* Contact: 
* BonYong Lee <bonyong.lee@samsung.com>
* 
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
* http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
* Contributors:
* - S-Core Co., Ltd
*
*/
package org.tizen.common.util;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotSame;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;
import static org.powermock.api.mockito.PowerMockito.mock;
import static org.powermock.api.mockito.PowerMockito.when;
import static org.powermock.api.mockito.PowerMockito.whenNew;
import static org.tizen.common.util.IOUtil.tryClose;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.InputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Stack;

import org.junit.Test;
import org.junit.runner.RunWith;
import org.powermock.core.classloader.annotations.PrepareForTest;
import org.powermock.modules.junit4.PowerMockRunner;

/**
 * FileUtilTest
 * 
 * Test case for {@link FileUtil}
 * 
 * @author BonYong Lee{@literal <bonyong.lee@samsung.com>} (S-Core)
 * 
 * @see FileUtil
 */
@RunWith( PowerMockRunner.class )
@PrepareForTest( FileUtil.class )
public class FileUtilTest {

    /**
     * Target directory for test
     */
    public static final String TEST_RESOURCE_DEST = "test/_test_files";

    /**
     * Source directory for test
     */
    public static final String TEST_RESOURCE_SRC = "test/test_files";

    /**
     * Test for {@link FileUtil#getFileExtension(String)}
     * 
     * @throws Exception in case of failure in test
     * 
     * @see FileUtil#getFileExtension(String)
     */
    @Test
    public void test_getFileExtension() throws Exception {
        final Object[][] TEST_CASES = new Object[][] {
                new Object[] { "aaaa.exe", "exe" },
                new Object[] { "index.html", "html" },
                new Object[] { "index.test.html", "html" },
                new Object[] { "index..html", "html" },
                new Object[] { ".profile", "profile" },
                new Object[] { "ReadMe ", null },
                new Object[] { "ReadMe", null },
                new Object[] { "ReadMe. ", " " },
                new Object[] { "ReadMe. test", " test" }
        };

        int i = 0;
        for ( final Object[] TEST_CASE : TEST_CASES ) {
            final String fileName = (String) TEST_CASE[0];
            final String expected = (String) TEST_CASE[1];
            final String result = FileUtil.getFileExtension( fileName );
            assertEquals( ++i + "th Test case :<" + fileName + ", " + expected + ">", expected, result );
        }
    }

    /**
     * Test for {@link FileUtil#getFileNameFromPath(String)}
     * 
     * @throws Exception If test fails
     */
    public void test_getFileNameFromPath() throws Exception {
        final String[][] TEST_CASES = {
                {"ASDFwew/jkl/wer23/gfdgfdg", "gfdgfdg"},
                {"wekljf\\hgrfg\\wefwv1.aaa", "wefwv1.aaa"},
                {null, null},
                {"ggg.sss", "ggg.sss"}
        };

        for( final String[] TEST_CASE: TEST_CASES ) {
            String result = FileUtil.getFileNameFromPath(TEST_CASE[0]);
            assertEquals(TEST_CASE[1], result);
        }
    }

    /**
     * Test for {@link FileUtil#readTextStream(InputStream input, String encoding)}
     *
     * @throws Exception in case of failure in test
     *
     * @see {@link FileUtil#readTextStream(InputStream input, String encoding)}
     */
    @Test
    public void test_readTextStream() throws Exception {
        final String text = "Test Text 123";
        final String unicodeText = "\uD55C\uAE00\uD14C\uC2A4\uD2B8";

        try {
            FileUtil.readTextStream(null, null);
            fail( "readTextStream must throw exception" );
        } catch ( final Exception e ) {
        }

        final InputStream input = new ByteArrayInputStream(text.getBytes());
        final String content = FileUtil.readTextStream(input, null);
        assertEquals(text, content);

        final InputStream unicodeInput = new ByteArrayInputStream(unicodeText.getBytes("UTF-8"));
        final String unicodeContent = FileUtil.readTextStream(unicodeInput, "UTF-8");
        assertEquals(unicodeText, unicodeContent);
        final String isoContent = FileUtil.readTextStream(unicodeInput, "ISO-8859-1");
        assertNotSame(unicodeText, isoContent);

        tryClose( input, unicodeContent );
    }

    /**
     * Test {@link FileUtil#copyRecursively(String, String)}
     *  
     * @throws Exception in case of failure in test
     * 
     * @see {@link FileUtil#copyRecursively(String, String)}
     */
    @Test
    public void test_copyRecursively1() throws Exception {
        try {
            if(!checkTestSrcExist()) {
                return;
            }

            FileUtil.copyRecursively(TEST_RESOURCE_SRC, TEST_RESOURCE_DEST);
            List<File> result = compareFiles(TEST_RESOURCE_SRC, TEST_RESOURCE_DEST);
            assertTrue(result.size() == 0);
        } finally {
            File destDir = new File(TEST_RESOURCE_DEST);
            if(destDir.exists()) {
                FileUtil.recursiveDelete(destDir);
            }
        }
    }

    /**
     * Test {@link FileUtil#copyRecursively(String, String, boolean)}}
     *  
     * @throws Exception in case of failure in test
     * 
     * @see {@link FileUtil#copyRecursively(String, String, boolean)}
     */
    @Test
    public void test_copyRecursively2() throws Exception {
        try {
            if(!checkTestSrcExist()) {
                return;
            }

            String overwriteFileName = "/about_files/LICENSE-2.0.htm";
            String originalFileName = "/about_files/freemarker-LICENSE.txt";
            String copiedFileName = FileUtil.appendPath(TEST_RESOURCE_DEST, originalFileName);

            long overwriteSize = new File(TEST_RESOURCE_SRC + overwriteFileName).getTotalSpace();
            long originalSize = new File(TEST_RESOURCE_SRC + originalFileName).getTotalSpace();

            FileUtil.copyTo(FileUtil.appendPath(TEST_RESOURCE_SRC, overwriteFileName), copiedFileName);
            FileUtil.copyRecursively(TEST_RESOURCE_SRC, TEST_RESOURCE_DEST, false);

            long copiedSize = new File(copiedFileName).getTotalSpace();
            assertTrue(overwriteSize == copiedSize);

            File destDir = new File(TEST_RESOURCE_DEST);
            if(destDir.exists()) {
                FileUtil.recursiveDelete(destDir);
            }

            List<File> result = compareFiles(TEST_RESOURCE_SRC, TEST_RESOURCE_DEST);
            assertTrue(result.size() == 0);

            FileUtil.copyRecursively(TEST_RESOURCE_SRC, TEST_RESOURCE_DEST, true);

            copiedSize = new File(copiedFileName).getTotalSpace();
            assertTrue(originalSize == copiedSize);

            result = compareFiles(TEST_RESOURCE_SRC, TEST_RESOURCE_DEST);
            assertTrue(result.size() == 0);
        } finally {
            File destDir = new File(TEST_RESOURCE_DEST);
            if(destDir.exists()) {
                FileUtil.recursiveDelete(destDir);
            }
        }
    }

    /**
     * Test {@link FileUtil#copyRecursively(String, String, boolean, File...)}
     *  
     * @throws Exception in case of failure in test
     * 
     * @see {@link FileUtil#copyRecursively(String, String, boolean, File...)}
     */
    @Test
    public void test_copyRecursively3() throws Exception {
        try {
            if(!checkTestSrcExist()) {
                return;
            }

            HashSet<File> filter = new HashSet<File>();
            File filter1 = new File(TEST_RESOURCE_SRC + "/about_files/LICENSE-2.0.htm");
            File filter2 = new File(TEST_RESOURCE_SRC + "/resource/text.txt");
            File filter3 = new File(TEST_RESOURCE_SRC + "/resource/resource/text.txt");

            filter.add(filter1);
            filter.add(filter2);
            filter.add(filter3);

            FileUtil.copyRecursively(TEST_RESOURCE_SRC, TEST_RESOURCE_DEST, false, filter1, filter2, filter3);

            List<File> result = compareFiles(TEST_RESOURCE_SRC, TEST_RESOURCE_DEST);

            for(File resultFile: result) {
                assertTrue(filter.contains(resultFile));
            }
        } finally {
            File destDir = new File(TEST_RESOURCE_DEST);
            if(destDir.exists()) {
                FileUtil.recursiveDelete(destDir);
            }
        }
    }

    private boolean checkTestSrcExist() {
        File src = new File(TEST_RESOURCE_SRC);
        return src.exists();
    }

    /**
     * This method is used for checking whether copyRecursively works fine.
     * If destDir does not contain some files in srcDir, they are contained in result List.
     * @param srcDir source directory
     * @param destDir destination direcoty to be compared with source directory.
     * @return Files in source directory destination directory does not contain.
     */
    public static List<File> compareFiles(String srcDir, String destDir) {
        List<File> result = new ArrayList<File>();
        File srcFiles = new File(srcDir);
        if(!srcFiles.exists()) {
            return result;
        }

        Stack<File> srcStack = new Stack<File>();
        Stack<File> destStack = new Stack<File>();

        while(!srcStack.isEmpty()) {
            File _srcFile = srcStack.pop();
            File _destFile = destStack.pop();

            if(!_destFile.exists()) {
                result.add(_srcFile);
            }

            if(_srcFile.isDirectory()) {
                for(File __srcFile:_srcFile.listFiles()) {
                    srcStack.add(__srcFile);
                    destStack.add(new File(_destFile, __srcFile.getName()));
                }
            }
        }

        return result;
    }

    /**
     * Test {@link FileUtil#readFromFile(URL)}
     * 
     * @throws Exception in case of failure in test
     * 
     * @see {@link FileUtil#readFromFile(URL)}
     * @throws Exception
     */
    @Test
    public void test_readFromFile() throws Exception {
        String result = "HOHOHO\nHAHAHA\nNAMKOONGHO";
        File file = new File("test/test_files/resource/text.txt");
        if(file.exists()) {
            URL url = file.toURI().toURL();
            assertTrue(result.equals(FileUtil.readFromFile(url)));
        }
    }

    /**
     * Test {@link FileUtil#appendPath(String, String)}
     * Test {@link FileUtil#appendPath(String, String, boolean)}
     * 
     * @throws Exception in case of failure in test
     *
     * @see {@link FileUtil#appendPath(String, String)}
     * @see {@link FileUtil#appendPath(String, String, boolean))}
     */
    @Test
    public void test_appendPath() throws Exception {
        String windowFirst = "\\a\\b\\c\\d\\";
        String windowLast = "\\e\\f\\g\\h\\";

        String linuxFirst = "/a/b/c/d/";
        String linuxLast = "/e/f/g/h/";

        String result = File.separatorChar + "a" + File.separatorChar + "b" + File.separatorChar + "c" + 
                File.separatorChar + "d" + File.separatorChar + "e" + File.separatorChar + "f" + File.separatorChar + 
                "g" + File.separatorChar + "h" + File.separatorChar;

        assertEquals(result, FileUtil.appendPath(windowFirst, linuxLast));
        assertEquals(result, FileUtil.appendPath(windowFirst, windowLast));
        assertEquals(result, FileUtil.appendPath(linuxFirst, linuxLast));
        assertEquals(result, FileUtil.appendPath(linuxFirst, windowLast));

        result = result.replace('\\', '/');
        assertEquals(result, FileUtil.appendPath(windowFirst, linuxLast, false));
        assertEquals(result, FileUtil.appendPath(windowFirst, windowLast, false));
        assertEquals(result, FileUtil.appendPath(linuxFirst, linuxLast, false));
        assertEquals(result, FileUtil.appendPath(linuxFirst, windowLast, false));

        result = result.replace('/', '\\');
        assertEquals(result, FileUtil.appendPath(windowFirst, linuxLast, true));
        assertEquals(result, FileUtil.appendPath(windowFirst, windowLast, true));
        assertEquals(result, FileUtil.appendPath(linuxFirst, linuxLast, true));
        assertEquals(result, FileUtil.appendPath(linuxFirst, windowLast, true));
    }

    /**
     * Test {@link FileUtil#convertToOSPath(String, boolean)}
     * Test {@link FileUtil#convertToOSPath(String)}
     * 
     * @throws Exception
     */
    public void test_convertToOSPath() throws Exception {
        String windowPath = "\\a\\b\\c\\d\\";
        String linuxPath = "/a/b/c/d/";

        assertEquals(linuxPath, FileUtil.convertToOSPath(windowPath, false));
        assertEquals(linuxPath, FileUtil.convertToOSPath(linuxPath, false));
        assertEquals(windowPath, FileUtil.convertToOSPath(windowPath, true));
        assertEquals(windowPath, FileUtil.convertToOSPath(linuxPath, true));

        String result = File.separatorChar + "a" + File.separatorChar + "b" + File.separatorChar + "c" + File.separatorChar + "d";

        assertEquals(result, FileUtil.convertToOSPath(windowPath));
        assertEquals(result, FileUtil.convertToOSPath(linuxPath));
    }

    /**
     * Test {@link FileUtil#isExist(String)}
     * 
     * @throws Exception in case of failure in test
     *
     * @see {@link FileUtil#isExist(String)}
     */
    @Test
    public void test_isExist() throws Exception {
        assertFalse( FileUtil.isExist( null ) );
        assertFalse( FileUtil.isExist( "" ) );

        final File textAbc = mock( File.class );
        when( textAbc.exists() ).thenReturn( Boolean.FALSE );
        final File resource = mock( File.class );
        when( resource.exists() ).thenReturn( Boolean.TRUE );
        final File textTxt = mock( File.class );
        when( textTxt.exists() ).thenReturn( Boolean.TRUE );
        whenNew( File.class ).withArguments( "test/test_files/resource/text.abc" ).thenReturn( textAbc );
        whenNew( File.class ).withArguments( "test/test_files/resource" ).thenReturn( resource );
        whenNew( File.class ).withArguments( "test/test_files/resource/" ).thenReturn( resource );
        whenNew( File.class ).withArguments( "test/test_files/resource/text.txt" ).thenReturn( textTxt );
        assertFalse( FileUtil.isExist( "test/test_files/resource/text.abc" ) );
        assertTrue( FileUtil.isExist( "test/test_files/resource" ) );
        assertTrue( FileUtil.isExist( "test/test_files/resource/" ) );
        assertTrue( FileUtil.isExist( "test/test_files/resource/text.txt" ) );
    }

}
