/*
 * Common
 *
 * Copyright (c) 2000 - 2011 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Contact: 
 * BonYong Lee <bonyong.lee@samsung.com>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Contributors:
 * - S-Core Co., Ltd
 *
 */
package org.tizen.common.util;

import static org.powermock.api.mockito.PowerMockito.mock;
import static org.powermock.api.mockito.PowerMockito.when;
import static org.powermock.api.mockito.PowerMockito.whenNew;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.powermock.api.mockito.PowerMockito;
import org.powermock.core.classloader.annotations.PrepareForTest;
import org.powermock.modules.junit4.PowerMockRunner;

@RunWith( PowerMockRunner.class )
@PrepareForTest( { HostUtil.class } )
public class HostUtilTest {

    @Before
    public void setUp() {

    }

    @After
    public void tearDown() {

    }
    
    /**
     * Test {@link HostUtil#getContents( String)}
     * 
     * @throws Exception in case of failure in test
     * 
     * @see HostUtil#getContents( String)
     */
    @SuppressWarnings("unchecked")
    @Test
    public void test_getContents() throws Exception {
        String fullPath = "test_path";
        FileReader file = PowerMockito.mock( FileReader.class );
        whenNew( FileReader.class ).withParameterTypes( String.class ).withArguments( fullPath ).thenReturn( file );
        Assert.notNull(HostUtil.getContents(fullPath));
        
        whenNew( FileReader.class ).withParameterTypes( String.class ).withArguments( "nullPath" ).thenThrow( IOException.class );
        Assert.isNull(HostUtil.getContents("nullPath"));
    }
    
    /**
     * Test {@link HostUtil#exists( String)}
     * 
     * @throws Exception in case of failure in test
     * 
     * @see HostUtil#exists( String)
     */
    @Test
    public void test_exists() throws Exception {
        String fullPath = "test_path";
        final File mockFile = mock( File.class );
        whenNew( File.class ).withArguments( fullPath ).thenReturn( mockFile );
        when( mockFile.exists() ).thenReturn( Boolean.TRUE );
        Assert.isTrue(HostUtil.exists(fullPath));
        Assert.isFalse(HostUtil.exists(null));
        final File mockFile2 = mock( File.class );
        when( mockFile2.exists() ).thenReturn( Boolean.FALSE );
        whenNew( File.class ).withArguments( "nullPath" ).thenReturn( mockFile2 );
        Assert.isFalse(HostUtil.exists("nullPath"));
    }
    
    /**
     * Test {@link HostUtil#execute( String)}
     * 
     * @throws Exception in case of failure in test
     * 
     * @see HostUtil#execute( String)
     */
    @Test
    public void test_execute() throws Exception {
        Assert.isTrue(HostUtil.execute("cd"));
        Assert.isFalse(HostUtil.execute(null));
        Assert.isFalse(HostUtil.execute("notcommand"));
    }
    
    /**
     * Test {@link HostUtil#batchExecute( String)}
     * 
     * @throws Exception in case of failure in test
     * 
     * @see HostUtil#batchExecute( String)
     */
    @Test
    public void test_batchExecute() throws Exception {
        Assert.isFalse(HostUtil.batchExecute(null));
        Assert.isTrue(HostUtil.batchExecute("cd"));
        Assert.isFalse(HostUtil.batchExecute("notcommand"));
    }
    
    /**
     * Test {@link HostUtil#returnExecute( String)}
     * 
     * @throws Exception in case of failure in test
     * 
     * @see HostUtil#returnExecute( String)
     */
    @Test
    public void test_returnExecute1() throws Exception {
        Assert.isNull(HostUtil.returnExecute(null));
        Assert.notNull(HostUtil.returnExecute("cd"));
    }
    
    /**
     * Test {@link HostUtil#returnExecute( String, String)}
     * 
     * @throws Exception in case of failure in test
     * 
     * @see HostUtil#returnExecute( String, String)
     */
    @Test
    public void test_returnExecute2() throws Exception {
        final String filePath = "path";
        Assert.isNull(HostUtil.returnExecute(null, filePath));
        Assert.notNull(HostUtil.returnExecute("cd", null));
        final ProcessBuilder processBuilder = mock( ProcessBuilder.class );
        final Process process = mock( Process.class );
        whenNew( ProcessBuilder.class ).withNoArguments().thenReturn( processBuilder );
        when( processBuilder.start() ).thenReturn( process );
        when( process.getInputStream() ).thenReturn( new ByteArrayInputStream( "".getBytes() ) );
        Assert.notNull(HostUtil.returnExecute("cd", null));
    }
    
    /**
     * Test {@link HostUtil#executeWithConsole( String, String)}
     * 
     * @throws Exception in case of failure in test
     * 
     * @see HostUtil#executeWithConsole( String, String)
     */
    @Test(expected = Exception.class)
    public void test_executeWithConsole() throws Exception {
        HostUtil.executeWithConsole("notcommand", "viewname");
    }
}
