/*
*  Common
*
* Copyright (c) 2000 - 2011 Samsung Electronics Co., Ltd. All rights reserved.
*
* Contact:
* ChangHyun Lee <changhyun1.lee@samsung.com>
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
* http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
* Contributors:
* - S-Core Co., Ltd
*
*/

package org.tizen.common.util;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

import org.junit.Test;


/**
 * MapUtilTest.
 *
 * Test case for {@link MapUtil}
 *
 * @author Changhyun Lee{@literal <changhyun1.lee@samsung.com>} (S-Core)
 *
 * @see MapUtil
 *
 */
public class MapUtilTest {
    /**
     * Test {@link MapUtil#length(Map)}
     *
     * @throws Exception in case of failure in test
     *
     * @see MapUtil#length(Map)
     */
    @Test
    public void test_length() throws Exception {
        assertEquals(0, MapUtil.length(null));
        Map<String, String> map = new HashMap<String, String>();
        map.put("aaa", "111");
        map.put("bbb", "222");
        assertEquals(map.size(), MapUtil.length(map));
    }

    /**
     * Test {@link MapUtil#isEmpty(Map)}
     *
     * @throws Exception in case of failure in test
     *
     * @see MapUtil#isEmpty(Map)
     */
    @Test
    public void test_isEmpty() throws Exception {
        assertTrue(MapUtil.isEmpty(null));

        Map<String, String> map = new HashMap<String, String>();
        assertTrue(MapUtil.isEmpty(map));

        map.put("aaa", "111");
        map.put("bbb", "222");
        assertFalse(MapUtil.isEmpty(map));
    }

    /**
     * Test {@link MapUtil#asMap(Object[], Object[])
     *
     * @throws Exception in case of failure in test
     *
     * @see MapUtil#asMap(Object[], Object[])
     */
    @Test
    public void test_asMap1() throws Exception {
        String[] keys = {"aaa", "bbb", "ccc"};
        String[] values = {"111", "222", "333"};
        String[] values2 = {"111", "222"};

        assertTrue(MapUtil.asMap(null, null) != null);
        assertTrue(MapUtil.asMap(keys, null) != null);
        assertTrue(MapUtil.asMap(null, values) != null);

        Map<String, String> map1 = MapUtil.asMap(keys, values);
        assertEquals(map1.size(), 3);
        Map<String, String> map2 = MapUtil.asMap(keys, values2);
        assertEquals(map2.size(), 2);
        Map<String, String> map3 = MapUtil.asMap(null, null);
        assertEquals(map3.size(), 0);
    }

    /**
     * Test {@link MapUtil#asMap(Object[][] objs)}
     *
     * @throws Exception in case of failure in test
     *
     * @see {@link MapUtil#asMap(Object[][] objs)}
     */
    @Test
    public void test_asMap2() throws Exception {
        final Object[][] TEST_CASES = new Object[][] {
            new Object[] { "aaa", "111" },
            new Object[] { "bbb", "222" },
        };

        final Object[][] TEST_CASES2 = new Object[][] {
            new Object[] {"aaa"},
            new Object[] {null},
            new Object[] { "aaa", "111" },
            new Object[] { "bbb", "222" },
        };

        assertTrue(MapUtil.asMap(null) != null);

        Map<Object, Object> map1 = MapUtil.asMap(TEST_CASES);
        assertTrue(map1 != null);
        assertEquals(map1.size(), 2);
        int iTestCase = 0;
        for ( final Object[] TEST_CASE : TEST_CASES ) {
            ++iTestCase;
            final Object[] object = TEST_CASE;
            assertEquals(iTestCase + " th test case", object[1], map1.get(object[0]));
        }

        Map<Object, Object> map2 = MapUtil.asMap(TEST_CASES2);
        assertTrue(map2 != null);
        assertEquals(map2.size(), 2);
    }

    /**
     * Test {@link MapUtil#mergePropertiesIntoMap(Properties, Map)}
     *
     * @throws Exception in case of failure in test
     *
     * @see MapUtil#mergePropertiesIntoMap(Properties, Map)
     */
    @Test
    public void test_mergePropertiesIntoMap() throws Exception {
        Properties properties = new Properties();
        properties.setProperty("aaa", "111");
        properties.setProperty("bbb", "222");

        HashMap<Object, Object> map = new HashMap<Object, Object>();

        try {
            MapUtil.mergePropertiesIntoMap(null, null);
            fail( "mergePropertiesIntoMap must throw exception" );
        } catch (Exception e) {
        }

        try {
            MapUtil.mergePropertiesIntoMap(properties, null);
            fail( "mergePropertiesIntoMap must throw exception" );
        } catch (Exception e) {
        }

        MapUtil.mergePropertiesIntoMap(null, map);

        assertEquals(map.size(), 0);
        MapUtil.mergePropertiesIntoMap(properties, map);
        assertEquals(map.size(), 2);
        assertEquals(properties.getProperty("aaa"), map.get("aaa"));
        assertEquals(properties.getProperty("bbb"), map.get("bbb"));
    }

    /**
     * Test {@link MapUtil#toString(Map)}
     *
     * @throws Exception in case of failure in test
     *
     * @see MapUtil#toString(Map)
     */
    @Test
    public void test_toString() throws Exception {
		final Map<String, String> testTarget = new HashMap<String, String>();
		testTarget.put( "testKey", "testValue" );
		
		final String string = MapUtil.toString( testTarget );
		
		assertTrue( string.contains( "testKey" ) );
		assertTrue( string.contains( "testValue" ) );
		
		assertNotNull( MapUtil.toString( null ) );
	}
}
