/*
 *  Common
 *
 * Copyright (c) 2000 - 2011 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Contact:
 * BonYong Lee <bonyong.lee@samsung.com>
 * Ho Namkoong <ho.namkoong@samsung.com>
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Contributors:
 * - S-Core Co., Ltd
 *
 */

package org.tizen.common.util.log;

import java.io.File;
import java.io.IOException;
import static junit.framework.Assert.*;

import org.apache.log4j.EnhancedPatternLayout;
import org.apache.log4j.Logger;
import org.apache.log4j.spi.LoggingEvent;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import static org.powermock.api.mockito.PowerMockito.*;
import org.tizen.common.util.FileUtil;
import org.tizen.common.util.StringUtil;

public class FileAppenderTest {

    private static final String TEST_LOG_FILE_FIRST = "log.2013-04-15 13'16'11";
    private static final String TEST_LOG_FILE_SECOND = "test2.txt";
    private static final String TEST_LOG_FILE_NOOP = "noop.txt";
    private static final String RESULT = "A";
    private static final String BUFFER_RESULT_FIRST = "AAAAA";
    private static final String BUFFER_RESULT_SECOND = "BBBBB";
    
    private FileAppender appender;
    private EnhancedPatternLayout TEST_LAYOUT = new EnhancedPatternLayout("%m");
    private LoggingEvent event;
    
    /**
     * Sets up the test
     * 
     * @throws Exception
     */
    @Before
    public void setUp() throws Exception {
        File testFile = new File(TEST_LOG_FILE_FIRST);
        if(testFile.exists()) {
            testFile.delete();
        }
        
        File testFile2 = new File(TEST_LOG_FILE_SECOND);
        if(testFile2.exists()) {
            testFile2.delete();
        }
        
        appender = new FileAppender(TEST_LOG_FILE_FIRST, TEST_LAYOUT, 5);
        event = new LoggingEvent( null, Logger.getLogger( getClass() ), Level.ERROR, RESULT, null );
    }

    /**
     * 
     * TestCase for {@link FileAppender#append(LoggingEvent)}
     * It tests timeout
     * 
     * @author ho.namkoong (ho.namkoong@samsung.com)
     * 
     * @throws Exception
     */
    @Test
    public void test_append_timeout() throws Exception {
        appender.append(event);
        String emptyResult = FileUtil.readTextFile(new File(TEST_LOG_FILE_FIRST), null);
        assertEquals(StringUtil.EMPTY_STRING, emptyResult);
        
        Thread.sleep(4000);
        
        String result = FileUtil.readTextFile(new File(TEST_LOG_FILE_FIRST), null);
        assertEquals(RESULT, result);
    }
    
    /**
     * 
     * TestCase for {@link FileAppender#append(LoggingEvent)}
     * It tests buffer overflow
     * 
     * @author ho.namkoong (ho.namkoong@samsung.com)
     * 
     * @throws Exception
     */
    @Test
    public void test_appender_bufferover() throws Exception {
        event = new LoggingEvent( null, Logger.getLogger( getClass() ), Level.ERROR, BUFFER_RESULT_FIRST, null );
        appender.append(event);
        String result = FileUtil.readTextFile(new File(TEST_LOG_FILE_FIRST), null);
        assertEquals(BUFFER_RESULT_FIRST, result);
        
        event = new LoggingEvent( null, Logger.getLogger( getClass() ), Level.ERROR, BUFFER_RESULT_SECOND, null );
        appender.append(event);
        result = FileUtil.readTextFile(new File(TEST_LOG_FILE_FIRST), null);
        assertEquals(BUFFER_RESULT_FIRST +  BUFFER_RESULT_SECOND, result);
    }
    
    /**
     * 
     * TestCase for {@link FileAppender#setFilePath(String)}
     * 
     * @author ho.namkoong (ho.namkoong@samsung.com)
     * 
     * @throws Exception
     */
    @Test
    public void test_setFilePath() throws Exception {
        event = new LoggingEvent( null, Logger.getLogger( getClass() ), Level.ERROR, BUFFER_RESULT_FIRST, null );
        appender.append(event);
        String result = FileUtil.readTextFile(new File(TEST_LOG_FILE_FIRST), null);
        assertEquals(BUFFER_RESULT_FIRST, result);
        
        appender.setFilePath(TEST_LOG_FILE_SECOND);
        
        event = new LoggingEvent( null, Logger.getLogger( getClass() ), Level.ERROR, BUFFER_RESULT_SECOND, null );
        appender.append(event);
        result = FileUtil.readTextFile(new File(TEST_LOG_FILE_FIRST), null);
        assertEquals(BUFFER_RESULT_FIRST, result);
        result = FileUtil.readTextFile(new File(TEST_LOG_FILE_SECOND), null);
        assertEquals(BUFFER_RESULT_SECOND, result);
    }
    
    /**
     * 
     * TestCase for {@link FileAppender#setBufferSize(int)}
     * 
     * @author ho.namkoong (ho.namkoong@samsung.com)
     * 
     * @throws Exception
     */
    @Test
    public void test_setBufferSize() throws Exception {
        event = new LoggingEvent( null, Logger.getLogger( getClass() ), Level.ERROR, BUFFER_RESULT_FIRST, null );
        appender.append(event);
        String result = FileUtil.readTextFile(new File(TEST_LOG_FILE_FIRST), null);
        assertEquals(BUFFER_RESULT_FIRST, result);
        
        appender.setBufferSize(10);
        
        event = new LoggingEvent( null, Logger.getLogger( getClass() ), Level.ERROR, BUFFER_RESULT_SECOND, null );
        appender.append(event);
        result = FileUtil.readTextFile(new File(TEST_LOG_FILE_FIRST), null);
        assertEquals(BUFFER_RESULT_FIRST, result);
        
        Thread.sleep(4000);
        
        result = FileUtil.readTextFile(new File(TEST_LOG_FILE_FIRST), null);
        assertEquals(BUFFER_RESULT_FIRST + BUFFER_RESULT_SECOND, result);
    }
    
    /**
     * 
     * TestCase for {@link FileAppender#append(LoggingEvent)}
     * It tests NOOP_OUTPUT_STREAM
     * 
     * @author ho.namkoong (ho.namkoong@samsung.com)
     * 
     * @throws Exception
     */
    @Test
    public void test_append_noop() throws Exception {
        File noopFile = new File(TEST_LOG_FILE_NOOP);
        
        File mockFile = mock(File.class);
        when(mockFile.exists()).thenReturn(false);
        when(mockFile.getAbsoluteFile()).thenReturn(mockFile);
        when(mockFile.getParentFile()).thenReturn(mockFile);
        when(mockFile.mkdirs()).thenReturn(true);
        when(mockFile.createNewFile()).thenThrow(new IOException());
        
        whenNew(File.class).withArguments(TEST_LOG_FILE_NOOP).thenReturn(mockFile);
        event = new LoggingEvent( null, Logger.getLogger( getClass() ), Level.ERROR, BUFFER_RESULT_FIRST, null );
        appender.append(event);
        assertFalse(noopFile.exists());
        
    }
    
    /**
     * 
     * Tears down the test
     * 
     * @author ho.namkoong (ho.namkoong@samsung.com)
     * 
     * @throws Exception
     */
    @After
    public void tearDown() throws Exception {
        File testFile = new File(TEST_LOG_FILE_FIRST);
        if(testFile.exists()) {
            testFile.delete();
        }
        
        File testFile2 = new File(TEST_LOG_FILE_SECOND);
        if(testFile2.exists()) {
            testFile2.delete();
        }
        
        TEST_LAYOUT.setConversionPattern("%m");
        appender.FILE_OUTPUT_STREAM.close();
        appender = null;
    }
    
}
