=begin 
  
 TizenPublic/analyzer.rb 
 
Copyright (c) 2000 - 2011 Samsung Electronics Co., Ltd. All rights reserved. 
 
Contact: 
Taejun Ha <taejun.ha@samsung.com> 
Jiil Hyoun <jiil.hyoun@samsung.com> 
Donghyuk Yang <donghyuk.yang@samsung.com> 
DongHee Yang <donghee.yang@samsung.com> 
 
Licensed under the Apache License, Version 2.0 (the "License"); 
you may not use this file except in compliance with the License. 
You may obtain a copy of the License at 
 
http://www.apache.org/licenses/LICENSE-2.0 
 
Unless required by applicable law or agreed to in writing, software 
distributed under the License is distributed on an "AS IS" BASIS, 
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
See the License for the specific language governing permissions and 
limitations under the License. 
 
Contributors: 
- S-Core Co., Ltd 
=end 

require File.dirname(__FILE__) + "/pcfile"
require File.dirname(__FILE__) + "/package"
require File.dirname(__FILE__) + "/headerfile"
require File.dirname(__FILE__) + "/devpackage"


class TizenPublicFrameworkAnalyzer
	@rootstrap = nil
	# pkg_name -> package
	@package_list = {}
	@header_list = {}
	@pc_list = {}
	@devpackage_list = []
	@public_package_list = []

	# category_name -> package list
	@framework = {}

	# packagke_name -> package name
	@substitutes = {}

	def initialize( rootstrap )
		@rootstrap = rootstrap
	end


	# analyze 
	def analyze	
		# scan all packages => @package_list
		scan_all_packages

		# initialize framework structure => @framework, @substitutes
		generate_framework_structure

		# generate header to header dependency => @header2header, @pkg2header, @pkg2pkg
		generate_header2header_dependency

		# generate dev packages
		generate_dev_packages 	  
	end


	def write_dev_package_xml
		File.open("#{@rootstrap.name}/DevPackages.xml","w") do |f|
			f.puts "<PackageInfo>"
			@devpackage_list.each do |devPackage|
				devpackage_attr = "  <DevPackage name=\"#{devPackage.name}\""
				devpackage_attr = devpackage_attr + " is_framework=\"#{devPackage.is_framework}\""
				devpackage_attr = devpackage_attr + " is_virtual=\"#{devPackage.is_virtual}\">"
				f.puts devpackage_attr

                if devPackage.is_virtual then
                    if devPackage.include_devpackage_list.empty?  then
                        $log.error "ERROR! devPackage[#{devPackage.name}] is virtual but there has not include_devpackage_list"
			            f.puts "  </DevPackage>"
                        next
                    end

					devPackage.include_devpackage_list.each do |include_devpackage|
						f.puts "    <include_package>#{include_devpackage}</include_package>"
					end
				else
					if not devPackage.include_path_list.empty?  then
						devPackage.include_path_list.each do |inc_path|
							f.puts "    <include_path>#{inc_path}</include_path>"
						end
					end
    
					if not devPackage.other_cflag_list.empty?  then
						devPackage.other_cflag_list.each do |c_flag|
							f.puts "    <other_cflag>#{c_flag}</other_cflag>"
						end
					end
    
					if not devPackage.other_lflag_list.empty?  then
						devPackage.other_lflag_list.each do |l_flag|
							f.puts "    <other_lflag>#{l_flag}</other_lflag>"
						end
					end
    
					if not devPackage.library_list.empty?  then
						devPackage.library_list.each do |library|
							f.puts "    <library>#{library}</library>"
						end
					end
    
					if not devPackage.library_path_list.empty?  then
						devPackage.library_path_list.each do |library_path|
							f.puts "    <library_path>#{library_path}</library_path>"
						end
					end
    
					if not devPackage.dependency_list.empty?  then
						devPackage.dependency_list.each do |dependency|
							f.puts "    <dependency>#{dependency}</dependency>"
						end
					end
    
					if not devPackage.description.empty?  then
						f.puts "    <description>#{devPackage.description}</description>"
					end
				end
			    f.puts "  </DevPackage>"
			end
			
			f.puts "</PackageInfo>"
		end
	end


	# get un-used header file path
	# these will be removed
	def get_unused_header_list
		
		# get all header list
		result = @header_list.values.map {|h| h.path}

		# get valid header list
		used = []
		@package_list.values.each do |pkg|
			if @public_package_list.include? pkg then
				used = used | pkg.header_list.map { |h| h.path }
				used = used | pkg.priv_header_list.map { |h| h.path }
			end
		end

		# remove dup
		used.uniq!

		# remove used list  for total
		result = result - used
	
		return result
	end




	private


	# scan packages and create package map
	def scan_all_packages
		$log.info "Scanning all installed Packages..."

        @package_list = {}
        @header_list = {}
        @pc_list = {}

		# for all packages
        @rootstrap.get_all_installed_packages.each do |pkg_name|
			$log.info "Scanning... #{pkg_name}"
            pkg = DebPackage.new(pkg_name) 
    
			# for all files in the package    
            @rootstrap.get_all_package_files(pkg_name).each do |rpath|
                if rpath.end_with?(".pc") then 

                    #add pc_file
					pc = PcFile.new(rpath)
					pc.parent = pkg
					pkg.pc_list.push pc

					# extract pc info
                    abs_path = "#{@rootstrap.path}#{rpath}"
					abs_path.gsub!("//","/")
					pc.load_from_file(abs_path)

					# add to global header list
					@pc_list[pc.name] = pc

                elsif rpath.end_with?(".h") or 
					rpath.end_with?(".x") or 
					rpath.end_with?(".def") then
 
                   	# add pkg2headers
					h = HeaderFile.new(rpath)
					h.parent = pkg
					pkg.header_list.push h
 
					# extract include statement from header file
                    abs_path = "#{@rootstrap.path}#{rpath}"
					abs_path.gsub!("//","/")
					h.include_list = extract_include_stmt(abs_path)

					# add to global header list
					@header_list[rpath] = h
				else
					next
                end
			end
			
			# add package to map , if valid
			if not pkg.pc_list.empty?  or not pkg.header_list.empty? then
            	@package_list[pkg.name] = pkg
			end
       	end
	end


	# extract #include
	# TODO: should be refactored
	def extract_include_stmt(src)
		return `sed -n "H; /\\*\\// {g; s/\\/\\*.*\\*\\///g;p;n;h;}; $ {g;p;}" #{src} | grep "^[[:space:]]*#[[:space:]]*include[[:space:]]*[<\\"]" | cut -d"e" -f2- | sed -e "s/[ \\t]*//g" -e "s/\\/\\/.*$//g" -e "s/^[<\\"]//g" -e "s/[>\\"].*//g"`.split
	end


	# generate framework structure 
	def generate_framework_structure
		# load "framework" skelecton file
		load_framework_skelecton

		# append "CAPI" frameworks
		append_capi_frameworks
	end


	
	# load framework skelcton file
	def	load_framework_skelecton
        @framework = {}
        @substitutes = {}

		# first , check working directory
		if File.exist? "#{$WORKING_DIR}/framework" then
			framework_path = "#{$WORKING_DIR}/framework"
		else
			framework_path = "#{File.dirname(__FILE__)}/framework"
		end
        File.open( framework_path,"r") do |f|
            category = ""
            f.each_line do |l|
                if l.start_with? "/" then 
                    category = l.strip
                    @framework[category] = []

				# if "*", then add to replace list
                elsif l.start_with? "*" then
					org = l.strip.sub("*","").split[0]
					replace = l.strip.sub("*","").split[1] 

					if @substitutes[org].nil? then
                    	@substitutes[org]=[replace]
					else	
                    	@substitutes[org].push replace
					end
                else
					pkg_name = l.strip 
                    pkg = @package_list[pkg_name] 

                    if not pkg.nil? then
                        @framework[category].push pkg
                    end 
                end 
            end 
        end 
	end


	# append capi frameworks 
    def append_capi_frameworks
        #make capi package contents list
		(@package_list.keys.select {|x| x =~ /capi-.*-dev/}).each do |package|
            # capi-appfw-xxx-yyy
            case package.split('-')[1]
            when "appfw"  then category = "/Application"
            when "media" then category = "/Multimedia"
            when "ui" then category = "/UI"
            when "uix" then category = "/UIX"
            else category = "/" + package.split('-')[1].capitalize
            end 

			# append the packages to frameworks
            pkg = @package_list[package]
            if not pkg.nil?  then
                if @framework.has_key? category then
                    @framework[category].push pkg
                else
                    @framework[category] = [pkg]
                end
            end 
        end
    end


	# generate header to header dependency
	def generate_header2header_dependency

		# for all packages in list
		@package_list.each_value do |pkg|
			$log.info "Analyzing...#{pkg.name}"

			# get include path list(used SBS specialized function)
            pc_list = pkg.pc_list.map {|pc| pc.name}
			include_search_paths = @rootstrap.ctrl.get_include_search_paths_using_pkgconfig pc_list
			# check all ( include option +  dependent header files) combination paths
			pkg.header_list.each do |header|

				# all #include statement in the header file
				for dpath in header.include_list
					for ipath in include_search_paths 
						abs_path = @rootstrap.path + ipath + "/" + dpath
						abs_path.gsub!("//","/")
						# check the combination path exists, if exists, add pkg2pkg dependency
						if File.file? abs_path then

							# set header to header dependency
							path = (ipath + "/" + dpath).gsub("//","/")
							dep_header = @header_list[path]
							header.dep_header_list.push dep_header

							# set required include search path
							header.req_incpath_list.push ipath
							
							# if different package, update other dependency info
							dep_pkg = dep_header.parent
							if not pkg.equal? dep_pkg then
								pkg.dep_package_list.push dep_pkg
								pkg.dep_header_list.push dep_header
							end
							break
						end
					end
				end
				
				# remove dup		
				header.dep_header_list.uniq!
			end

			# remove dup		
			pkg.dep_package_list.uniq!
			pkg.dep_header_list.uniq!

		end
	end

	
	# create dev packages 
	def generate_dev_packages 	
		# init
	 	@devpackage_list = []
		@public_package_list = []

		# for all framework 
        @framework.each do |category,package_list|

			# process category 
			c = add_category_node category

			# for all packages in category
            package_list.each do |pkg|
				# add public package list
				@public_package_list.push pkg

				# for all pc name
                pkg.pc_list.each do |pc|
                    name = pc.name

					# if there is alias name, skip
					if is_alias_name? name then next end

					# if has alias names,
					if has_alias_name? name then 
						add_leaf_node(get_alias_name(name), pc,c)
					else 
						add_leaf_node(name,pc,c)
					end
                end
            end 
        end 

		# set pc dependency
		set_dev_package_dependency

		# resolve private header issue
		resolve_private_headers
	end


	# add new category node
	def add_category_node(category_pattern)

		parent_pattern = nil
		name = category_pattern
		if category_pattern.count("/") > 1 then
			name = category_pattern.split("/")[-1]
			new_index = category_pattern.length - (1 + name.length) - 1
			parent_pattern = category_pattern[0..new_index]
            c = DevPackage.new(name, false, true)
		else
			name = category_pattern[1..-1]
            c = DevPackage.new(name, true, true)
		end

		# if already exists in list, return
		devpkg = get_node(name)
		if not devpkg.nil? then 
			return devpkg
		end

		# if parent exist, set parent
		if not parent_pattern.nil? then
			parent = add_category_node( parent_pattern )
			parent.include_devpackage_list.push name
		end

		# add dev list
		@devpackage_list.push c
		$log.info "Framework category added: #{name}"

		return c
	end
	


	# get dev package
	def get_node(name)
		for devpkg in @devpackage_list
			if devpkg.name == name then
				return devpkg
			end
		end
		return nil	
	end


	# add new leaf node
	def add_leaf_node(name, pc, parent)
		# create new
        p = DevPackage.new(name, false, false)

		# set info
        pc.cflags_I.each     {|x| p.include_path_list.push x }
        pc.cflags_other.each {|x| p.other_cflag_list.push x }
        pc.lflags_l.each     {|x| p.library_list.push x } 
        pc.lflags_L.each     {|x| p.library_path_list.push x }
        pc.lflags_other.each {|x| p.other_lflag_list.push x } 
        p.description = pc.description

		# set parent
    	parent.include_devpackage_list.push name

		# add dev list
		@devpackage_list.push p
	end


	def is_alias_name? name
   		if @substitutes.values.include? name then
			return true
		else
			return false
		end
	end


	def has_alias_name? name
   		aliases = @substitutes[name]
		if aliases.nil? or aliases.empty? then
			return false
		else
			return true
		end
	end


	def get_alias_name name
   		aliases = @substitutes[name]
		if aliases.nil? then
			return name
		else
			return aliases[0]
		end
	end


	def get_original_pc_name aname
		org_name = aname
		@substitutes.each do |key,values|
			org_name = key if values.include? aname and not @pc_list[key].nil?
		end
		
		return org_name
	end

	
	# set dev package dependency  and get private header
	def set_dev_package_dependency
		private_headers = {}

		# for all dev packages
		for dev_pc in @devpackage_list
			# if not leaf node, skip
			if dev_pc.is_virtual  then next end

			# get real name
			pc_name = get_original_pc_name dev_pc.name
			# for all dependent headers	
			pkg = @pc_list[pc_name].parent
			for header in pkg.dep_header_list
				# get public pc list and private header list	
				tmp = propagate_header(header,[])
				
				# add public pc dependency	
				dev_pc.dependency_list=dev_pc.dependency_list | tmp[0]
	
				# add private headers
				pkg.priv_header_list = pkg.priv_header_list | tmp[1]
			end

			# remove dup
			dev_pc.dependency_list.uniq!
			pkg.priv_header_list.uniq!
		end

        # for the packages that has no pc file
        for pkg in @public_package_list
            if not pkg.priv_header_list.empty? then
                next
            end

			for header in pkg.dep_header_list
				# get private header list	
				tmp = propagate_header(header,[])
				
				# add private headers
				pkg.priv_header_list = pkg.priv_header_list | tmp[1]
			end

			# remove dup
			pkg.priv_header_list.uniq!
        end
	end


	# get public pc list and private header list
	def propagate_header(header, done_list)
		public_pc_list = []
		private_header_list = []	

		# if already done, skip
		if done_list.include? header.path then
			return [[],[]]			
		end
		done_list.push header.path

		# if public package header, add its pcs
		pkg = header.parent
		if @public_package_list.include? pkg then
			pkg.pc_list.each do |x|
				public_pc_list.push get_alias_name(x.name)
			end
		# otherwise, identify 
		else
			# add into list
			private_header_list.push header
			
			# propagate sub dependent headers
			for dep_header in header.dep_header_list
				result_pair = propagate_header(dep_header, done_list)
				public_pc_list = public_pc_list | result_pair[0]
				private_header_list = private_header_list | result_pair[1]
			end
		end
	
		public_pc_list.uniq!
		private_header_list.uniq!

		return [public_pc_list, private_header_list]
	end


	# if private headers exist, add inclcude search path to devpackage
	def resolve_private_headers
		# for all dev packages
		for dev_pc in @devpackage_list
			# if not leaf node, skip
			if dev_pc.is_virtual  then next end

			# get parent package name
			pc_name = get_original_pc_name dev_pc.name
			pkg = @pc_list[pc_name].parent

			# skip, if empty
			if pkg.priv_header_list.empty? then next end
	
			# for each private header
			for private_header in pkg.priv_header_list
				dev_pc.include_path_list=dev_pc.include_path_list | private_header.req_incpath_list
			end
			
			# remove dup
			dev_pc.include_path_list.uniq!	
		end	
	end
end
