#ifndef __DALI_TOOLKIT_IMAGE_VIEW_H__
#define __DALI_TOOLKIT_IMAGE_VIEW_H__

/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

// INTERNAL INCLUDES
#include <dali-toolkit/public-api/controls/control.h>

namespace Dali
{

namespace Toolkit
{

namespace Internal DALI_INTERNAL
{
class ImageView;
}
/**
 * @addtogroup dali_toolkit_controls_image_view
 * @{
 */

/**
 * @brief ImageView is a class for displaying an Image.
 * @since_tizen 2.4
 *
 */
class DALI_IMPORT_API ImageView : public Control
{
public:
  /**
   * @brief The start and end property ranges for this control.
   * @since_tizen 2.4
   */
  enum PropertyRange
  {
    PROPERTY_START_INDEX = Control::CONTROL_PROPERTY_END_INDEX + 1,
    PROPERTY_END_INDEX =   PROPERTY_START_INDEX + 1000              ///< Reserve property indices @since_tizen 2.4
  };

  /**
   * @brief An enumeration of properties belonging to the ImageView class.
   * @since_tizen 2.4
   */
  struct Property
  {
    enum
    {
      RESOURCE_URL = PROPERTY_START_INDEX, ///< type string @since_tizen 2.4 @see SetImage()
    };
  };

public:

  /**
   * @brief Create an uninitialized ImageView.
   * @since_tizen 2.4
   */
  ImageView();

  /**
   * @brief Create an initialized ImageView.
   *
   * @since_tizen 2.4
   * @return A handle to a newly allocated Dali ImageView.
   */
  static ImageView New();

  /**
   * @brief Create an initialized ImageView from an Image.
   *
   * If the handle is empty, ImageView will display nothing
   * @since_tizen 2.4
   * @param[in] image The Image to display.
   * @return A handle to a newly allocated ImageView.
   */
  static ImageView New( Image image );

  /**
   * @brief Create an initialized ImageView from an Image resource url
   *
   * If the string is empty, ImageView will display nothing
   * @since_tizen 2.4
   * @param[in] url The url of the image resource to display.
   * @return A handle to a newly allocated ImageView.
   */
  static ImageView New( const std::string& url );

  /**
   * @brief Destructor
   *
   * This is non-virtual since derived Handle types must not contain data or virtual methods.
   * @since_tizen 2.4
   */
  ~ImageView();

  /**
   * @brief Copy constructor.
   *
   * @since_tizen 2.4
   * @param[in] imageView ImageView to copy. The copied ImageView will point at the same implementation
   */
  ImageView( const ImageView& imageView );

  /**
   * @brief Assignment operator.
   *
   * @since_tizen 2.4
   * @param[in] imageView The ImageView to assign from.
   * @return The updated ImageView.
   */
  ImageView& operator=( const ImageView& imageView );

  /**
   * @brief Downcast a handle to ImageView handle.
   *
   * If handle points to a ImageView the downcast produces valid
   * handle. If not the returned handle is left uninitialized.
   *
   * @since_tizen 2.4
   * @param[in] handle Handle to an object
   * @return handle to a ImageView or an uninitialized handle
   */
  static ImageView DownCast( BaseHandle handle );

  /**
   * @brief Sets this ImageView from an Image
   *
   * If the handle is empty, ImageView will display nothing
   * @since_tizen 2.4
   * @param[in] image The Image to display.
   */
  void SetImage( Image image );

  /**
   * @brief Gets the Image
   *
   * @since_tizen 2.4
   * @remarks Calls to this method should be avoided as this may return an empty handle if the image has not been created yet.
   * @return The Image currently set to this ImageView
   */
  Image GetImage() const;

public: // Not intended for application developers

  /**
   * @brief Creates a handle using the Toolkit::Internal implementation.
   *
   * @since_tizen 2.4
   * @param[in]  implementation  The ImageView implementation.
   */
  DALI_INTERNAL ImageView( Internal::ImageView& implementation );

  /**
   * @brief Allows the creation of this ImageView from an Internal::CustomActor pointer.
   *
   * @since_tizen 2.4
   * @param[in]  internal  A pointer to the internal CustomActor.
   */
  DALI_INTERNAL ImageView( Dali::Internal::CustomActor* internal );

};

/**
 * @}
 */
} // namespace Toolkit

} // namespace Dali

#endif // __DALI_TOOLKIT_IMAGE_VIEW_H__
