#ifndef __DALI_TOOLKIT_TEXT_FIELD_H__
#define __DALI_TOOLKIT_TEXT_FIELD_H__

/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

// INTERNAL INCLUDES
#include <dali-toolkit/public-api/controls/control.h>

namespace Dali
{

namespace Toolkit
{

namespace Internal DALI_INTERNAL
{
class TextField;
}
/**
 * @addtogroup dali_toolkit_controls_text_controls
 * @{
 */

/**
 * @brief A control which provides a single-line editable text field.
 *
 * @since_tizen 2.4
 */
class DALI_IMPORT_API TextField : public Control
{
public:

  /**
   * @brief The start and end property ranges for this control.
   * @since_tizen 2.4
   */
  enum PropertyRange
  {
    PROPERTY_START_INDEX = Control::CONTROL_PROPERTY_END_INDEX + 1,
    PROPERTY_END_INDEX =   PROPERTY_START_INDEX + 1000              ///< Reserve property indices @since_tizen 2.4
  };

  /**
   * @brief An enumeration of properties belonging to the TextLabel class.
   * @since_tizen 2.4
   */
  struct Property
  {
    enum
    {
      RENDERING_BACKEND = PROPERTY_START_INDEX, ///< The type of rendering,                                                    type INT @since_tizen 2.4 @see RenderingType
      TEXT,                                     ///< The text to display in UTF-8 format,                                      type STRING @since_tizen 2.4
      PLACEHOLDER_TEXT,                         ///< The text to display when the TextField is empty and inactive,             type STRING @since_tizen 2.4
      PLACEHOLDER_TEXT_FOCUSED,                 ///< The text to display when the TextField is empty with key-input focus,     type STRING @since_tizen 2.4
      FONT_FAMILY,                              ///< The requested font family,                                                type STRING @since_tizen 2.4
      FONT_STYLE,                               ///< The requested font style,                                                 type STRING @since_tizen 2.4
      POINT_SIZE,                               ///< The size of font in points,                                               type FLOAT @since_tizen 2.4
      MAX_LENGTH,                               ///< The maximum number of characters that can be inserted,                    type INTEGER @since_tizen 2.4
      EXCEED_POLICY,                            ///< Specifies how the text is truncated when it does not fit,                 type INTEGER @since_tizen 2.4
      HORIZONTAL_ALIGNMENT,                     ///< The line horizontal alignment,                                            type STRING,  values "BEGIN", "CENTER", "END" @since_tizen 2.4
      VERTICAL_ALIGNMENT,                       ///< The line vertical alignment,                                              type STRING,  values   "TOP",   "CENTER",   "BOTTOM" @since_tizen 2.4
      TEXT_COLOR,                               ///< The text color,                                                           type VECTOR4 @since_tizen 2.4
      PLACEHOLDER_TEXT_COLOR,                   ///< The placeholder-text color,                                               type VECTOR4 @since_tizen 2.4
      SHADOW_OFFSET,                            ///< The drop shadow offset 0 indicates no shadow,                             type VECTOR2 @since_tizen 2.4
      SHADOW_COLOR,                             ///< The color of a drop shadow,                                               type VECTOR4 @since_tizen 2.4
      PRIMARY_CURSOR_COLOR,                     ///< The color to apply to the primary cursor,                                 type VECTOR4 @since_tizen 2.4
      SECONDARY_CURSOR_COLOR,                   ///< The color to apply to the secondary cursor,                               type VECTOR4 @since_tizen 2.4
      ENABLE_CURSOR_BLINK,                      ///< Whether the cursor should blink or not,                                   type BOOLEAN @since_tizen 2.4
      CURSOR_BLINK_INTERVAL,                    ///< The time interval in seconds between cursor on/off states,                type FLOAT @since_tizen 2.4
      CURSOR_BLINK_DURATION,                    ///< The cursor will stop blinking after this number of seconds (if non-zero), type FLOAT @since_tizen 2.4
      CURSOR_WIDTH,                             ///< The cursor width,                                                         type INTEGER @since_tizen 2.4
      GRAB_HANDLE_IMAGE,                        ///< The image to display for the grab handle,                                 type STRING @since_tizen 2.4
      GRAB_HANDLE_PRESSED_IMAGE,                ///< The image to display when the grab handle is pressed,                     type STRING @since_tizen 2.4
      SCROLL_THRESHOLD,                         ///< Scrolling will occur if the cursor is this close to the control border,   type FLOAT @since_tizen 2.4
      SCROLL_SPEED,                             ///< The scroll speed in pixels per second,                                    type FLOAT @since_tizen 2.4
      SELECTION_HANDLE_IMAGE_LEFT,              ///< The image to display for the left selection handle,                       type MAP @since_tizen 2.4
      SELECTION_HANDLE_IMAGE_RIGHT,             ///< The image to display for the right selection handle,                      type MAP @since_tizen 2.4
      SELECTION_HANDLE_PRESSED_IMAGE_LEFT,      ///< The image to display when the left selection handle is pressed,           type MAP @since_tizen 2.4
      SELECTION_HANDLE_PRESSED_IMAGE_RIGHT,     ///< The image to display when the right selection handle is pressed,          type MAP @since_tizen 2.4
      SELECTION_HANDLE_MARKER_IMAGE_LEFT,       ///< The image to display for the left selection handle marker,                type MAP @since_tizen 2.4
      SELECTION_HANDLE_MARKER_IMAGE_RIGHT,      ///< The image to display for the right selection handle marker,               type MAP @since_tizen 2.4
      SELECTION_HIGHLIGHT_COLOR,                ///< The color of the selection highlight,                                     type VECTOR4 @since_tizen 2.4
      DECORATION_BOUNDING_BOX,                  ///< The decorations (handles etc) will positioned within this area on-screen, type RECTANGLE @since_tizen 2.4
      INPUT_METHOD_SETTINGS                     ///< The settings to relating to the System's Input Method, Key and Value      type MAP @since_tizen 2.4
    };
  };

  /**
   * @brief Specifies how the text is truncated when it does not fit
   *
   * The default value is \e EXCEED_POLICY_CLIP.
   * @since_tizen 2.4
   */
  enum ExceedPolicy
  {
    EXCEED_POLICY_ORIGINAL,        ///< The text will be display at original size, and may exceed the TextField boundary. @since_tizen 2.4
    EXCEED_POLICY_CLIP             ///< The end of text will be clipped to fit within the TextField. @since_tizen 2.4
  };

  // Type Defs

  /// @brief Text changed signal type
  /// @since_tizen 2.4
  typedef Signal<void ( TextField ) > TextChangedSignalType;

  /// @brief Max characters exceed signal type
  /// @since_tizen 2.4
  typedef Signal<void ( TextField ) > MaxLengthReachedSignalType;

  /**
   * @brief Create the TextField control.
   * @since_tizen 2.4
   * @return A handle to the TextField control.
   */
  static TextField New();

  /**
   * @brief Creates an empty handle.
   * @since_tizen 2.4
   */
  TextField();

  /**
   * @brief Copy constructor.
   *
   * @since_tizen 2.4
   * @param[in] handle The handle to copy from.
   */
  TextField( const TextField& handle );

  /**
   * @brief Assignment operator.
   *
   * @since_tizen 2.4
   * @param[in] handle The handle to copy from.
   * @return A reference to this.
   */
  TextField& operator=( const TextField& handle );

  /**
   * @brief Destructor
   *
   * This is non-virtual since derived Handle types must not contain data or virtual methods.
   * @since_tizen 2.4
   */
  ~TextField();

  /**
   * @brief Downcast a handle to TextField handle.
   *
   * If the BaseHandle points is a TextField the downcast returns a valid handle.
   * If not the returned handle is left empty.
   *
   * @since_tizen 2.4
   * @param[in] handle Handle to an object
   * @return handle to a TextField or an empty handle
   */
  static TextField DownCast( BaseHandle handle );

  // Signals

  /**
   * @brief This signal is emitted when the text changes.
   *
   * A callback of the following type may be connected:
   * @code
   *   void YourCallbackName( TextField textField );
   * @endcode
   * @since_tizen 2.4
   * @return The signal to connect to.
   */
  TextChangedSignalType& TextChangedSignal();

  /**
   * @brief This signal is emitted when inserted text exceeds the maximum character limit.
   *
   * A callback of the following type may be connected:
   * @code
   *   void YourCallbackName( TextField textField );
   * @endcode
   * @since_tizen 2.4
   * @return The signal to connect to.
   */
  MaxLengthReachedSignalType& MaxLengthReachedSignal();

public: // Not intended for application developers

  /**
   * @brief Creates a handle using the Toolkit::Internal implementation.
   *
   * @since_tizen 2.4
   * @param[in] implementation The Control implementation.
   */
  DALI_INTERNAL TextField( Internal::TextField& implementation );

  /**
   * @brief Allows the creation of this Control from an Internal::CustomActor pointer.
   *
   * @since_tizen 2.4
   * @param[in]  internal  A pointer to the internal CustomActor.
   */
  explicit DALI_INTERNAL TextField( Dali::Internal::CustomActor* internal );
};

/**
 * @}
 */
} // namespace Toolkit

} // namespace Dali

#endif // __DALI_TOOLKIT_TEXT_FIELD_H__
