#ifndef __DALI_TOOLKIT_TEXT_LABEL_H__
#define __DALI_TOOLKIT_TEXT_LABEL_H__

/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

// INTERNAL INCLUDES
#include <dali-toolkit/public-api/controls/control.h>

namespace Dali
{

namespace Toolkit
{

namespace Internal DALI_INTERNAL
{
class TextLabel;
}
/**
 * @addtogroup dali_toolkit_controls_text_controls
 * @{
 */

/**
 * @brief A control which renders a short text string.
 *
 * Text labels are lightweight, non-editable and do not respond to user input.
 * @since_tizen 2.4
 */
class DALI_IMPORT_API TextLabel : public Control
{
public:

  /**
   * @brief The start and end property ranges for this control.
   * @since_tizen 2.4
   */
  enum PropertyRange
  {
    PROPERTY_START_INDEX = Control::CONTROL_PROPERTY_END_INDEX + 1,
    PROPERTY_END_INDEX =   PROPERTY_START_INDEX + 1000              ///< Reserve property indices @since_tizen 2.4
  };

  /**
   * @brief An enumeration of properties belonging to the TextLabel class.
   * @since_tizen 2.4
   */
  struct Property
  {
    enum
    {
      RENDERING_BACKEND = PROPERTY_START_INDEX, ///< The type of rendering,                                                    type INT @since_tizen 2.4 @see RenderingType
      TEXT,                                     ///< The text to display in UTF-8 format,              type STRING @since_tizen 2.4
      FONT_FAMILY,                              ///< The requested font family,                        type STRING @since_tizen 2.4
      FONT_STYLE,                               ///< The requested font style,                         type STRING @since_tizen 2.4
      POINT_SIZE,                               ///< The size of font in points,                       type FLOAT @since_tizen 2.4
      MULTI_LINE,                               ///< The single-line or multi-line layout option,      type BOOLEAN @since_tizen 2.4
      HORIZONTAL_ALIGNMENT,                     ///< The line horizontal alignment,                    type STRING,  values "BEGIN", "CENTER", "END" @since_tizen 2.4
      VERTICAL_ALIGNMENT,                       ///< The line vertical alignment,                      type STRING,  values   "TOP",   "CENTER",   "BOTTOM" @since_tizen 2.4
      TEXT_COLOR,                               ///< The text color,                                   type VECTOR4 @since_tizen 2.4
      SHADOW_OFFSET,                            ///< The drop shadow offset 0 indicates no shadow,     type VECTOR2 @since_tizen 2.4
      SHADOW_COLOR,                             ///< The color of a drop shadow,                       type VECTOR4 @since_tizen 2.4
      UNDERLINE_ENABLED,                        ///< The underline enabled flag,                       type BOOLEAN @since_tizen 2.4
      UNDERLINE_COLOR,                          ///< The color of the underline,                       type VECTOR4 @since_tizen 2.4
      UNDERLINE_HEIGHT                          ///< Overrides the underline height from font metrics, type FLOAT @since_tizen 2.4
    };
  };

  /**
   * @brief Create the TextLabel control.
   *
   * @since_tizen 2.4
   * @return A handle to the TextLabel control.
   */
  static TextLabel New();

  /**
   * @brief Create the TextLabel control.
   *
   * @since_tizen 2.4
   * @param[in] text The text to display.
   * @return A handle to the TextLabel control.
   */
  static TextLabel New( const std::string& text );

  /**
   * @brief Creates an empty handle.
   * @since_tizen 2.4
   */
  TextLabel();

  /**
   * @brief Copy constructor.
   *
   * @since_tizen 2.4
   * @param[in] handle The handle to copy from.
   */
  TextLabel( const TextLabel& handle );

  /**
   * @brief Assignment operator.
   *
   * @since_tizen 2.4
   * @param[in] handle The handle to copy from.
   * @return A reference to this.
   */
  TextLabel& operator=( const TextLabel& handle );

  /**
   * @brief Destructor
   *
   * This is non-virtual since derived Handle types must not contain data or virtual methods.
   * @since_tizen 2.4
   */
  ~TextLabel();

  /**
   * @brief Downcast a handle to TextLabel handle.
   *
   * If the BaseHandle points is a TextLabel the downcast returns a valid handle.
   * If not the returned handle is left empty.
   *
   * @since_tizen 2.4
   * @param[in] handle Handle to an object
   * @return handle to a TextLabel or an empty handle
   */
  static TextLabel DownCast( BaseHandle handle );

public: // Not intended for application developers

  /**
   * @brief Creates a handle using the Toolkit::Internal implementation.
   *
   * @since_tizen 2.4
   * @param[in] implementation The Control implementation.
   */
  DALI_INTERNAL TextLabel( Internal::TextLabel& implementation );

  /**
   * @brief Allows the creation of this Control from an Internal::CustomActor pointer.
   *
   * @since_tizen 2.4
   * @param[in]  internal  A pointer to the internal CustomActor.
   */
  explicit DALI_INTERNAL TextLabel( Dali::Internal::CustomActor* internal );
};

/**
 * @}
 */
} // namespace Toolkit

} // namespace Dali

#endif // __DALI_TOOLKIT_TEXT_LABEL_H__
