#ifndef __DALI_WINDOW_H__
#define __DALI_WINDOW_H__

/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

// EXTERNAL INCLUDES
#include <string>
#include <dali/public-api/math/rect.h>
#include <dali/public-api/math/vector2.h>
#include <dali/public-api/object/base-handle.h>
#include <dali/public-api/object/any.h>
#include <dali/public-api/signals/dali-signal.h>

namespace Dali
{
/**
 * @addtogroup dali_adaptor_framework
 * @{
 */

typedef Dali::Rect<int> PositionSize; ///< Type for widow poisition and size. @since_tizen 2.4

namespace Internal DALI_INTERNAL
{
namespace Adaptor
{
class Window;
}
}

class DragAndDropDetector;
class Orientation;

/**
 * @brief The window class is used internally for drawing.
 *
 * A Window has an orientation and indicator properties.
 * You can get a valid Window handle by calling Dali::Application::GetWindow().
 * @since_tizen 2.4
 */
class DALI_IMPORT_API Window : public BaseHandle
{
public:
  typedef Signal< void (bool) > IndicatorSignalType;  ///< Indicator signal callback type @since_tizen 2.4

public:

  // Enumerations

  /**
   * @brief Orientation of the window.
   * @since_tizen 2.4
   */
  enum WindowOrientation
  {
    PORTRAIT = 0,  ///< Portrait orientation @since_tizen 2.4
    LANDSCAPE = 90,  ///< Landscape orientation @since_tizen 2.4
    PORTRAIT_INVERSE = 180,  ///< Portrait inverse orientation @since_tizen 2.4
    LANDSCAPE_INVERSE = 270  ///< Landscape inverse orientation @since_tizen 2.4
  };

  /**
   * @brief Opacity of the indicator.
   * @since_tizen 2.4
   */
  enum IndicatorBgOpacity
  {
    OPAQUE = 100, ///< Fully opaque indicator Bg @since_tizen 2.4
    TRANSLUCENT = 50, ///< Semi translucent indicator Bg @since_tizen 2.4
    TRANSPARENT = 0 ///< Fully transparent indicator Bg @since_tizen 2.4
  };

  /**
   * @brief Visible mode of the indicator.
   * @since_tizen 2.4
   */
  enum IndicatorVisibleMode
  {
    INVISIBLE = 0, ///< Hide indicator @since_tizen 2.4
    VISIBLE = 1, ///< Show indicator @since_tizen 2.4
    AUTO = 2 ///< Hide in default, will show when necessary @since_tizen 2.4
  };

  // Methods

  /**
   * @brief Create an initialized handle to a new Window.
   * @since_tizen 2.4
   * @param[in] windowPosition The position and size of the window
   * @param[in] name The window title
   * @param[in] isTransparent Whether window is transparent
   * @return A new window
   */
  static Window New(PositionSize windowPosition, const std::string& name, bool isTransparent = false);

  /**
   * @brief Create an initialized handle to a new Window.
   * @since_tizen 2.4
   * @param[in] windowPosition The position and size of the window
   * @param[in] name The window title
   * @param[in] className The window class name
   * @param[in] isTransparent Whether window is transparent
   * @return A new window
   */
  static Window New(PositionSize windowPosition, const std::string& name, const std::string& className, bool isTransparent = false);

  /**
   * @brief Create an uninitalized handle.
   *
   * This can be initialized using Dali::Window::New()
   * @since_tizen 2.4
   */
  Window();

  /**
   * @brief Destructor
   *
   * This is non-virtual since derived Handle types must not contain data or virtual methods.
   * @since_tizen 2.4
   */
  ~Window();

  /**
   * @brief This copy constructor is required for (smart) pointer semantics.
   *
   * @since_tizen 2.4
   * @param [in] handle A reference to the copied handle
   */
  Window(const Window& handle);

  /**
   * @brief This assignment operator is required for (smart) pointer semantics.
   *
   * @since_tizen 2.4
   * @param [in] rhs  A reference to the copied handle
   * @return A reference to this
   */
  Window& operator=(const Window& rhs);

  /**
   * @brief This sets whether the indicator bar should be shown or not.
   * @since_tizen 2.4
   * @param[in] visibleMode Visible mode for indicator bar, VISIBLE in default
   */
  void ShowIndicator( IndicatorVisibleMode visibleMode );

  /**
   * @brief This sets the opacity mode of indicator bar.
   * @since_tizen 2.4
   * @param[in] opacity The opacity mode
   */
  void SetIndicatorBgOpacity( IndicatorBgOpacity opacity );

  /**
   * @brief This sets the orientation of indicator bar.
   *
   * It does not implicitly show the indicator if it is currently
   * hidden.
   * @since_tizen 2.4
   * @param[in] orientation The orientation
   */
  void RotateIndicator(WindowOrientation orientation);

  /**
   * @brief Set the window name and class string.
   * @since_tizen 2.4
   * @param[in] name The name of the window
   * @param[in] klass The class of the window
   */
  void SetClass(std::string name, std::string klass);

  /**
   * @brief Raise window to top of window stack.
   * @since_tizen 2.4
   */
  void Raise();

  /**
   * @brief Lower window to bottom of window stack.
   * @since_tizen 2.4
   */
  void Lower();

  /**
   * @brief Activate window to top of window stack even it is iconified.
   * @since_tizen 2.4
   */
  void Activate();

  /**
   * @brief Add an orientation to the list of available orientations.
   * @since_tizen 2.4
   */
  void AddAvailableOrientation( WindowOrientation orientation );

  /**
   * @brief Remove an orientation from the list of available orientations.
   * @since_tizen 2.4
   */
  void RemoveAvailableOrientation( WindowOrientation orientation );

  /**
   * @brief Set a preferred orientation.
   * @since_tizen 2.4
   * @param[in] orientation The preferred orientation
   * @pre Orientation is in the list of available orientations
   */
  void SetPreferredOrientation( WindowOrientation orientation );

  /**
   * @brief Get the preferred orientation.
   * @since_tizen 2.4
   * @return The preferred orientation if previously set, or none.
   */
  WindowOrientation GetPreferredOrientation();

  /**
   * @brief Returns the Drag & drop detector which can be used to receive drag & drop events.
   * @since_tizen 2.4
   * @return A handle to the DragAndDropDetector.
   */
  DragAndDropDetector GetDragAndDropDetector() const;

  /**
   * @brief Get the native handle of the window.
   * @since_tizen 2.4
   * @return The native handle of the window or an empty handle.
   */
  Any GetNativeHandle() const;

public: // Signals
  /**
   * @brief The user should connect to this signal to get a timing when indicator was shown / hidden.
   * @since_tizen 2.4
   */
  IndicatorSignalType& IndicatorVisibilityChangedSignal();

public: // Not intended for application developers
  /**
   * @brief This constructor is used by Dali::Application::GetWindow().
   * @since_tizen 2.4
   * @param[in] window A pointer to the window.
   */
  explicit DALI_INTERNAL Window( Internal::Adaptor::Window* window );
};

/**
 * @}
 */
} // namespace Dali

#endif // __DALI_WINDOW_H__
