#ifndef __DALI_ENCODED_BUFFER_IMAGE_H__
#define __DALI_ENCODED_BUFFER_IMAGE_H__

/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

// EXTERNAL INCLUDES
#include <stdint.h>

// INTERNAL INCLUDES
#include <dali/public-api/images/image.h>
#include <dali/public-api/images/image-operations.h>
#include <dali/public-api/math/uint-16-pair.h>

namespace Dali
{
/**
 * @addtogroup dali_core_images
 * @{
 */

namespace Internal DALI_INTERNAL
{
class EncodedBufferImage;
}

typedef Uint16Pair ImageDimensions; ///< Image dimension type @since_tizen 2.4


/**
 * @brief EncodedBufferImage represents an image resource that can be added to
 * ImageViews.
 *
 * A memory buffer of encoded image data is provided by the application and
 * decoded asynchronously on a background thread to fill the image's
 * pixel data.
 *
 * The buffer of data provided to an EncodedBufferImage factory function
 * (New) should be filled with encoded image data in one of %Dali's
 * supported image file formats, with the sequence of bytes in the buffer
 * exactly matching the sequence of bytes that a file in the file system
 * holding the encoded image data would have.
 *
 * The application may free the encoded image buffer passed to one of the
 * New() static factory member functions as soon as they return.
 *
 * <h3> Signals </h3>
 *
 * Image::UploadedSignal is emitted when the decoded image data gets
 * uploaded to the OpenGL ES implementation.
 * @since_tizen 2.4
 */
class DALI_IMPORT_API EncodedBufferImage : public Image
{
public:
  /**
   * @brief Constructor which creates an uninitialized EncodedBufferImage object.
   *
   * Use @ref Image::New to create an initialised object.
   * @since_tizen 2.4
   */
  EncodedBufferImage();

  /**
   * @brief Create an initialised image object from an encoded image buffer in memory.
   *
   * @since_tizen 2.4
   * @param [in] encodedImage The encoded bytes of an image, in a supported
   * image format such as PNG, JPEG, GIF, BMP, KTX, ICO, and WBMP, organised
   * exactly as it would be as a file in the filesystem.
   * The caller retains ownership of this buffer and is free to modify or
   * discard it as soon as the function returns.
   * @param [in] encodedImageByteCount The size in bytes of the buffer pointed to
   * by encodedImage.
   * @return A handle to a newly allocated object.
   */
  static EncodedBufferImage New( const uint8_t * const encodedImage, std::size_t encodedImageByteCount );

  /**
   * @brief Downcast a handle to EncodedBufferImage handle.
   *
   * If handle points to a EncodedBufferImage the
   * downcast produces valid handle. If not the returned handle is left uninitialized.
   * @since_tizen 2.4
   * @param[in] handle Handle to an object
   * @return Handle to a EncodedBufferImage or an uninitialized handle
   */
  static EncodedBufferImage DownCast( BaseHandle handle );

  /**
   * @brief Destructor
   *
   * This is non-virtual since derived Handle types must not contain data or virtual methods.
   * @since_tizen 2.4
   */
  ~EncodedBufferImage();

  /**
   * @brief This copy constructor is required for (smart) pointer semantics.
   *
   * @since_tizen 2.4
   * @param [in] handle A reference to the copied handle
   */
  EncodedBufferImage(const EncodedBufferImage& handle);

  /**
   * @brief This assignment operator is required for (smart) pointer semantics.
   *
   * @since_tizen 2.4
   * @param [in] rhs  A reference to the copied handle
   * @return A reference to this
   */
  EncodedBufferImage& operator=(const EncodedBufferImage& rhs);

public: // Not intended for application developers

  explicit DALI_INTERNAL EncodedBufferImage(Internal::EncodedBufferImage*);
};

/**
 * @}
 */
} // namespace Dali

#endif // __DALI_ENCODED_BUFFER_IMAGE_H__
