#ifndef _ELM_FLIP_EO_LEGACY_H_
#define _ELM_FLIP_EO_LEGACY_H_

#ifndef _ELM_FLIP_EO_CLASS_TYPE
#define _ELM_FLIP_EO_CLASS_TYPE

typedef Eo Elm_Flip;

#endif

#ifndef _ELM_FLIP_EO_TYPES
#define _ELM_FLIP_EO_TYPES


#endif

/**
 *
 * @MOBILE_ONLY
 *
 * @brief Set the interactive flip mode
 *
 * @if MOBILE @since_tizen 2.3
 * @endif
 *
 * @remark This sets if the flip should be interactive (allow user to click and
 * drag a side of the flip to reveal the back page and cause it to flip).
 * By default a flip is not interactive. You may also need to set which
 * sides of the flip are "active" for flipping and how much space they use
 * (a minimum of a finger size) with elm_flip_interaction_direction_enabled_set()
 * and elm_flip_interaction_direction_hitsize_set()
 *
 * @remark The four available mode of interaction are:
 * @li #ELM_FLIP_INTERACTION_NONE - No interaction is allowed
 * @li #ELM_FLIP_INTERACTION_ROTATE - Interaction will cause rotate animation
 * @li #ELM_FLIP_INTERACTION_CUBE - Interaction will cause cube animation
 * @li #ELM_FLIP_INTERACTION_PAGE - Interaction will cause page animation
 *
 * @remark #ELM_FLIP_INTERACTION_ROTATE won't cause
 * #ELM_FLIP_ROTATE_XZ_CENTER_AXIS or #ELM_FLIP_ROTATE_YZ_CENTER_AXIS to
 * happen, those can only be achieved with elm_flip_go();
 *
 * @ingroup Flip
 *
 * @param[in] obj The elm flip object
 * @param[in] mode The interactive flip mode to use
 */
EAPI void elm_flip_interaction_set(Elm_Flip *obj, Elm_Flip_Interaction mode);

/**
 *
 * @MOBILE_ONLY
 *
 * @brief Get the interactive flip mode
 *
 * @if MOBILE @since_tizen 2.3
 * @endif
 *
 * @return The interactive flip mode
 *
 * @remark Returns the interactive flip mode set by elm_flip_interaction_set()
 *
 * @ingroup Flip
 *
 * @param[in] obj The elm flip object
 */
EAPI Elm_Flip_Interaction elm_flip_interaction_get(const Elm_Flip *obj);

/**
 *
 * @MOBILE_ONLY
 *
 * @brief Get flip front visibility state
 *
 * @if MOBILE @since_tizen 2.3
 * @endif
 *
 * @return EINA_TRUE if front front is showing, EINA_FALSE if the back is
 * showing.
 *
 * @ingroup Flip
 *
 * @param[in] obj The elm flip object
 */
EAPI Eina_Bool elm_flip_front_visible_get(const Elm_Flip *obj);

/**
 *
 * @MOBILE_ONLY
 *
 * @brief Set the amount of the flip that is sensitive to interactive flip
 *
 * @if MOBILE @since_tizen 2.3
 * @endif
 *
 * @remark Set the amount of the flip that is sensitive to interactive flip, with 0
 * representing no area in the flip and 1 representing the entire flip. There
 * is however a consideration to be made in that the area will never be
 * smaller than the finger size set (as set in your Elementary configuration),
 * and dragging must always start from the opposite half of the flip (eg. right
 * half of the flip when dragging to the left).
 *
 * @remark Note that the @c dir parameter is not actually related to the direction of
 * the drag, it only refers to the area in the flip where interaction can
 * occur (top, bottom, left, right).
 *
 * @remark Negative values of @c hitsize will disable this hit area.
 *
 * @see elm_flip_interaction_set()
 *
 * @ingroup Flip
 * 
 *
 * @param[in] obj The elm flip object
 * @param[in] dir The hit area to set
 * @param[in] hitsize The amount of that dimension (0.0 to 1.0) to use
 */
EAPI void elm_flip_interaction_direction_hitsize_set(Elm_Flip *obj, Elm_Flip_Direction dir, double hitsize);

/**
 *
 * @MOBILE_ONLY
 * @brief Get the amount of the flip that is sensitive to interactive flip
 *
 * @if MOBILE @since_tizen 2.3
 * @endif
 *
 * @return The size set for that direction
 *
 * @remark Returns the amount of sensitive area set by
 * elm_flip_interaction_direction_hitsize_set().
 *
 * @ingroup Flip
 * 
 *
 * @param[in] obj The elm flip object
 * @param[in] dir The direction to check
 */
EAPI double elm_flip_interaction_direction_hitsize_get(Elm_Flip *obj, Elm_Flip_Direction dir);

/**
 *
 * @MOBILE_ONLY
 *
 * @brief Set which directions of the flip respond to interactive flip
 *
 * @if MOBILE @since_tizen 2.3
 * @endif
 *
 * @remark By default all directions are disabled, so you may want to enable the
 * desired directions for flipping if you need interactive flipping. You must
 * call this function once for each direction that should be enabled.
 *
 * @remark You can also set the appropriate hit area size by calling
 * @c elm_flip_interaction_direction_hitsize_set(). By default, a minimum
 * hit area will be created on the opposite edge of the flip.
 *
 * @see elm_flip_interaction_set()
 *
 * @ingroup Flip
 * 
 *
 * @param[in] obj The elm flip object
 * @param[in] dir The direction to change
 * @param[in] enabled If that direction is enabled or not
 */
EAPI void elm_flip_interaction_direction_enabled_set(Elm_Flip *obj, Elm_Flip_Direction dir, Eina_Bool enabled);

/**
 *
 * @MOBILE_ONLY
 *
 * @brief Get the enabled state of that flip direction
 *
 * @if MOBILE @since_tizen 2.3
 * @endif
 *
 * @return If that direction is enabled or not
 *
 * @remark Get the enabled state set by elm_flip_interaction_direction_enabled_set()
 *
 * @see elm_flip_interaction_set()
 *
 * @ingroup Flip
 * 
 *
 * @param[in] obj The elm flip object
 * @param[in] dir The direction to check
 */
EAPI Eina_Bool elm_flip_interaction_direction_enabled_get(Elm_Flip *obj, Elm_Flip_Direction dir);

/**
 *
 * @MOBILE_ONLY
 *
 * @brief Runs the flip animation
 *
 * @if MOBILE @since_tizen 2.3
 * @endif
 *
 * @remark Flips the front and back contents using the @p mode animation. This
 * effectively hides the currently visible content and shows the hidden one.
 *
 * @remark There a number of possible animations to use for the flipping:
 * @li #ELM_FLIP_ROTATE_X_CENTER_AXIS - Rotate the currently visible content
 * around a horizontal axis in the middle of its height, the other content
 * is shown as the other side of the flip.
 * @li #ELM_FLIP_ROTATE_Y_CENTER_AXIS - Rotate the currently visible content
 * around a vertical axis in the middle of its width, the other content is
 * shown as the other side of the flip.
 * @li #ELM_FLIP_ROTATE_XZ_CENTER_AXIS - Rotate the currently visible content
 * around a diagonal axis in the middle of its width, the other content is
 * shown as the other side of the flip.
 * @li #ELM_FLIP_ROTATE_YZ_CENTER_AXIS - Rotate the currently visible content
 * around a diagonal axis in the middle of its height, the other content is
 * shown as the other side of the flip.
 * @li #ELM_FLIP_CUBE_LEFT - Rotate the currently visible content to the left
 * as if the flip was a cube, the other content is show as the right face of
 * the cube.
 * @li #ELM_FLIP_CUBE_RIGHT - Rotate the currently visible content to the
 * right as if the flip was a cube, the other content is show as the left
 * face of the cube.
 * @li #ELM_FLIP_CUBE_UP - Rotate the currently visible content up as if the
 * flip was a cube, the other content is show as the bottom face of the cube.
 * @li #ELM_FLIP_CUBE_DOWN - Rotate the currently visible content down as if
 * the flip was a cube, the other content is show as the upper face of the
 * cube.
 * @li #ELM_FLIP_PAGE_LEFT - Move the currently visible content to the left as
 * if the flip was a book, the other content is shown as the page below that.
 * @li #ELM_FLIP_PAGE_RIGHT - Move the currently visible content to the right
 * as if the flip was a book, the other content is shown as the page below
 * that.
 * @li #ELM_FLIP_PAGE_UP - Move the currently visible content up as if the
 * flip was a book, the other content is shown as the page below that.
 * @li #ELM_FLIP_PAGE_DOWN - Move the currently visible content down as if the
 * flip was a book, the other content is shown as the page below that.
 *
 * @image html elm_flip.png
 * @image latex elm_flip.eps width=\textwidth
 *
 * @see elm_flip_go_to()
 *
 * @ingroup Flip
 * 
 *
 * @param[in] obj The elm flip object
 * @param[in] mode The mode type
 */
EAPI void elm_flip_go(Elm_Flip *obj, Elm_Flip_Mode mode);

/**
 *
 * @MOBILE_ONLY
 *
 * @brief Runs the flip animation to front or back.
 *
 * @if MOBILE @since_tizen 2.3
 * @endif
 *
 * @remark Flips the front and back contents using the @p mode animation. This
 * effectively hides the currently visible content and shows the hidden one.
 *
 * @remark There a number of possible animations to use for the flipping:
 * @li #ELM_FLIP_ROTATE_X_CENTER_AXIS - Rotate the currently visible content
 * around a horizontal axis in the middle of its height, the other content
 * is shown as the other side of the flip.
 * @li #ELM_FLIP_ROTATE_Y_CENTER_AXIS - Rotate the currently visible content
 * around a vertical axis in the middle of its width, the other content is
 * shown as the other side of the flip.
 * @li #ELM_FLIP_ROTATE_XZ_CENTER_AXIS - Rotate the currently visible content
 * around a diagonal axis in the middle of its width, the other content is
 * shown as the other side of the flip.
 * @li #ELM_FLIP_ROTATE_YZ_CENTER_AXIS - Rotate the currently visible content
 * around a diagonal axis in the middle of its height, the other content is
 * shown as the other side of the flip.
 * @li #ELM_FLIP_CUBE_LEFT - Rotate the currently visible content to the left
 * as if the flip was a cube, the other content is show as the right face of
 * the cube.
 * @li #ELM_FLIP_CUBE_RIGHT - Rotate the currently visible content to the
 * right as if the flip was a cube, the other content is show as the left
 * face of the cube.
 * @li #ELM_FLIP_CUBE_UP - Rotate the currently visible content up as if the
 * flip was a cube, the other content is show as the bottom face of the cube.
 * @li #ELM_FLIP_CUBE_DOWN - Rotate the currently visible content down as if
 * the flip was a cube, the other content is show as the upper face of the
 * cube.
 * @li #ELM_FLIP_PAGE_LEFT - Move the currently visible content to the left as
 * if the flip was a book, the other content is shown as the page below that.
 * @li #ELM_FLIP_PAGE_RIGHT - Move the currently visible content to the right
 * as if the flip was a book, the other content is shown as the page below
 * that.
 * @li #ELM_FLIP_PAGE_UP - Move the currently visible content up as if the
 * flip was a book, the other content is shown as the page below that.
 * @li #ELM_FLIP_PAGE_DOWN - Move the currently visible content down as if the
 * flip was a book, the other content is shown as the page below that.
 *
 * @image html elm_flip.png
 * @image latex elm_flip.eps width=\textwidth
 *
 * @since 1.7
 *
 * @ingroup Flip
 * 
 *
 * @param[in] obj The elm flip object
 * @param[in] front if @c EINA_TRUE, makes front visible, otherwise makes back.
 * @param[in] mode The mode type
 */
EAPI void elm_flip_go_to(Elm_Flip *obj, Eina_Bool front, Elm_Flip_Mode mode);

#endif
