#ifndef _EVAS_GRID_EO_LEGACY_H_
#define _EVAS_GRID_EO_LEGACY_H_

#ifndef _EVAS_GRID_EO_CLASS_TYPE
#define _EVAS_GRID_EO_CLASS_TYPE

typedef Eo Evas_Grid;

#endif

#ifndef _EVAS_GRID_EO_TYPES
#define _EVAS_GRID_EO_TYPES


#endif

/**
 *
 * @brief Sets the mirrored mode of the grid. In mirrored mode the grid items go
 * from right to left instead of left to right. That is, 0,0 is top right, not
 * to left.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @ingroup Evas_Object_Grid
 *
 * @since 1.1
 *
 * @param[in] obj The grid object
 * @param[in] mirrored the mirrored mode to set
 */
EAPI void evas_object_grid_mirrored_set(Evas_Grid *obj, Eina_Bool mirrored);

/**
 *
 * @brief Gets the mirrored mode of the grid.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @ingroup Evas_Object_Grid
 *
 * @return @c EINA_TRUE if it's a mirrored grid, @c EINA_FALSE otherwise.
 * @see evas_object_grid_mirrored_set()
 * @since 1.1
 *
 * @param[in] obj The grid object
 */
EAPI Eina_Bool evas_object_grid_mirrored_get(const Evas_Grid *obj);

/**
 *
 * @brief Set the virtual resolution for the grid
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @ingroup Evas_Object_Grid
 *
 * @since 1.1
 *
 * @param[in] obj The grid object
 * @param[in] w The virtual horizontal size (resolution) in integer units
 * @param[in] h The virtual vertical size (resolution) in integer units
 */
EAPI void evas_object_grid_size_set(Evas_Grid *obj, int w, int h);

/**
 *
 * @brief Get the current virtual resolution
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @ingroup Evas_Object_Grid
 *
 * @see evas_object_grid_size_set()
 * @since 1.1
 *
 * @param[in] obj The grid object
 * @param[out] w The virtual horizontal size (resolution) in integer units
 * @param[out] h The virtual vertical size (resolution) in integer units
 */
EAPI void evas_object_grid_size_get(const Evas_Grid *obj, int *w, int *h);

/**
 *
 * @brief Get the list of children for the grid.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @ingroup Evas_Object_Grid
 *
 * @remark This is a duplicate of the list kept by the grid internally.
 * It's up to the user to destroy it when it no longer needs it.
 * It's possible to remove objects from the grid when walking this
 * list, but these removals won't be reflected on it.
 * @since 1.1
 *
 * @param[in] obj The grid object
 */
EAPI Eina_List *evas_object_grid_children_get(const Evas_Grid *obj) EINA_WARN_UNUSED_RESULT;

/**
 *
 * @brief Get an accessor to get random access to the list of children for the grid.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @ingroup Evas_Object_Grid
 *
 * @remark Do not remove or delete objects while walking the list.
 * @since 1.1
 * 
 *
 * @param[in] obj The grid object
 */
EAPI Eina_Accessor *evas_object_grid_accessor_new(const Evas_Grid *obj) EINA_WARN_UNUSED_RESULT;

/**
 *
 * @brief Faster way to remove all child objects from a grid object.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @ingroup Evas_Object_Grid
 *
 * @since 1.1
 * 
 *
 * @param[in] obj The grid object
 * @param[in] clear if true, it will delete just removed children.
 */
EAPI void evas_object_grid_clear(Evas_Grid *obj, Eina_Bool clear);

/**
 *
 * @brief Get an iterator to walk the list of children for the grid.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @ingroup Evas_Object_Grid
 *
 * @remark Do not remove or delete objects while walking the list.
 * @since 1.1
 * 
 *
 * @param[in] obj The grid object
 */
EAPI Eina_Iterator *evas_object_grid_iterator_new(const Evas_Grid *obj) EINA_WARN_UNUSED_RESULT;

/**
 *
 * @brief Create a grid that is child of a given element @a parent.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @ingroup Evas_Object_Grid
 *
 * @see evas_object_grid_add()
 * @since 1.1
 * 
 *
 * @param[in] obj The grid object
 */
EAPI Evas_Object *evas_object_grid_add_to(Evas_Grid *obj) EINA_WARN_UNUSED_RESULT;

/**
 *
 * @brief Remove child from grid.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @ingroup Evas_Object_Grid
 *
 * @remark removing a child will immediately call a walk over children in order
 * to recalculate numbers of columns and rows. If you plan to remove
 * all children, use evas_object_grid_clear() instead.
 *
 * @return 1 on success, 0 on failure.
 * @since 1.1
 * 
 *
 * @param[in] obj The grid object
 * @param[in] child No description supplied.
 */
EAPI Eina_Bool evas_object_grid_unpack(Evas_Grid *obj, Evas_Object *child) EINA_ARG_NONNULL(2);

/**
 *
 * @brief Get the pack options for a grid child
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @ingroup Evas_Object_Grid
 *
 * @remark Get the pack x, y, width and height in virtual coordinates set by
 * evas_object_grid_pack()
 * @return 1 on success, 0 on failure.
 * @since 1.1
 * 
 *
 * @param[in] obj The grid object
 * @param[in] child The grid child to query for coordinates
 * @param[out] x The pointer to where the x coordinate will be returned
 * @param[out] y The pointer to where the y coordinate will be returned
 * @param[out] w The pointer to where the width will be returned
 * @param[out] h The pointer to where the height will be returned
 */
EAPI Eina_Bool evas_object_grid_pack_get(const Evas_Grid *obj, Evas_Object *child, int *x, int *y, int *w, int *h);

/**
 *
 * @brief Add a new child to a grid object.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @ingroup Evas_Object_Grid
 *
 * @return 1 on success, 0 on failure.
 * @since 1.1
 * 
 *
 * @param[in] obj The grid object
 * @param[in] child The child object to add.
 * @param[in] x The virtual x coordinate of the child
 * @param[in] y The virtual y coordinate of the child
 * @param[in] w The virtual width of the child
 * @param[in] h The virtual height of the child
 */
EAPI Eina_Bool evas_object_grid_pack(Evas_Grid *obj, Evas_Object *child, int x, int y, int w, int h) EINA_ARG_NONNULL(2);

#endif
