#ifndef __DALI_TOOLKIT_BUTTON_H__
#define __DALI_TOOLKIT_BUTTON_H__

/*
 * Copyright (c) 2015 Samsung Electronics Co., Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

// INTERNAL INCLUDES
#include <dali-toolkit/public-api/controls/control.h>

namespace Dali
{

namespace Toolkit
{

namespace Internal DALI_INTERNAL
{
class Button;
}
/**
 * @addtogroup dali_toolkit_controls_buttons
 * @{
 */

/**
 * @brief Button is a base class for different kind of buttons.
 *
 * This class provides the disabled property and the clicked signal.
 *
 * A ClickedSignal() is emitted when the button is touched and the touch point doesn't leave the boundary of the button.
 *
 * When the \e disabled property is set to \e true, no signal is emitted.
 *
 * Button provides the following properties which modify the signals emitted:
 * <ul>
 *   <li>\e autorepeating
 *       When \e autorepeating is set to \e true, a Button::PressedSignal(), Button::ReleasedSignal() and Button::ClickedSignal() signals are emitted at regular
 *       intervals while the button is touched.
 *       The intervals could be modified with the Button::SetInitialAutoRepeatingDelay and Button::SetNextAutoRepeatingDelay methods.
 *
 *       A \e togglable button can't be \e autorepeating. If the \e autorepeating property is set to \e true, then the \e togglable property is set to
 *       false but no signal is emitted.
 *
 *   <li>\e togglable
 *       When \e togglable is set to \e true, a Button::StateChangedSignal() signal is emitted, with the selected state.
 * </ul>
 *
 * The button's appearance can be modified by setting properties for the various image filenames.
 *
 * The \e background is always shown and doesn't change if the button is pressed or released. The \e button image is shown over the \e background image when the
 * button is not pressed and is replaced by the \e selected image when the button is pressed. The text label is placed always on the top of all images.
 *
 * When the button is disabled, \e background, \e button and \e selected images are replaced by their \e disabled images.
 *
 * Is not mandatory set all images. A button could be defined only by setting its \e background image or by setting its \e background and \e selected images.
 *
 *
 * @since_tizen 2.4
 */
class DALI_IMPORT_API Button : public Control
{
public:

  /**
   * @brief The start and end property ranges for this control
   * @since_tizen 2.4
   */
  enum PropertyRange
  {
    PROPERTY_START_INDEX = Control::CONTROL_PROPERTY_END_INDEX + 1,
    PROPERTY_END_INDEX =   PROPERTY_START_INDEX + 1000              ///< Reserve property indices @since_tizen 2.4
  };

  /**
   * @brief An enumeration of properties belonging to the Button class
   * @since_tizen 2.4
   */
  struct Property
  {
    enum
    {
      DISABLED = PROPERTY_START_INDEX, ///< type bool @since_tizen 2.4        @see SetDisabled()
      AUTO_REPEATING,                  ///< type bool @since_tizen 2.4        @see SetAutoRepeating()
      INITIAL_AUTO_REPEATING_DELAY,    ///< type float @since_tizen 2.4       @see SetInitialAutoRepeatingDelay()
      NEXT_AUTO_REPEATING_DELAY,       ///< type float @since_tizen 2.4       @see SetNextAutoRepeatingDelay()
      TOGGLABLE,                       ///< type bool @since_tizen 2.4        @see SetTogglableButton()
      SELECTED,                        ///< type bool @since_tizen 2.4        @see SetSelected()
      UNSELECTED_STATE_IMAGE,          ///< type std::string @since_tizen 2.4 @see SetUnselectedImage()
      SELECTED_STATE_IMAGE,            ///< type std::string @since_tizen 2.4 @see SetSelectedImage()
      DISABLED_STATE_IMAGE,            ///< type std::string @since_tizen 2.4 @see SetDisabledImage()
      UNSELECTED_COLOR,                ///< type Vector4 @since_tizen 2.4
      SELECTED_COLOR,                  ///< type Vector4 @since_tizen 2.4
      LABEL,                           ///< type Property::Map @since_tizen 2.4

      // Deprecated properties:
      LABEL_TEXT,                      ///< type std::string @since_tizen 2.4 @see SetLabelText()
    };
  };

public:

  /**
   * @brief Create an uninitialized Button.
   *
   * Only derived versions can be instantiated.  Calling member
   * functions with an uninitialized Dali::Object is not allowed.
   * @since_tizen 2.4
   */
  Button();

  /**
   * @brief Copy constructor
   * @since_tizen 2.4
   */
  Button( const Button& button );

  /**
   * @brief Assignment operator
   * @since_tizen 2.4
   */
  Button& operator=( const Button& button );

  /**
   * @brief Downcast a handle to Button handle.
   *
   * If handle points to a Button the downcast produces valid
   * handle. If not the returned handle is left uninitialized.
   *
   * @since_tizen 2.4
   * @param[in] handle Handle to an object
   * @return A handle to a Button or an uninitialized handle
   */
  static Button DownCast( BaseHandle handle );

  /**
   * @brief Destructor
   *
   * This is non-virtual since derived Handle types must not contain data or virtual methods.
   * @since_tizen 2.4
   */
  ~Button();

  /**
   * @brief Sets the button as \e disabled.
   *
   * No signals are emitted when the \e disabled property is set.
   *
   * @since_tizen 2.4
   * @param[in] disabled property
   */
  void SetDisabled( bool disabled );

  /**
   * @brief Returns if the button is disabled.
   * @since_tizen 2.4
   * @return \e True if the button is \e disabled.
   */
  bool IsDisabled() const;

  /**
   * @brief Sets the \e autorepeating property.
   *
   * If the \e autorepeating property is set to \e true, then the \e togglable property is set to false
   * but no signal is emitted.
   *
   * @since_tizen 2.4
   * @param[in] autoRepeating \e autorepeating property.
   */
  void SetAutoRepeating( bool autoRepeating );

  /**
   * @brief Returns if the autorepeating property is set.
   * @since_tizen 2.4
   * @return \e True if the \e autorepeating property is set.
   */
  bool IsAutoRepeating() const;

  /**
   * @brief Sets the initial autorepeating delay.
   *
   * By default this value is set to 0.15 seconds.
   *
   * @since_tizen 2.4
   * @param[in] initialAutoRepeatingDelay in seconds.
   * @pre initialAutoRepeatingDelay must be greater than zero.
   */
  void SetInitialAutoRepeatingDelay( float initialAutoRepeatingDelay );

  /**
   * @brief Gets the initial autorepeating delay in seconds.
   * @since_tizen 2.4
   * @return The initial autorepeating delay in seconds
   */
  float GetInitialAutoRepeatingDelay() const;

  /**
   * @brief Sets the next autorepeating delay.
   *
   * By default this value is set to 0.05 seconds.
   *
   * @since_tizen 2.4
   * @param[in] nextAutoRepeatingDelay in seconds.
   * @pre nextAutoRepeatingDelay must be greater than zero.
   */
  void SetNextAutoRepeatingDelay( float nextAutoRepeatingDelay );

  /**
   * @brief Gets the next autorepeating delay in seconds.
   * @since_tizen 2.4
   * @return The next autorepeating delay in seconds
   */
  float GetNextAutoRepeatingDelay() const;

  /**
   * @brief Sets the \e togglable property.
   *
   * If the \e togglable property is set to \e true, then the \e autorepeating property is set to false.
   *
   * @since_tizen 2.4
   * @param[in] togglable property
   */
  void SetTogglableButton( bool togglable );

  /**
   * @brief Returns if the togglable property is set.
   * @since_tizen 2.4
   * @return \e True if the \e togglable property is set.
   */
  bool IsTogglableButton() const;

  /**
   * @brief Sets the button as selected or unselected.
   *
   * \e togglable property must be set to \e true.
   *
   * Emits a Button::StateChangedSignal() signal.
   *
   * @since_tizen 2.4
   * @param[in] selected property
   */
  void SetSelected( bool selected );

  /**
   * @brief Returns if the selected property is set and the button is togglable.
   * @since_tizen 2.4
   * @return \e True if the button is \e selected.
   */
  bool IsSelected() const;

  /**
   * @brief Sets the animation time.
   *
   * @since_tizen 2.4
   * @param[in] animationTime The animation time in seconds
   */
  void SetAnimationTime( float animationTime );

  /**
   * @brief Retrieves button's animation time.
   *
   * @since_tizen 2.4
   * @return The animation time in seconds
   */
  float GetAnimationTime() const;

  /**
   * @brief Sets the button's label.
   *
   * @since_tizen 2.4
   * @param[in] label The label text
   */
  void SetLabelText( const std::string& label );

  /**
   * @brief Gets the label.
   *
   * @since_tizen 2.4
   * @return The label text
   */
  std::string GetLabelText() const;

  /**
   * @brief Sets the unselected button image.
   *
   * @since_tizen 2.4
   * @param[in] filename The button image
   */
  void SetUnselectedImage( const std::string& filename );

  /**
   * @brief Sets the background image.
   *
   * @since_tizen 2.4
   * @param[in] filename The background image
   */
  void SetBackgroundImage( const std::string& filename );

  /**
   * @brief Sets the selected image.
   *
   * @since_tizen 2.4
   * @param[in] filename The selected image
   */
  void SetSelectedImage( const std::string& filename );

  /**
   * @brief Sets the selected background image.
   *
   * @since_tizen 2.4
   * @param[in] filename The selected background image
   */
  void SetSelectedBackgroundImage( const std::string& filename );

  /**
   * @brief Sets the disabled background image.
   *
   * @since_tizen 2.4
   * @param[in] filename The disabled background image
   */
  void SetDisabledBackgroundImage( const std::string& filename );

  /**
   * @brief Sets the disabled button image.
   *
   * @since_tizen 2.4
   * @param[in] filename The disabled button image
   */
  void SetDisabledImage( const std::string& filename );

  /**
   * @brief Sets the disabled selected button image.
   *
   * @since_tizen 2.4
   * @param[in] filename The disabled selected button image
   */
  void SetDisabledSelectedImage( const std::string& filename );

  /**
   * @brief Sets the label with an actor.
   * @since_tizen 2.4
   * @remarks Avoid using this method as it's a legacy code. Instead, use @ref SetLabelText.
   * @param[in]  label The actor to use as a label
   */
  void SetLabel( Actor label );

  /**
   * @brief Sets the button image.
   * @since_tizen 2.4
   * @remarks Avoid using this method as it's a legacy code. Instead, use @ref SetUnselectedImage.
   * @param[in]  image The button image
   */
  void SetButtonImage( Image image );

  /**
   * @brief Sets the selected image.
   * @since_tizen 2.4
   * @remarks Avoid using this method as it's a legacy code. Instead, use SetSelectedImage( const std::string& filename ).
   * @param[in]  image The selected image
   */
  void SetSelectedImage( Image image );

  /**
   * @brief Gets the button image.
   * @since_tizen 2.4
   * @remarks Avoid using this method as it's a legacy code.
   * @return     An actor with the button image
   */
  Actor GetButtonImage() const;

  /**
   * @brief Gets the selected image.
   * @since_tizen 2.4
   * @remarks Avoid using this method as it's a legacy code.
   * @return     An actor with the selected image
   */
  Actor GetSelectedImage() const;

public: //Signals

  /**
   * @brief Button signal type
   * @since_tizen 2.4
   */
  typedef Signal< bool ( Button ) > ButtonSignalType;

  /**
   * @brief This signal is emitted when the button is touched.
   *
   * A callback of the following type may be connected:
   * @code
   *   bool YourCallbackName( Button button );
   * @endcode
   * @since_tizen 2.4
   * @return The signal to connect to
   */
  ButtonSignalType& PressedSignal();

  /**
   * @brief This signal is emitted when the button is touched and the touch point leaves the boundary of the button.
   *
   * A callback of the following type may be connected:
   * @code
   *   bool YourCallbackName( Button button );
   * @endcode
   * @since_tizen 2.4
   * @return The signal to connect to
   */
  ButtonSignalType& ReleasedSignal();

  /**
   * @brief This signal is emitted when the button is touched and the touch point doesn't leave the boundary of the button.
   *
   * A callback of the following type may be connected:
   * @code
   *   bool YourCallbackName( Button button );
   * @endcode
   * @since_tizen 2.4
   * @return The signal to connect to
   */
  ButtonSignalType& ClickedSignal();

  /**
   * @brief This signal is emitted when the button's state is changed.
   *
   * The application can get the state by calling IsSelected().
   *
   * A callback of the following type may be connected:
   * @code
   *   bool YourCallbackName( Button button );
   * @endcode
   * @since_tizen 2.4
   * @return The signal to connect to
   */
  ButtonSignalType& StateChangedSignal();

public: // Not intended for application developers

  /**
   * @brief Creates a handle using the Toolkit::Internal implementation.
   *
   * @since_tizen 2.4
   * @param[in]  implementation  The Control implementation
   */
  DALI_INTERNAL Button( Internal::Button& implementation );

  /**
   * @brief Allows the creation of this Control from an Internal::CustomActor pointer.
   *
   * @since_tizen 2.4
   * @param[in]  internal  A pointer to the internal CustomActor
   */
  DALI_INTERNAL Button( Dali::Internal::CustomActor* internal );
};

/**
 * @}
 */
} // namespace Toolkit

} // namespace Dali

#endif // __DALI_TOOLKIT_BUTTON_H__
