#ifndef _ELM_SLIDER_EO_H_
#define _ELM_SLIDER_EO_H_

#ifndef _ELM_SLIDER_EO_CLASS_TYPE
#define _ELM_SLIDER_EO_CLASS_TYPE

typedef Eo Elm_Slider;

#endif

#ifndef _ELM_SLIDER_EO_TYPES
#define _ELM_SLIDER_EO_TYPES


#endif
#define ELM_SLIDER_CLASS elm_slider_class_get()

const Eo_Class *elm_slider_class_get(void) EINA_CONST;

/**
 *
 * @brief Set the orientation of a given slider widget.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @remark Use this function to change how your slider is to be
 * disposed: vertically or horizontally.
 *
 * @remark By default it's displayed horizontally.
 *
 * @see elm_slider_horizontal_get()
 *
 * @ingroup Slider
 *
 * @param[in] horizontal Use @c EINA_TRUE to make @p obj to be
@b horizontal, @c EINA_FALSE to make it @b vertical.
 *
 */
EOAPI void  elm_obj_slider_horizontal_set(Eina_Bool horizontal);

/**
 *
 * @brief Get the orientation of a given slider widget
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @return @c EINA_TRUE, if @p obj is set to be @b horizontal,
 * @c EINA_FALSE if it's @b vertical (and on errors).
 *
 * @see elm_slider_horizontal_set() for more details.
 *
 * @ingroup Slider
 *
 *
 */
EOAPI Eina_Bool  elm_obj_slider_horizontal_get(void);

/**
 *
 * @brief Set the value the slider displays.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @remark Value will be presented on the unit label following format specified with
 * elm_slider_unit_format_set() and on indicator with
 * elm_slider_indicator_format_set().
 *
 * @warning The value must to be between min and max values. This values
 * are set by elm_slider_min_max_set().
 *
 * @see elm_slider_value_get()
 * @see elm_slider_unit_format_set()
 * @see elm_slider_indicator_format_set()
 * @see elm_slider_min_max_set()
 *
 * @ingroup Slider
 *
 * @param[in] val The value to be displayed.
 *
 */
EOAPI void  elm_obj_slider_value_set(double val);

/**
 *
 * @brief Get the value displayed by the slider.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @return The value displayed.
 *
 * @see elm_slider_value_set() for details.
 *
 * @ingroup Slider
 *
 *
 */
EOAPI double  elm_obj_slider_value_get(void);

/**
 *
 * @brief Set the format string for the indicator label.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @remark The slider may display its value somewhere else then unit label,
 * for example, above the slider knob that is dragged around. This function
 * sets the format string used for this.
 *
 * @remark If @c NULL, indicator label won't be visible. If not it sets the format
 * string for the label text. To the label text is provided a floating point
 * value, so the label text can display up to 1 floating point value.
 * Note that this is optional.
 *
 * @remark Use a format string such as "%1.2f meters" for example, and it will
 * display values like: "3.14 meters" for a value equal to 3.14159.
 *
 * @remark Default is indicator label disabled.
 *
 * @see elm_slider_indicator_format_get()
 *
 * @ingroup Slider
 *
 * @param[in] indicator The format string for the indicator display.
 *
 */
EOAPI void  elm_obj_slider_indicator_format_set(const char *indicator);

/**
 *
 * @brief Get the indicator label format of the slider.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @return The indicator label format string in UTF-8.
 *
 * @remark The slider may display its value somewhere else then unit label,
 * for example, above the slider knob that is dragged around. This function
 * gets the format string used for this.
 *
 * @see elm_slider_indicator_format_set() for more
 * information on how this works.
 *
 * @ingroup Slider
 *
 *
 */
EOAPI const char * elm_obj_slider_indicator_format_get(void);

/**
 *
 * @brief Invert a given slider widget's displaying values order
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @remark A slider may be @b inverted, in which state it gets its
 * values inverted, with high vales being on the left or top and
 * low values on the right or bottom, as opposed to normally have
 * the low values on the former and high values on the latter,
 * respectively, for horizontal and vertical modes.
 *
 * @see elm_slider_inverted_get()
 *
 * @ingroup Slider
 *
 * @param[in] inverted Use @c EINA_TRUE to make @p obj inverted,
@c EINA_FALSE to bring it back to default, non-inverted values.
 *
 */
EOAPI void  elm_obj_slider_inverted_set(Eina_Bool inverted);

/**
 *
 * @brief Get whether a given slider widget's displaying values are
 * inverted or not.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @return @c EINA_TRUE, if @p obj has inverted values,
 * @c EINA_FALSE otherwise (and on errors).
 *
 * @see elm_slider_inverted_set() for more details.
 *
 * @ingroup Slider
 *
 *
 */
EOAPI Eina_Bool  elm_obj_slider_inverted_get(void);

/**
 *
 * @brief Set whether to enlarge slider indicator (augmented knob) or not.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @remark By default, indicator will be bigger while dragged by the user.
 *
 * @warning It won't display values set with
 * elm_slider_indicator_format_set() if you disable indicator.
 *
 * @ingroup Slider
 *
 * @param[in] show @c EINA_TRUE will make it enlarge, @c EINA_FALSE will
let the knob always at default size.
 *
 */
EOAPI void  elm_obj_slider_indicator_show_set(Eina_Bool show);

/**
 *
 * @brief Get whether a given slider widget's enlarging indicator or not.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @return @c EINA_TRUE, if @p obj is enlarging indicator, or
 * @c EINA_FALSE otherwise (and on errors).
 *
 * @see elm_slider_indicator_show_set() for details.
 *
 * @ingroup Slider
 *
 *
 */
EOAPI Eina_Bool  elm_obj_slider_indicator_show_get(void);

/**
 *
 * @brief Set the minimum and maximum values for the slider.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @remark Define the allowed range of values to be selected by the user.
 *
 * @remark If actual value is less than @p min, it will be updated to @p min. If it
 * is bigger then @p max, will be updated to @p max. Actual value can be
 * get with elm_slider_value_get().
 *
 * @remark By default, min is equal to 0.0, and max is equal to 1.0.
 *
 * @warning Maximum must be greater than minimum, otherwise behavior
 * is undefined.
 *
 * @see elm_slider_min_max_get()
 *
 * @ingroup Slider
 *
 * @param[in] min The minimum value.
 * @param[in] max The maximum value.
 *
 */
EOAPI void  elm_obj_slider_min_max_set(double min, double max);

/**
 *
 * @brief Get the minimum and maximum values of the slider.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @remark If only one value is needed, the other pointer can be passed
 * as @c NULL.
 *
 * @see elm_slider_min_max_set() for details.
 *
 * @ingroup Slider
 *
 * @param[out] min The minimum value.
 * @param[out] max The maximum value.
 *
 */
EOAPI void  elm_obj_slider_min_max_get(double *min, double *max);

/**
 *
 * @brief Set the format string for the unit label.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @remark Unit label is displayed all the time, if set, after slider's bar.
 * In horizontal mode, at right and in vertical mode, at bottom.
 *
 * @remark If @c NULL, unit label won't be visible. If not it sets the format
 * string for the label text. To the label text is provided a floating point
 * value, so the label text can display up to 1 floating point value.
 * Note that this is optional.
 *
 * @remark Use a format string such as "%1.2f meters" for example, and it will
 * display values like: "3.14 meters" for a value equal to 3.14159.
 *
 * @remark Default is unit label disabled.
 *
 * @see elm_slider_indicator_format_get()
 *
 * @ingroup Slider
 *
 * @param[in] units The format string for the unit display.
 *
 */
EOAPI void  elm_obj_slider_unit_format_set(const char *units);

/**
 *
 * @brief Get the unit label format of the slider.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @return The unit label format string in UTF-8.
 *
 * @remark Unit label is displayed all the time, if set, after slider's bar.
 * In horizontal mode, at right and in vertical mode, at bottom.
 *
 * @see elm_slider_unit_format_set() for more
 * information on how this works.
 *
 * @ingroup Slider
 *
 *
 */
EOAPI const char * elm_obj_slider_unit_format_get(void);

/**
 *
 * @internal
 *
 * Show the indicator of slider on focus
 *
 * @ingroup Slider
 *
 * @param[in] flag @ EINA_TRUE sets the indicator visible
 *
 */
EOAPI void  elm_obj_slider_indicator_show_on_focus_set(Eina_Bool flag);

/**
 *
 * @internal
 *
 * Get whether the indicator of the slider is set or not
 *
 * @return @c EINA_TRUE, if indicator is set
 * @c EINA_FALSE otherwise.
 *
 * @ingroup Slider
 *
 *
 */
EOAPI Eina_Bool  elm_obj_slider_indicator_show_on_focus_get(void);

/**
 *
 * @brief Set the (exact) length of the bar region of a given slider widget.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @remark This sets the minimum width (when in horizontal mode) or height
 * (when in vertical mode) of the actual bar area of the slider
 * @p obj. This in turn affects the object's minimum size. Use
 * this when you're not setting other size hints expanding on the
 * given direction (like weight and alignment hints) and you would
 * like it to have a specific size.
 *
 * @remark Icon, end, label, indicator and unit text around @p obj
 * will require their
 * own space, which will make @p obj to require more the @p size,
 * actually.
 *
 * @see elm_slider_span_size_get()
 *
 * @ingroup Slider
 *
 * @param[in] size The length of the slider's bar region.
 *
 */
EOAPI void  elm_obj_slider_span_size_set(Evas_Coord size);

/**
 *
 * @brief Get the length set for the bar region of a given slider widget
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @return The length of the slider's bar region.
 *
 * @remark If that size was not set previously, with
 * elm_slider_span_size_set(), this call will return @c 0.
 *
 * @ingroup Slider
 *
 *
 */
EOAPI Evas_Coord  elm_obj_slider_span_size_get(void);

/**
 *
 * @brief Set the step by which slider indicator will move.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @remark This value is used when draggable object is moved automatically i.e., in case
 * of key event when up/down/left/right key is pressed or in case when
 * accessibility is set and flick event is used to inc/dec slider values.
 * By default step value is equal to 0.05.
 *
 * @see elm_slider_step_get() for more details.
 *
 * @ingroup Slider
 * @since 1.8
 *
 * @param[in] step The step value.
 *
 */
EOAPI void  elm_obj_slider_step_set(double step);

/**
 *
 * @brief Get the step by which slider indicator moves.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @return The step value.
 *
 * @see elm_slider_step_set() for more details.
 *
 * @ingroup Slider
 * @since 1.8
 *
 *
 */
EOAPI double  elm_obj_slider_step_get(void);

/**
 *
 * @brief Set the visible mode of indicator.
 *
 * @if MOBILE @since_tizen 2.4
 * @elseif WEARABLE @since_tizen 2.4
 * @endif
 *
 * @see elm_slider_indicator_visible_mode_get() for more details.
 *
 * @ingroup Slider
 *
 * @param[in] indicator_visible_mode The indicator visible mode.
 *
 */
EOAPI void  elm_obj_slider_indicator_visible_mode_set(Elm_Slider_Indicator_Visible_Mode indicator_visible_mode);

/**
 *
 * @brief Get the visible mode of indicator.
 *
 * @if MOBILE @since_tizen 2.4
 * @elseif WEARABLE @since_tizen 2.4
 * @endif
 *
 * @return The indicator visible mode.
 *
 * @see elm_slider_indicator_visible_mode_set() for more details.
 *
 * @ingroup Slider
 *
 *
 */
EOAPI Elm_Slider_Indicator_Visible_Mode  elm_obj_slider_indicator_visible_mode_get(void);

/**
 *
 * @brief Set the format function pointer for the indicator label
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @remark Set the callback function to format the indicator string.
 *
 * @see elm_slider_indicator_format_set() for more info on how this works.
 *
 * @ingroup Slider
 *
 * @param[in] func The indicator format function.
 * @param[in] free_func The freeing function for the format string.
 *
 */
EOAPI void  elm_obj_slider_indicator_format_function_set(slider_func_type func, slider_freefunc_type free_func);

/**
 *
 * @brief Set the format function pointer for the units label
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @remark Set the callback function to format the units string.
 *
 * @see elm_slider_unit_format_set() for more info on how this works.
 *
 * @ingroup Slider
 *
 * @param[in] func The units format function.
 * @param[in] free_func The freeing function for the format string.
 *
 */
EOAPI void  elm_obj_slider_units_format_function_set(slider_func_type func, slider_freefunc_type free_func);

EOAPI extern const Eo_Event_Description _ELM_SLIDER_EVENT_CHANGED;
EOAPI extern const Eo_Event_Description _ELM_SLIDER_EVENT_DELAY_CHANGED;
EOAPI extern const Eo_Event_Description _ELM_SLIDER_EVENT_SLIDER_DRAG_START;
EOAPI extern const Eo_Event_Description _ELM_SLIDER_EVENT_SLIDER_DRAG_STOP;
EOAPI extern const Eo_Event_Description _ELM_SLIDER_EVENT_LANGUAGE_CHANGED;
EOAPI extern const Eo_Event_Description _ELM_SLIDER_EVENT_ACCESS_CHANGED;
EOAPI extern const Eo_Event_Description _ELM_SLIDER_EVENT_FOCUSED;
EOAPI extern const Eo_Event_Description _ELM_SLIDER_EVENT_UNFOCUSED;

/**
 * No description
 */
#define ELM_SLIDER_EVENT_CHANGED (&(_ELM_SLIDER_EVENT_CHANGED))

/**
 * No description
 */
#define ELM_SLIDER_EVENT_DELAY_CHANGED (&(_ELM_SLIDER_EVENT_DELAY_CHANGED))

/**
 * No description
 */
#define ELM_SLIDER_EVENT_SLIDER_DRAG_START (&(_ELM_SLIDER_EVENT_SLIDER_DRAG_START))

/**
 * No description
 */
#define ELM_SLIDER_EVENT_SLIDER_DRAG_STOP (&(_ELM_SLIDER_EVENT_SLIDER_DRAG_STOP))

/**
 * No description
 */
#define ELM_SLIDER_EVENT_LANGUAGE_CHANGED (&(_ELM_SLIDER_EVENT_LANGUAGE_CHANGED))

/**
 * No description
 */
#define ELM_SLIDER_EVENT_ACCESS_CHANGED (&(_ELM_SLIDER_EVENT_ACCESS_CHANGED))

/**
 * No description
 */
#define ELM_SLIDER_EVENT_FOCUSED (&(_ELM_SLIDER_EVENT_FOCUSED))

/**
 * No description
 */
#define ELM_SLIDER_EVENT_UNFOCUSED (&(_ELM_SLIDER_EVENT_UNFOCUSED))

#endif
