/**
 * @brief Adds a new genlist widget to the given parent Elementary
 *        (container) object.
 *
 * @details This function inserts a new genlist widget on the canvas.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @param[in] parent The parent object
 * @return A new genlist widget handle, otherwise @c NULL in case of an error
 *
 * @see elm_genlist_item_append()
 * @see elm_object_item_del()
 * @see elm_genlist_clear()
 *
 * @ingroup Elm_Genlist
 */
EAPI Evas_Object                  *elm_genlist_add(Evas_Object *parent);

/**
 * @brief Removes all items from a given genlist widget.
 *
 * @details This removes (and deletes) all items in @a obj, making it empty.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @param[in] obj The genlist object
 *
 * @see elm_object_item_del() to remove just one item.
 *
 * @ingroup Elm_Genlist
 */
EAPI void                          elm_genlist_clear(Evas_Object *obj);

/**
 * @brief Enables or disables multi-selection in the genlist.
 *
 * @details This enables (@c EINA_TRUE) or disables (@c EINA_FALSE) multi-selection in
 *          the list. This allows more than @c 1 item to be selected. To retrieve the list
 *          of selected items, use elm_genlist_selected_items_get().
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @param[in] obj The genlist object
 * @param[in] multi The boolean value that enables or disables multi-selection \n
 *              Default is disabled.
 *
 * @see elm_genlist_selected_items_get()
 * @see elm_genlist_multi_select_get()
 *
 * @ingroup Elm_Genlist
 */
EAPI void                          elm_genlist_multi_select_set(Evas_Object *obj, Eina_Bool multi);

/**
 * @brief Gets whether multi-selection in genlist is enabled or disabled.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @param[in] obj The genlist object
 * @return The boolean value that indicates whether multi-selection is enabled or disabled
 *         (@c EINA_TRUE = enabled/@c EINA_FALSE = disabled). Default is @c EINA_FALSE.
 *
 * @see elm_genlist_multi_select_set()
 *
 * @ingroup Elm_Genlist
 */
EAPI Eina_Bool                     elm_genlist_multi_select_get(const Evas_Object *obj);

/**
 * @brief Sets the horizontal stretching mode.
 *
 * @details This sets the mode used for sizing items horizontally. Valid modes
 *          are #ELM_LIST_LIMIT, #ELM_LIST_SCROLL, and #ELM_LIST_COMPRESS. The default is
 *          ELM_LIST_SCROLL. This mode means that if items are too wide to fit,
 *          the scroller scrolls horizontally. Otherwise items are expanded
 *          to fill the width of the viewport of the scroller. If it is
 *          ELM_LIST_LIMIT, items are expanded to the viewport width and
 *          limited to that size. If it is ELM_LIST_COMPRESS, the item width is
 *          fixed (restricted to a minimum of) to the list width when calculating its
 *          size in order to allow the height to be calculated based on it. This allows,
 *          for instance, a text block to wrap lines if the Edje part is configured with
 *          "text.min: 0 1".
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @remarks ELM_LIST_COMPRESS makes list resize slower as it
 *          recalculates every item height again whenever the list width
 *          changes
 * @remarks The homogeneous mode is so that all items in the genlist are of the same
 *          width/height. With ELM_LIST_COMPRESS, genlist items are initialized fast.
 *          However, there are no sub-objects in the genlist which can be
 *          on the flying resizable (such as TEXTBLOCK). If so, then some dynamic
 *          resizable objects in the genlist would not be diplayed properly.
 *
 * @param[in] obj The genlist object
 * @param[in] mode The mode to use (either #ELM_LIST_SCROLL or #ELM_LIST_LIMIT)
 *
 * @see elm_genlist_mode_get()
 *
 * @ingroup Elm_Genlist
 */
EAPI void                          elm_genlist_mode_set(Evas_Object *obj, Elm_List_Mode mode);

/**
 * @brief Gets the horizontal stretching mode.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @param[in] obj The genlist object
 * @return The mode to use
 * (#ELM_LIST_LIMIT, #ELM_LIST_SCROLL)
 *
 * @see elm_genlist_mode_set()
 *
 * @ingroup Elm_Genlist
 */
EAPI Elm_List_Mode                 elm_genlist_mode_get(const Evas_Object *obj);

/**
 * @internal
 *
 * @brief Enables or disables horizontal and vertical bouncing effect.
 *
 * @details This enables or disables the scroller bouncing effect for the
 *          genlist. See elm_scroller_bounce_set() for details.
 *
 * @param obj The genlist object
 * @param h_bounce The boolean value that allows horizontal bouncing (@c EINA_TRUE = on, @c
 *                 EINA_FALSE = off) \n
 *                 Default is @c EINA_FALSE.
 * @param v_bounce The boolean value that allows vertical bouncing (@c EINA_TRUE = on, @c
 *                 EINA_FALSE = off) \n
 *                 Default is @c EINA_TRUE.
 *
 * @deprecated Use elm_scroller_bounce_set() instead.
 *
 * @see elm_scroller_bounce_set()
 * @see elm_genlist_bounce_get()
 *
 * @ingroup Elm_Genlist
 */
EINA_DEPRECATED EAPI void          elm_genlist_bounce_set(Evas_Object *obj, Eina_Bool h_bounce, Eina_Bool v_bounce);

/**
 * @internal
 *
 * @brief Gets whether the horizontal and vertical bouncing effect is enabled.
 *
 * @param obj The genlist object
 * @param h_bounce The pointer to a bool that indicates if horizontal bouncing is set
 * @param v_bounce The pointer to a bool that indicates if vertical bouncing is set
 *
 * @deprecated Use elm_scroller_bounce_get() instead.
 *
 * @see elm_scroller_bounce_get()
 * @see elm_genlist_bounce_set()
 *
 * @ingroup Elm_Genlist
 */
EINA_DEPRECATED EAPI void          elm_genlist_bounce_get(const Evas_Object *obj, Eina_Bool *h_bounce, Eina_Bool *v_bounce);

/**
 * @brief Appends a new item to a given genlist widget.
 *
 * @details This adds the given item to the end of the list or the end of
 *          the children list if the @a parent is given.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @param[in] obj The genlist object
 * @param[in] itc The item class for the item
 * @param[in] data The item data
 * @param[in] parent The parent item, otherwise @c NULL if there is no parent item
 * @param[in] type The item type
 * @param[in] func The convenience function that is called when the item is selected
 * @param[in] func_data The data passed to @a func mentioned above
 * @return A handle to the added item, otherwise @c NULL if it is not possible
 *
 * @see elm_genlist_item_prepend()
 * @see elm_genlist_item_insert_before()
 * @see elm_genlist_item_insert_after()
 * @see elm_object_item_del()
 *
 * @ingroup Elm_Genlist
 */
EAPI Elm_Object_Item             *elm_genlist_item_append(Evas_Object *obj, const Elm_Genlist_Item_Class *itc, const void *data, Elm_Object_Item *parent, Elm_Genlist_Item_Type type, Evas_Smart_Cb func, const void *func_data);

/**
 * @brief Prepends a new item to a given genlist widget.
 *
 * @details This adds an item to the beginning of the list or beginning of the
 *          children of the parent if given.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @param[in] obj The genlist object
 * @param[in] itc The item class for the item
 * @param[in] data The item data
 * @param[in] parent The parent item, otherwise @c NULL if there is no parent item
 * @param[in] type The item type
 * @param[in] func The convenience function that is called when the item is selected
 * @param[in] func_data The data passed to @a func mentioned above
 * @return A handle to the added item, otherwise @c NULL if it is not possible
 *
 * @see elm_genlist_item_append()
 * @see elm_genlist_item_insert_before()
 * @see elm_genlist_item_insert_after()
 * @see elm_object_item_del()
 *
 * @ingroup Elm_Genlist
 */
EAPI Elm_Object_Item             *elm_genlist_item_prepend(Evas_Object *obj, const Elm_Genlist_Item_Class *itc, const void *data, Elm_Object_Item *parent, Elm_Genlist_Item_Type type, Evas_Smart_Cb func, const void *func_data);

/**
 * @brief Inserts an item before another in a genlist widget.
 *
 * @details This inserts an item before another in the list. It is the
 *          same tree level or group as the item before which it is inserted.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @param[in] obj The genlist object
 * @param[in] itc The item class for the item
 * @param[in] data The item data
 * @param[in] parent The parent item, otherwise @c NULL if there is no parent item
 * @param[in] before The item before which to place this new one
 * @param[in] type The item type
 * @param[in] func The convenience function that is called when the item is selected
 * @param[in] func_data The data passed to @a func mentioned above
 * @return A handle to the item added, otherwise @c NULL if it is not possible
 *
 * @see elm_genlist_item_append()
 * @see elm_genlist_item_prepend()
 * @see elm_genlist_item_insert_after()
 * @see elm_object_item_del()
 *
 * @ingroup Elm_Genlist
 */
EAPI Elm_Object_Item             *elm_genlist_item_insert_before(Evas_Object *obj, const Elm_Genlist_Item_Class *itc, const void *data, Elm_Object_Item *parent, Elm_Object_Item *before, Elm_Genlist_Item_Type type, Evas_Smart_Cb func, const void *func_data);

/**
 * @brief Inserts an item after another in a genlist widget.
 *
 * @details This inserts an item after another in the list. It is in the
 *          same tree level or group as the item after which it is inserted.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @param[in] obj The genlist object
 * @param[in] itc The item class for the item
 * @param[in] data The item data
 * @param[in] parent The parent item, otherwise @c NULL if there is no parent item
 * @param[in] after The item after which to place this new one
 * @param[in] type The item type
 * @param[in] func The convenience function that is called when the item is selected
 * @param[in] func_data The data passed to @a func mentioned above
 * @return A handle to the item added, otherwise @c NULL if it is not possible
 *
 * @see elm_genlist_item_append()
 * @see elm_genlist_item_prepend()
 * @see elm_genlist_item_insert_before()
 * @see elm_object_item_del()
 *
 * @ingroup Elm_Genlist
 */
EAPI Elm_Object_Item             *elm_genlist_item_insert_after(Evas_Object *obj, const Elm_Genlist_Item_Class *itc, const void *data, Elm_Object_Item *parent, Elm_Object_Item *after, Elm_Genlist_Item_Type type, Evas_Smart_Cb func, const void *func_data);

/**
 * @brief Inserts a new item into the sorted genlist object.
 *
 * @details This inserts an item in the genlist based on a user defined comparison
 *          function. The two arguments passed to the function @a func are genlist item
 *          handles to compare.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @param[in] obj The genlist object
 * @param[in] itc The item class for the item
 * @param[in] data The item data
 * @param[in] parent The parent item, otherwise @c NULL if there is no parent item
 * @param[in] type The item type
 * @param[in] comp The function called for sorting
 * @param[in] func The convenience function that is called when the item is selected
 * @param[in] func_data The data passed to @a func mentioned above
 * @return A handle to the item added, otherwise @c NULL if it is not possible
 *
 * @see elm_genlist_item_append()
 * @see elm_genlist_item_prepend()
 * @see elm_genlist_item_insert_after()
 * @see elm_object_item_del()
 *
 * @ingroup Elm_Genlist
 */
EAPI Elm_Object_Item             *elm_genlist_item_sorted_insert(Evas_Object *obj, const Elm_Genlist_Item_Class *itc, const void *data, Elm_Object_Item *parent, Elm_Genlist_Item_Type type, Eina_Compare_Cb comp, Evas_Smart_Cb func, const void *func_data);

/* Operations to retrieve existing items */
/**
 * @brief Gets the selected item in the genlist.
 *
 * @details This gets the selected item in the list (if multi-selection is enabled, only
 *          the item that is first selected in the list is returned, which is not very
 *          useful, so see elm_genlist_selected_items_get() to know when multi-selection is
 *          used).
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @remarks If no item is selected, @c NULL is returned.
 *
 * @param[in] obj The genlist object
 * @return The selected item, otherwise @c NULL if none are selected
 *
 * @see elm_genlist_selected_items_get()
 *
 * @ingroup Elm_Genlist
 */
EAPI Elm_Object_Item             *elm_genlist_selected_item_get(const Evas_Object *obj);

/**
 * @brief Gets a list of selected items in the genlist.
 *
 * @details It returns a list of selected items. This list pointer is only valid as
 *          long as the selection doesn't change (no items are selected or unselected, or
 *          unselected implicitly by deletion). The list contains genlist item
 *          pointers. The order of the items in this list is the order in which they were
 *          selected, i.e. the first item in this list is the first item that is
 *          selected, and so on.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @remarks If not in the multi-select mode, use
 *          elm_genlist_selected_item_get() instead.
 *
 * @param[in] obj The genlist object
 * @return The list of selected items, otherwise @c NULL if none are selected
 *
 * @see elm_genlist_multi_select_set()
 * @see elm_genlist_selected_item_get()
 *
 * @ingroup Elm_Genlist
 */
EAPI Eina_List              *elm_genlist_selected_items_get(const Evas_Object *obj);

/**
 * @brief Gets a list of realized items in the genlist.
 *
 * @details This returns a list of realized items in the genlist. The list
 *          contains genlist item pointers. The list must be freed by the
 *          caller when done with eina_list_free(). The item pointers in the
 *          list are only valid as long as those items are not deleted or the
 *          genlist is not deleted.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @param[in] obj The genlist object
 * @return The list of realized items, otherwise @c NULL if none are realized
 *
 * @see elm_genlist_realized_items_update()
 *
 * @ingroup Elm_Genlist
 */
EAPI Eina_List                    *elm_genlist_realized_items_get(const Evas_Object *obj);

/**
 * @brief Gets the first item in the genlist.
 *
 * @details This returns the first item in the list.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @param[in] obj The genlist object
 * @return The first item, otherwise @c NULL if there are no items
 *
 * @ingroup Elm_Genlist
 */
EAPI Elm_Object_Item             *elm_genlist_first_item_get(const Evas_Object *obj);

/**
 * @brief Gets the last item in the genlist.
 *
 * @details This returns the last item in the list.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @param[in] obj The genlist object
 * @return The last item, otherwise @c NULL if there are no items
 *
 * @ingroup Elm_Genlist
 */
EAPI Elm_Object_Item             *elm_genlist_last_item_get(const Evas_Object *obj);

/**
 * @internal
 *
 * @brief Sets the scrollbar policy.
 *
 * @details This sets the scrollbar visibility policy for the given genlist
 *          scroller. #ELM_SCROLLER_POLICY_AUTO means the scrollbar is
 *          made visible if it is needed, and otherwise kept hidden. #ELM_SCROLLER_POLICY_ON
 *          turns it on at all times, and #ELM_SCROLLER_POLICY_OFF always keeps it off.
 *          This applies for the horizontal and vertical scrollbars respectively.
 *          The default is #ELM_SCROLLER_POLICY_AUTO.
 *
 * @param obj The genlist object
 * @param policy_h The horizontal scrollbar policy
 * @param policy_v The vertical scrollbar policy
 *
 * @deprecated Use elm_scroller_policy_set() instead.
 *
 * @see elm_scroller_policy_set()
 *
 * @ingroup Elm_Genlist
 */
EINA_DEPRECATED EAPI void          elm_genlist_scroller_policy_set(Evas_Object *obj, Elm_Scroller_Policy policy_h, Elm_Scroller_Policy policy_v);

/**
 * @internal
 *
 * @brief Gets the scrollbar policy.
 *
 * @param obj The genlist object
 * @param policy_h The pointer to store the horizontal scrollbar policy
 * @param policy_v The pointer to store the vertical scrollbar policy
 *
 * @deprecated Use elm_scroller_policy_get() instead.
 *
 * @see elm_scroller_policy_get()
 *
 * @ingroup Elm_Genlist
 */
EINA_DEPRECATED EAPI void          elm_genlist_scroller_policy_get(const Evas_Object *obj, Elm_Scroller_Policy *policy_h, Elm_Scroller_Policy *policy_v);

/**
 * @brief Updates the content of all the realized items.
 *
 * @details This updates all the realized items by calling all the item class functions again
 *          to get the content, text and states. Use this when the original
 *          item data has changed and the changes are desired to reflect.
 *
 *          To update just one item, use elm_genlist_item_update().
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @param[in] obj The genlist object
 *
 * @see elm_genlist_realized_items_get()
 * @see elm_genlist_item_update()
 *
 * @ingroup Elm_Genlist
 */
EAPI void                          elm_genlist_realized_items_update(Evas_Object *obj);

/**
 * @brief Returns the number of items that are currently in a list.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @remarks This behavior is O(1) and includes items which may or may not be realized.
 *
 * @param[in] obj The list
 * @return The total number of items in the list
 *
 * @ingroup Elm_Genlist
 */
EAPI unsigned int elm_genlist_items_count(const Evas_Object *obj);

/**
 * @brief Enables or disables the homogeneous mode.
 *
 * @details This enables the homogeneous mode where items are of the same
 *          height and width so that genlist may perform lazy-loading at its
 *          maximum (which increases the performance for scrolling the list).
 *          In the normal mode, genlist pre-calculates all the items' sizes even
 *          though they are not in use. So items' callbacks are called for more times than
 *          expected. But the homogeneous mode skips the item size pre-calculation
 *          process so items' callbacks are called only when the item is needed.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @remarks This also works well with group index.
 *
 * @param[in] obj The genlist object
 * @param[in] homogeneous The boolean value assuming that the items within the genlist are of the
 *                    same height and width (@c EINA_TRUE = on, @c EINA_FALSE = off) \n
 *                    Default is @c EINA_FALSE.
 *
 * @see elm_genlist_mode_set()
 * @see elm_genlist_homogeneous_get()
 *
 * @ingroup Elm_Genlist
 */
EAPI void                          elm_genlist_homogeneous_set(Evas_Object *obj, Eina_Bool homogeneous);

/**
 * @brief Gets whether the homogeneous mode is enabled.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @param[in] obj The genlist object
 * @return The boolean value assuming that the items within the genlist are of the same height
 *         and width (@c EINA_TRUE = on, @c EINA_FALSE = off)
 *
 * @see elm_genlist_homogeneous_set()
 *
 * @ingroup Elm_Genlist
 */
EAPI Eina_Bool                     elm_genlist_homogeneous_get(const Evas_Object *obj);

/**
 * @brief Sets the maximum number of items within an item block.
 *
 * @details This configures the block count to tune the target with, for a particular
 *          performance matrix.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @remarks A block of objects are used to reduce the number of operations occurring due to
 *          large number of objects on the screen. It can determine the visibility, or if the
 *          object has changed, its theme needs to be updated by doing this kind of
 *          calculation to the entire block, instead of every object.
 *
 * @remarks The default value for the block count is enough for most lists, so unless
 *          your sure that you have a lot of objects visible on the screen at the same
 *          time, don't try to change this.
 *
 * @param[in] obj The genlist object
 * @param[in] count The maximum number of items within an item block \n
 *              Default is @c 32.
 *
 * @see elm_genlist_block_count_get()
 * @see @ref Elm_Genlist_Implementation
 *
 * @ingroup Elm_Genlist
 */
EAPI void                          elm_genlist_block_count_set(Evas_Object *obj, int count);

/**
 * @brief Gets the maximum number of items within an item block.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @param[in] obj The genlist object
 * @return The maximum number of items within an item block
 *
 * @see elm_genlist_block_count_set()
 *
 * @ingroup Elm_Genlist
 */
EAPI int                           elm_genlist_block_count_get(const Evas_Object *obj);

/**
 * @brief Sets the timeout in seconds for the longpress event.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @remarks This option changes the time it takes to send an event @c "longpressed"
 *          after the mouse down signal is sent to the list. If this event occurs, no
 *          @c "clicked" event is sent.
 *
 * @remarks If you set the longpress timeout value with this API, your genlist
 *          is not affected by the longpress value of the elementary config value
 *          later.
 *
 * @param[in] obj The genlist object
 * @param[in] timeout The timeout in seconds \n
 *                The default value is elm config value(1.0).
 *
 * @see elm_genlist_longpress_timeout_set()
 *
 * @ingroup Elm_Genlist
 */
EAPI void                          elm_genlist_longpress_timeout_set(Evas_Object *obj, double timeout);

/**
 * @brief Gets the timeout in seconds for the longpress event.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @param[in] obj The genlist object
 * @return The timeout in seconds
 *
 * @see elm_genlist_longpress_timeout_get()
 *
 * @ingroup Elm_Genlist
 */
EAPI double                        elm_genlist_longpress_timeout_get(const Evas_Object *obj);

/**
 * @brief Gets the item that is at the x, y canvas coordinates.
 *
 * @details This returns the item at the given coordinates (which are canvas
 *          relative, not object-relative). If an item is at that coordinate,
 *          that item handle is returned, and if @a posret is not @c NULL, the
 *          integer it is pointing to is set to either @c -1, @c 0, or @c 1, depending on whether
 *          the coordinate is on the upper portion of that item (-1), in the
 *          middle section (0), or on the lower part (1). If @c NULL is returned as
 *          an item (no item found there), then posret may indicate @c -1 or @c 1
 *          depending on whether the coordinate is above or below the items in
 *          the genlist respectively.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @param[in] obj The genlist object
 * @param[in] x The input x coordinate
 * @param[in] y The input y coordinate
 * @param[out] posret The position relative to the returned item
 * @return The item at the coordinates, otherwise @c NULL if there are none
 *
 * @ingroup Elm_Genlist
 */
EAPI Elm_Object_Item             *elm_genlist_at_xy_item_get(const Evas_Object *obj, Evas_Coord x, Evas_Coord y, int *posret);

/**
 * @internal
 * @remarks Tizen no feature
 *
 * @brief Gets the active genlist mode item.
 *
 * @details This function returns the item that is activated with a mode, by the
 *          function elm_genlist_item_decorate_mode_set().
 *
 * @param[in] obj The genlist object
 * @return The active item for that current mode, otherwise @c NULL if no item is
 *         activated with a mode
 *
 * @see elm_genlist_item_decorate_mode_set()
 * @see elm_genlist_mode_get()
 *
 * @ingroup Elm_Genlist
 */
EAPI Elm_Object_Item              *elm_genlist_decorated_item_get(const Evas_Object *obj);

/**
 * @brief Sets the reorder mode.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @remarks After turning on the reorder mode, longpress on a normal item triggers
 *          reordering of the item. You can move the item up and down. However, reordering
 *          does not work with group items.
 *
 * @param[in] obj The genlist object
 * @param[in] reorder_mode The reorder mode
 * (@c EINA_TRUE = on, @c EINA_FALSE = off)
 *
 * @ingroup Elm_Genlist
 */
EAPI void                          elm_genlist_reorder_mode_set(Evas_Object *obj, Eina_Bool reorder_mode);

/**
 * @brief Gets the reorder mode.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @param[in] obj The genlist object
 * @return The reorder mode
 *         (@c EINA_TRUE = on, @c EINA_FALSE = off)
 *
 * @ingroup Elm_Genlist
 */
EAPI Eina_Bool                     elm_genlist_reorder_mode_get(const Evas_Object *obj);

/**
 * @internal
 * @remarks Tizen no feature
 *
 * @brief Sets the genlist decorate mode.
 *
 * @details This sets the genlist decorate mode for all items.
 *
 * @param obj The genlist object
 * @param decorated The decorate mode status
 *                  (@c EINA_TRUE = decorate mode, @c EINA_FALSE = normal mode
 *
 * @ingroup Elm_Genlist
 */
EAPI void               elm_genlist_decorate_mode_set(Evas_Object *obj, Eina_Bool decorated);

/**
 * @internal
 * @remarks Tizen no feature
 *
 * @brief Gets the genlist decorate mode.
 *
 * @param obj The genlist object
 * @return The decorate mode status
 *         (@c EINA_TRUE = decorate mode, @c EINA_FALSE = normal mode
 *
 * @ingroup Elm_Genlist
 */
EAPI Eina_Bool          elm_genlist_decorate_mode_get(const Evas_Object *obj);

/**
 * @internal
 * @remarks Tizen no feature
 *
 * @brief Sets the genlist tree effect.
 *
 * @param obj The genlist object
 * @param enabled The tree effect status
 *                (@c EINA_TRUE = enabled, @c EINA_FALSE = disabled
 *
 * @ingroup Elm_Genlist
 */
EAPI void               elm_genlist_tree_effect_enabled_set(Evas_Object *obj, Eina_Bool enabled);

/**
 * @internal
 * @remarks Tizen no feature
 *
 * @brief Gets the genlist tree effect.
 *
 * @param obj The genlist object
 * @return The tree effect status
 *         (@c EINA_TRUE = enabled, @c EINA_FALSE = disabled
 *
 * @ingroup Elm_Genlist
 */
EAPI Eina_Bool          elm_genlist_tree_effect_enabled_get(const Evas_Object *obj);

/**
 * @brief Sets the genlist select mode.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @remarks elm_genlist_select_mode_set() changes the item select mode in the genlist widget.
 *          - ELM_OBJECT_SELECT_MODE_DEFAULT : Items call their selection @a func and
 *            callback on first getting selected. Any further clicks
 *            do nothing, unless you set the always select mode.
 *          - ELM_OBJECT_SELECT_MODE_ALWAYS :  This means that, even if selected,
 *            every click calls the selected callbacks.
 *          - ELM_OBJECT_SELECT_MODE_NONE : This turns off the ability to select items
 *            entirely and they neither appear selected nor call selected
 *            callback functions.
 *
 * @param[in] obj The genlist object
 * @param[in] mode The select mode
 *
 * @see elm_genlist_select_mode_get()
 *
 * @ingroup Elm_Genlist
 */
EAPI void elm_genlist_select_mode_set(Evas_Object *obj, Elm_Object_Select_Mode mode);

/**
 * @brief Gets the genlist select mode.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @param[in] obj The genlist object
 * @return The select mode
 *         (If getting the mode fails, it returns @c ELM_OBJECT_SELECT_MODE_MAX)
 *
 * @see elm_genlist_select_mode_set()
 *
 * @ingroup Elm_Genlist
 */
EAPI Elm_Object_Select_Mode elm_genlist_select_mode_get(const Evas_Object *obj);

/**
 * @brief Sets whether the genlist items should be highlighted when an item is selected.
 *
 * @details This turns on/off the highlight effect when an item is selected and
 *          it gets or does not get highlighted. The selected and clicked
 *          callback functions are still called.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @remarks Highlight is enabled by default.
 *
 * @param[in] obj The genlist object
 * @param[in] highlight If @c EINA_TRUE highlighting is enabled,
 *                  otherwise @c EINA_FALSE to disable it
 *
 * @see elm_genlist_highlight_mode_get().
 *
 * @ingroup Elm_Genlist
 */
EAPI void               elm_genlist_highlight_mode_set(Evas_Object *obj, Eina_Bool highlight);

/**
 * @brief Gets whether the genlist items should be highlighted when an item is selected.
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @param[in] obj The genlist object
 * @return @c EINA_TRUE indicates that items can be highlighted,
 *         otherwise @c EINA_FALSE indicates that they can't \n
 *         If @a obj is @c NULL, @c EINA_FALSE is returned.
 *
 * @see elm_genlist_highlight_mode_set()
 *
 * @ingroup Elm_Genlist
 */
EAPI Eina_Bool          elm_genlist_highlight_mode_get(const Evas_Object *obj);

/**
 * @brief Gets the nth item in a given genlist widget, placed at position @a nth, in
 *        its internal items list.
 *
 * @since 1.8
 *
 * @if MOBILE @since_tizen 2.3
 * @elseif WEARABLE @since_tizen 2.3.1
 * @endif
 *
 * @param[in] obj The genlist object
 * @param[in] nth The number of the item to grab (@c 0 being the first)
 *
 * @return The item stored in @a obj at position @a nth, otherwise @c NULL if there is
 *         no item with that index (and on errors)
 *
 * @ingroup Elm_Genlist
 */
EAPI Elm_Object_Item *
elm_genlist_nth_item_get(const Evas_Object *obj, unsigned int nth);

/**
 * @internal
 * @remarks Tizen only feature
 *
 * @brief Sets the genlist realization mode.
 *
 * @remarks By default, genlist disables the realization mode and genlists realize and
 *          unrealize some items when needed. If the realization mode is on,
 *          all items are realized when genlist is created and no items are unrealized.
 *          If this mode is on and the content size is changed, the item size
 *          changes accordingly. By default, genlist does not change item sizes
 *          eventhough they can be changed for performance reasons.
 *          This consumes more memory and decrease performance. So if the application
 *          appends many items, do not use the realization mode.
 *
 * @param obj The genlist object
 * @param mode The realization mode
 *             (@c EINA_TRUE = on, @c EINA_FALSE = off)
 *
 * @ingroup Elm_Genlist
 */
EAPI void
elm_genlist_realization_mode_set(Evas_Object *obj, Eina_Bool mode);

/**
 * @internal
 * @remarks Tizen only feature
 *
 * @brief Gets the genlist realization mode.
 *
 * @param obj The genlist object
 * @return The realization mode
 *         (@c EINA_TRUE = on, @c EINA_FALSE = off)
 *
 * @ingroup Elm_Genlist
 */
EAPI Eina_Bool
elm_genlist_realization_mode_get(Evas_Object *obj);

/**
 * @internal
 * @remarks Tizen only feature
 *
 * @brief Starts reordering for a specific item. it moves by move event.
 *
 * @param item The genlist item object
 *
 * @ingroup Elm_Genlist
 */
EAPI void
elm_genlist_item_reorder_start(Elm_Object_Item *item);

/**
 * @internal
 * @remarks Tizen only feature
 *
 * @brief Stop reordering and relocate the item at touch released position.
 * @param item The genlist item object
 *
 * @ingroup Elm_Genlist
 */
EAPI void
elm_genlist_item_reorder_stop(Elm_Object_Item *item);

/**
 * @internal
 * @remarks Tizen only feature
 *
 * This API is not supported yet.
 * @param key The filter key
 *
 * @ingroup Elm_Genlist
 */
EAPI void
elm_genlist_filter_set(Evas_Object *obj, void * key);

/**
 * @internal
 *
 * @ingroup Elm_Genlist
 */
EAPI Eina_Iterator *
elm_genlist_filter_iterator_new(Evas_Object *obj);

/**
 * @defgroup Elm_Genlist_Item Genlist Item
 * @ingroup Elm_Genlist
 *
 * @{
 */
#include "elm_genlist_item.eo.legacy.h"
/**
 * @}
 */

#include "elm_genlist.eo.legacy.h"
