/**
 * event-manager.js
 * Copyright (c) 2015 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Contact:
 * DongHee Yang <donghee.yang@samsung.com>
 * Sungmin Kim <sm.art.kim@samsung.com>
 * Jiil Hyoun <jiil.hyoun@samsung.com>
 * Jonghwan Park <iwin100.park@samsung.com>
 * Kitae Kim <kt920.kim@samsung.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Contributors:
 * - S-Core Co., Ltd
**/

var _ = require('underscore');
var dibs = require('./dibs.js');


module.exports.create = function (parent) {
    var triggerPlugins = {};
    var extensions = dibs.plugin.getExtensions('dibs.base.trigger');

    _.each(extensions, function (ext) {
        triggerPlugins[ext.triggerType] = ext.module;
    });

    return new TriggerManager(parent, triggerPlugins);
};


function TriggerManager(parent, plugins) {
    // var self = this;
    var parentServer = parent;
    var triggerPlugins = plugins;
    var triggerList = [];


    /*
     * @function addTriggerAction
     * @param {object} trigger - trigger information
     *  { id: {string}, name: {string}, type: {string}, serverType: {string}, data: {object}, status: {string}}
     * @param {returnCallback} callback -
     */
    this.addTriggerAction = function (trigger, callback) {
        if (_.isEmpty(triggerPlugins)) {
            dibs.log.info('no trigger plugins');
            return callback(null);
        }

        if (dibs.thisServer.id !== parentServer.id) {
            // remote
            dibs.getServer(parentServer.id).__addTriggerAction(trigger, callback);
        } else {
            // local
            if (triggerPlugins[trigger.type] !== undefined) {
                triggerPlugins[trigger.type].addTrigger(trigger, function (err, listenerObj) {
                    if (!err) {
                        // contains eventListener object in case
                        if (listenerObj) {
                            dibs.log.info(trigger.type + ' uses EventListener type');
                            trigger.listenerObj = listenerObj;
                        }

                        triggerList.push(trigger);
                    }

                    callback(err);
                });
            } else {
                callback(new Error('undefined trigger type ' + trigger.type));
            }
        }
    };


    /*
     * @function removeTriggerAction
     * @param {string} triggerId - trigger id
     * @param {returnCallback} callback -
     */
    this.removeTriggerAction = function (triggerId, callback) {
        if (dibs.thisServer.id !== parentServer.id) {
            // remote
            dibs.getServer(parentServer.id).__removeTriggerAction(triggerId, callback);
        } else {
            // local
            var index = _.findIndex(triggerList, { id: triggerId });
            if (index === -1) {
                callback(new Error('there is no ' + triggerId));
            } else {
                var trigger = triggerList[index];

                if (triggerPlugins[trigger.type] !== undefined) {
                    triggerPlugins[trigger.type].removeTrigger(trigger, function (err) {
                        if (!err) {
                            triggerList.splice(index, 1);
                        }
                        callback(err);
                    });
                } else {
                    callback(new Error('undefined trigger type ' + trigger.type));
                }
            }
        }
    };


    // search triggers
    this.searchTriggerActions = function (condition, callback) {
        dibs.log.info('no implementation in searchTriggerActions');
        callback(null);
    };


    // modify trigger
    this.modifyTriggerAction = function (trigger, callback) {
        dibs.log.info('no implementation in modifyTriggerAction');
        callback(null);
    };

}
