/**
 * db-connection.js
 * Copyright (c) 2016 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Contact:
 * Sungmin Kim <sm.art.kim@samsung.com>
 * Jonghwan Park <iwin100.park@samsung.com>
 * Kitae Kim <kt920.kim@samsung.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Contributors:
 * - S-Core Co., Ltd
**/

var async = require('async');
var genericPool = require('generic-pool');
var mysql = require('mysql');
var mongodb = require('mongodb');

var dibs = require('../core/dibs.js');


module.exports.createDBConnectionPool = function () {
    var pool = genericPool.Pool({
        name: 'web-mysql',
        max: 50,
        idleTimeoutMillis: 60000,
        create: function (callback) {
            var datamgr = dibs.getServersByType('datamgr')[0];

            var config = {
                host: datamgr.config.get('db_host'),
                port: datamgr.config.get('db_port'),
                user: datamgr.config.get('db_user'),
                password: datamgr.config.get('db_password'),
                database: datamgr.config.get('db_name')
            };

            // createConnection(config, callback);
            dibs.thisServer.log.info('Create MySQL connections!!');
            var client = mysql.createConnection(config);
            async.retry({
                times: 10,
                internal: 1000
            },
            function (cb) {
                client.connect(function (err1) {
                    if (err1) {
                        dibs.thisServer.log.error(err1);
                    }
                    cb(err1, null);
                });
            },
            function (err, results) {
                if (err) {
                    dibs.thisServer.log.error(err);
                }
                callback(err, client);
            });
        },
        destroy: function (client) {
            dibs.thisServer.log.info('Close MySQL connections!!');
            client.end();
        },
        log: function (str, level) {
            if (level === 'error') {
                dibs.thisServer.log.error(str);
            }
        }
    });

    pool.getConnection = function (callback) {
        pool.acquire(function (err, conn) {
            callback(err, conn);
        });
    };

    pool.releaseConnection = function (client) {
        pool.release(client);
    };

    return pool;
};


module.exports.createMongoDBConnectionPool = function (callback) {
    var host = dibs.thisServer.config.get('remote_log_db_host');
    var name = dibs.thisServer.config.get('remote_log_db_name');
    var port = dibs.thisServer.config.get('remote_log_db_port') || 27017;
    var mongoUrl = 'mongodb://' + host + ':' + port + '/' + name;

    dibs.log.info('Try connecting client to mongodb server');
    mongodb.MongoClient.connect(mongoUrl, function (err, db) {
        if (err) {
            dibs.log.error('Connecting mongodb failure');
            dibs.log.error(err);
        }

        callback(err, db);
    });
};
