/**
 * base-remote.js
 * Copyright (c) 2000 - 2015 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Contact:
 * DongHee Yang <donghee.yang@samsung.com>
 * Sungmin Kim <sm.art.kim@samsung.com>
 * Jiil Hyoun <jiil.hyoun@samsung.com>
 * Jonghwan Park <iwin100.park@samsung.com>
 * Kitae Kim <kt920.kim@samsung.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Contributors:
 * - S-Core Co., Ltd
**/

var dibs = require('../../core/dibs.js');
var DError = require('../../core/exception.js');


/**
 * @module core/base-remote
 */

/**
 * @function __updateServerInformation
 * @param {string} servers - servers
 * @param {module:lib/utils.callback_error} callback - callback
 * @memberOf module:core/base-remote
 */
// callback( err )
module.exports.__updateServerInformation = function (servers, callback) {

    var master = servers.filter(function (e) {
        return e.type === 'master';
    })[0];

    // check master's address
    var allowMasterAddr = dibs.thisServer.startOptions.masterAddr;
    if (master.host !== allowMasterAddr.split(':')[0] ||
        master.port !== parseInt(allowMasterAddr.split(':')[1], 10)) {

        callback(new DError('RPC010'));
    } else {
        // update server infos
        dibs.thisServer.updateServerInformations(servers);
        callback(null);
    }
};

/**
 * @callback callback_err_status
 * @param {error|undefined} error
 * @param {module:core/base-server.status} status
 * @memberOf moduel:core/base-remote
 */

/**
 * @function __getStatus
 * @param {module:core/base-server.callback_err_status} callback - callback
 * @memberOf module:core/base-remote
 */

// callback( err, status )
module.exports.__getStatus = function (cb) {
    dibs.thisServer.getStatus(function (err, status) {
        if (err) {
            console.error('cannot get server status in __getStatus()');
        }
        cb(err, status);
    });
};

/**
 * @callback callback_err_environment
 * @param {error|undefined} error
 * @param {module:core/base-server.server.environment} environment
 * @memberOf moduel:core/base-remote
 */

/**
 * @function __getServerEnvironment
 * @param {module:core/base-server.callback_err_status} callback - callback
 * @memberOf module:core/base-remote
 */
module.exports.__getServerEnvironment = function (cb) {
    cb(null, dibs.thisServer.environments);
};


/**
 * @function __getServerConfiguration
 * @param {module:core/base-server.callback_err_status} callback - callback
 * @memberOf module:core/base-remote
 */
module.exports.__getServerConfiguration = function (callback) {
    if (dibs.thisServer.status === 'RUNNING') {
        callback(null, dibs.thisServer.config.getConfigData());
    } else {
        callback(null, null);
    }
};

/**
 * @function terminateServer
 * @param {module:lib/utils.callback_error} callback - callback
 * @memberOf module:core/base-remote
 */


// callback( err )
module.exports.terminateServer = function (callback) {
    // check master's address
    var master = dibs.getAllServers().filter(function (e) {
        return e.type === 'master';
    })[0];
    var allowMasterAddr = dibs.thisServer.startOptions.masterAddr;
    if (master && allowMasterAddr &&
        (master.host !== allowMasterAddr.split(':')[0] ||
        master.port !== parseInt(allowMasterAddr.split(':')[1], 10))) {

        callback(new DError('RPC010'));
    } else {
        dibs.thisServer.terminate({}, callback);
    }
};


module.exports.__addEventListener = function (evtListener, callback) {
    dibs.thisServer.addEventListener2(evtListener, callback);
};


module.exports.__removeEventListener = function (evtListener, callback) {
    dibs.thisServer.removeEventListener(evtListener, callback);
};


module.exports.__executeEventListeners = function (evtObj, callback) {
    dibs.thisServer.executeEventListeners(evtObj, callback);
};


module.exports.__getEventListeners = function (event, callback) {
    dibs.thisServer.getEventListeners(event, callback);
};


module.exports.__addTriggerAction = function (trigger, callback) {
    dibs.thisServer.addTriggerAction(trigger, callback);
};

module.exports.__removeTriggerAction = function (triggerId, callback) {
    dibs.thisServer.removeTriggerAction(triggerId, callback);
};

module.exports.__searchTriggerActions = function (condition, callback) {
    dibs.thisServer.searchTriggerActions(condition, callback);
};

module.exports.__modifyTriggerAction = function (trigger, callback) {
    dibs.thisServer.modifyTriggerAction(trigger, callback);
};


var prevEvtObj = null;
module.exports.testAddEventListener = function (sid, evtName, evtOpts, callback) {
    var testFunc = function (evtObj, cb) {
        prevEvtObj = evtObj;
        cb(null);
    };
    if (dibs.thisServer.id === sid) {
        dibs.thisServer.addEventListener(evtName, evtOpts, testFunc, callback);
    } else {
        dibs.getServer(sid).addEventListener(evtName, evtOpts, testFunc, callback);
    }
};


module.exports.testRemoveEventListener = function (sid, listener, callback) {
    if (dibs.thisServer.id === sid) {
        dibs.thisServer.removeEventListener(listener, callback);
    } else {
        dibs.getServer(sid).removeEventListener(listener, callback);
    }
};


module.exports.testEmitEvent = function (evtObj, callback) {
    dibs.thisServer.emitEvent(evtObj);
    callback(null);
};


module.exports.testResetPrevEventObject = function (callback) {
    prevEvtObj = null;
    callback(null);
};


module.exports.testGetPrevEventObject = function (callback) {
    callback(null, prevEvtObj);
};


module.exports.testGetEventListeners = function (event, callback) {
    dibs.thisServer.getEventListeners(event, callback);
};
