/**
 * migrate.js
 * Copyright (c) 2000 - 2015 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Contact:
 * DongHee Yang <donghee.yang@samsung.com>
 * Sungmin Kim <sm.art.kim@samsung.com>
 * Jiil Hyoun <jiil.hyoun@samsung.com>
 * Jonghwan Park <iwin100.park@samsung.com>
 * Kitae Kim <kt920.kim@samsung.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Contributors:
 * - S-Core Co., Ltd
**/

var fs = require('fs');
var path = require('path');

/*
 * steps structure  list of
 *
 * sql string as string like  "CREATE TABLE xxxx"
 *
 * or object like
 *
 * {sql: "string" , genObj : function(before row){...} }
 *
 * sql string
 * genObj fucntion parameta row is that before query result row
 */
var migrateSteps = [
    // 1
    [
        'CREATE TABLE DB_VERSION (version INT)',
        'INSERT INTO DB_VERSION SET version = 1',
        'CREATE TABLE images ( id INT PRIMARY KEY NOT NULL AUTO_INCREMENT, ' +
        'name VARCHAR(32) NOT NULL, ' +
        'size INT(32) NOT NULL, ' +
        'contents BLOB NOT NULL )',
        //notification tables
        'CREATE TABLE notification_group ( id INT PRIMARY KEY NOT NULL AUTO_INCREMENT, ' +
        'description VARCHAR(256) NULL) ',
        'CREATE TABLE notifications ( id INT PRIMARY KEY NOT NULL AUTO_INCREMENT, ' +
        'notification_group_id INT NOT NULL, ' +
        'event VARCHAR(64) NOT NULL, ' +
        'method VARCHAR(64) NOT NULL, ' +
        'target_type VARCHAR(64) NOT NULL, ' +
        'target_user_id INT NULL, ' +
        'target_group_id INT NULL, ' +
        'target_data VARCHAR(2048) NULL, ' +
        'INDEX fk_notification_notification_group1_idx (notification_group_id ASC), ' +
        'CONSTRAINT fk_notification_notification_group1 FOREIGN KEY (notification_group_id) REFERENCES notification_group (id)) ',
        // user tables
        'CREATE TABLE users ( id INT PRIMARY KEY NOT NULL AUTO_INCREMENT, ' +
        'name VARCHAR(256) NOT NULL, ' +
        'email VARCHAR(256) NOT NULL, ' +
        'password_hash VARCHAR(256) NOT NULL, ' +
        'password_salt VARCHAR(256) NOT NULL, ' +
        'image_id INT NULL, ' +
        'status VARCHAR(32) NOT NULL DEFAULT \'OPEN\' COMMENT \'OPEN/CLOSE\' ) DEFAULT CHARSET=utf8 ',
        'CREATE TABLE user_info ( id INT PRIMARY KEY NOT NULL AUTO_INCREMENT, ' +
        'user_id INT NOT NULL, ' +
        'property VARCHAR(256) NULL, ' +
        'value VARCHAR(1024) NULL, ' +
        'CONSTRAINT fk_user_info_users1 FOREIGN KEY (user_id) REFERENCES users (id) )',
        'CREATE TABLE user_board ( id INT PRIMARY KEY NOT NULL AUTO_INCREMENT, ' +
        'user_id INT NOT NULL, time VARCHAR(64) NOT NULL, ' +
        'name VARCHAR(256) NOT NULL, ' +
        'content VARCHAR(1024) NULL, ' +
        'INDEX fk_user_board_users1_idx (user_id ASC),' +
        'CONSTRAINT fk_user_board_users1 FOREIGN KEY (user_id) REFERENCES users (id) )',
        {
            sql: 'INSERT INTO images SET ?',
            genObj: function (row) {
                var ipath = path.join(__dirname, 'images', 'admin.jpg');
                return {
                    id: 1,
                    name: 'admin.jpg',
                    size: fs.statSync(ipath).size,
                    contents: fs.readFileSync(ipath)
                };
            }
        },
        // insert admin user
        {
            sql: 'INSERT INTO users SET ?',
            genObj: function (row) {
                return {
                    id: 1,
                    name: 'admin',
                    email: 'admin@user',
                    password_hash: '$2a$10$H.w3ssI9KfuvNEXXp1qxD.b3Wm8alJG.HXviUofe4nErDn.TKUAka',
                    password_salt: '$2a$10$H.w3ssI9KfuvNEXXp1qxD.',
                    image_id: row.insertId,
                    status: 'OPEN'
                };
            }
        },
        {
            sql: 'INSERT INTO images SET ?',
            genObj: function (row) {
                var ipath = path.join(__dirname, 'images', 'anonymous.jpg');
                return {
                    id: 2,
                    name: 'anonymous.jpg',
                    size: fs.statSync(ipath).size,
                    contents: fs.readFileSync(ipath)
                };
            }
        },
        // group tables
        'CREATE TABLE groups ( id INT PRIMARY KEY NOT NULL AUTO_INCREMENT, ' +
        'name VARCHAR(256) NOT NULL, ' +
        'parent_id INT NULL, ' +
        'type VARCHAR(32) NULL, ' +
        'status VARCHAR(32) NOT NULL DEFAULT \'OPEN\' COMMENT \'OPEN/CLOSE\', ' +
        'description VARCHAR(256) NULL, ' +
        'CONSTRAINT fk_groups_groups1 FOREIGN KEY (parent_id) REFERENCES groups (id) ) DEFAULT CHARSET=utf8',
        'CREATE TABLE group_info (id INT PRIMARY KEY NOT NULL AUTO_INCREMENT, ' +
        'group_id INT NOT NULL, ' +
        'property VARCHAR(256) NULL, ' +
        'value VARCHAR(1024) NULL, ' +
        'CONSTRAINT fk_group_info_groups1 FOREIGN KEY (group_id ) REFERENCES groups (id) )',
        // insert administrator group
        'INSERT INTO groups ( id' +
        ', name' +
        ', parent_id' +
        ', type' +
        ', status' +
        ', description)' +
        'VALUES (1 ' +
        ', \'administrator\'' +
        ', NULL' +
        ', NULL' +
        ', \'OPEN\'' +
        ', \'This is a administrator group.\')',
        // environment tables
        'CREATE TABLE environments ( id VARCHAR(128) PRIMARY KEY NOT NULL, ' +
        'name VARCHAR(256) NULL, ' +
        'description VARCHAR(1024) NULL )',
        'CREATE TABLE environment_info ( id INT PRIMARY KEY NOT NULL AUTO_INCREMENT, ' +
        'environment_id VARCHAR(128) NOT NULL, ' +
        'property VARCHAR(256) NULL, ' +
        'value VARCHAR(1024) NULL, ' +
        'CONSTRAINT fk_environment_info_environments1 FOREIGN KEY (environment_id) REFERENCES environments (id) )',
        // distribution tables
        'CREATE TABLE distributions ( id INT PRIMARY KEY NOT NULL AUTO_INCREMENT, ' +
        'user_id INT NOT NULL, ' +
        'name VARCHAR(256) NOT NULL, ' +
        'type VARCHAR(256) NOT NULL, ' +
        'status VARCHAR(32) NOT NULL DEFAULT \'OPEN\' COMMENT \'OPEN/CLOSE\', ' +
        'access VARCHAR(8) NOT NULL DEFAULT \'PRIVATE\' COMMENT \'PUBLIC/PRIVATE\', ' +
        'sync VARCHAR(64) NULL COMMENT \'Synchronize type\', ' +
        'description VARCHAR(1024) NULL, ' +
        'notification_group_id INT, ' +
        'CONSTRAINT fk_distributions_users1 FOREIGN KEY (user_id) REFERENCES users (id), ' +
        'CONSTRAINT fk_distributions_notification_group1 FOREIGN KEY (notification_group_id) REFERENCES notification_group (id) )',
        'CREATE TABLE distribution_info ( id INT PRIMARY KEY NOT NULL AUTO_INCREMENT, ' +
        'distribution_id INT NOT NULL, ' +
        'property VARCHAR(256) NOT NULL, ' +
        'value VARCHAR(1024) NOT NULL, ' +
        'type VARCHAR(32) NULL, ' +
        'CONSTRAINT fk_distribution_info_distributions1 FOREIGN KEY (distribution_id) REFERENCES distributions (id) )',
        // project tables
        'CREATE TABLE projects ( id INT PRIMARY KEY NOT NULL AUTO_INCREMENT, ' +
        'distribution_id INT NOT NULL, ' +
        'name VARCHAR(256) NOT NULL, ' +
        'type VARCHAR(256) NOT NULL, ' +
        'status VARCHAR(32) NOT NULL DEFAULT \'OPEN\' COMMENT \'OPEN/CLOSE\',' +
        'notification_group_id INT, ' +
        'CONSTRAINT fk_project_distributions1 FOREIGN KEY (distribution_id) REFERENCES distributions (id), ' +
        'CONSTRAINT fk_projects_notification_group1 FOREIGN KEY (notification_group_id) REFERENCES notification_group (id) )',
        'CREATE TABLE project_info ( id INT PRIMARY KEY NOT NULL AUTO_INCREMENT, ' +
        'project_id INT NOT NULL, ' +
        'property VARCHAR(256) NOT NULL, ' +
        'value VARCHAR(10000) NOT NULL, ' +
        'type VARCHAR(32) NULL, ' +
        'CONSTRAINT fk_project_info_projects1 FOREIGN KEY (project_id) REFERENCES projects (id) )',
        // job tables
        'CREATE TABLE jobs ( id INT PRIMARY KEY NOT NULL AUTO_INCREMENT, ' +
        'user_email VARCHAR(256) NOT NULL, ' +
        'distribution_name VARCHAR(256) NOT NULL, ' +
        'distribution_type VARCHAR(256) NOT NULL, ' +
        'project_name VARCHAR(256), ' +
        'project_type VARCHAR(256) NOT NULL, ' +
        'environment_name VARCHAR(256), ' +
        'parent_id INT NULL, ' +
        'type VARCHAR(32) NULL, ' +
        'status VARCHAR(32) NOT NULL, ' +
        'start_time VARCHAR(64) NULL, ' +
        'end_time VARCHAR(64) NULL, ' +
        'error_code VARCHAR(32) NULL, ' +
        'snapshot_name VARCHAR(64) NULL, ' +
        'description VARCHAR(256) NULL, ' +
        'INDEX fk_jobs_jobs1_idx (parent_id ASC), ' +
        'CONSTRAINT fk_jobs_jobs1 FOREIGN KEY (parent_id) REFERENCES jobs (id) )',
        'CREATE TABLE job_info ( id INT PRIMARY KEY NOT NULL AUTO_INCREMENT, ' +
        'job_id INT NOT NULL, ' +
        'property VARCHAR(256) NOT NULL, ' +
        'value MEDIUMTEXT NULL, ' +
        'type VARCHAR(32) NULL, ' +
        'INDEX fk_job_info_jobs1_idx (job_id ASC), ' +
        'CONSTRAINT fk_job_info_jobs1 FOREIGN KEY (job_id) REFERENCES jobs (id) )',
        'CREATE TABLE job_board ( id INT PRIMARY KEY NOT NULL AUTO_INCREMENT, ' +
        'job_id INT NOT NULL, ' +
        'time VARCHAR(64) NOT NULL, ' +
        'type VARCHAR(64) NOT NULL, ' +
        'name VARCHAR(64) NOT NULL, ' +
        'INDEX fk_job_board_jobs1_idx (job_id ASC), ' +
        'CONSTRAINT fk_job_board_jobs1 FOREIGN KEY (job_id) REFERENCES jobs (id) )',
        // relation tables
        'CREATE TABLE user_group ( id INT PRIMARY KEY NOT NULL AUTO_INCREMENT, ' +
        'user_id INT NOT NULL, ' +
        'group_id INT NOT NULL COMMENT \'sub-group id\', ' +
        'CONSTRAINT fk_user_group_users1 FOREIGN KEY (user_id) REFERENCES users (id), ' +
        'CONSTRAINT fk_user_group_groups1 FOREIGN KEY (group_id) REFERENCES groups (id) )',
        'CREATE TABLE group_project ( id INT PRIMARY KEY NOT NULL AUTO_INCREMENT, ' +
        'group_id INT NOT NULL, ' +
        'project_id INT NOT NULL, ' +
        'CONSTRAINT fk_group_project_projects1 FOREIGN KEY (project_id) REFERENCES projects (id), ' +
        'CONSTRAINT fk_group_project_groups1 FOREIGN KEY (group_id) REFERENCES groups (id) )',
        'CREATE TABLE project_env ( id INT PRIMARY KEY NOT NULL AUTO_INCREMENT, ' +
        'project_id INT NOT NULL, ' +
        'environment_id VARCHAR(128) NOT NULL, ' +
        'CONSTRAINT fk_project_env_projects1 FOREIGN KEY (project_id) REFERENCES projects (id), ' +
        'CONSTRAINT fk_project_env_environments1 FOREIGN KEY (environment_id) REFERENCES environments (id) )',
        // insert admin's group relation
        'INSERT INTO user_group ( user_id, group_id) VALUES (1, 1)'
    ],
    // 2
    ['ALTER TABLE jobs ADD init_server_id VARCHAR(128)',
        'ALTER TABLE jobs ADD exec_server_id VARCHAR(128)'],
    // 3 CHANGE COLUMNS user_id -> group_id
    ['ALTER TABLE distributions ADD COLUMN group_id INT(11) NOT NULL AFTER id',
        'UPDATE distributions SET group_id = (SELECT group_id FROM user_group WHERE user_group.user_id = distributions.user_id ORDER BY id LIMIT 1)',
        'ALTER TABLE distributions ADD CONSTRAINT fk_distributions_groups1 FOREIGN KEY (group_id) REFERENCES groups (id)',
        'ALTER TABLE distributions DROP FOREIGN KEY fk_distributions_users1',
        'ALTER TABLE distributions DROP COLUMN user_id'],
    // 4 Add sync-manager@user to administrator group
    [
        // insert sync-manager user
        {
            sql: 'INSERT INTO users SET ?',
            genObj: function (row) {
                return {
                    name: 'sync-mananger',
                    email: 'sync-manager@user',
                    password_hash: '$2a$10$H.w3ssI9KfuvNEXXp1qxD.b3Wm8alJG.HXviUofe4nErDn.TKUAka',
                    password_salt: '$2a$10$H.w3ssI9KfuvNEXXp1qxD.',
                    image_id: 1,
                    status: 'OPEN'
                };
            }
        },
        // add sync-manager to administrator group
        'INSERT INTO user_group (user_id, group_id) SELECT id, 1 FROM users WHERE email = \'sync-manager@user\''
    ],
    // 5 Change data type (project_info.value)
    ['ALTER TABLE project_info CHANGE COLUMN value value MEDIUMTEXT NOT NULL '
    ],
    // 6 Add approval tables
    ['CREATE TABLE approvals ( id INT PRIMARY KEY NOT NULL AUTO_INCREMENT, ' +
    'user_id INT NOT NULL, ' +
    'approval_type VARCHAR(32) NOT NULL, ' +
    'subject VARCHAR(255) NOT NULL, ' +
    'contents VARCHAR(2048) NOT NULL, ' +
    'summary VARCHAR(2048) NULL, ' +
    'submission_date VARCHAR(32) NOT NULL, ' +
    'submission_status VARCHAR(32) NOT NULL, ' +
    'CONSTRAINT fk_approvals_users1 FOREIGN KEY (user_id) REFERENCES users (id) )',
        'CREATE TABLE approval_route ( id INT PRIMARY KEY NOT NULL AUTO_INCREMENT, ' +
        'approval_id INT NOT NULL, ' +
        'user_id INT NOT NULL, ' +
        'approval_order INT NOT NULL, ' +
        'approval_type VARCHAR(32) NOT NULL, ' +
        'approval_status VARCHAR(32) NOT NULL, ' +
        'approval_date VARCHAR(32), ' +
        'approval_message VARCHAR(1024), ' +
        'CONSTRAINT fk_approval_route_approvals1 FOREIGN KEY (approval_id) REFERENCES approvals (id), ' +
        'CONSTRAINT fk_approval_route_users1 FOREIGN KEY (user_id) REFERENCES users (id) )'
    ],
    // 7 Add 'approval_id' column to jobs table
    ['ALTER TABLE jobs ADD approval_id INT NULL',
        'ALTER TABLE jobs ADD CONSTRAINT fk_jobs_approvals1 FOREIGN KEY (approval_id) REFERENCES approvals (id)'
    ],
    // 8 Add 'status_info' column to jobs table
    ['ALTER TABLE jobs ADD status_info VARCHAR(1024) NULL'],
    // 9 Character set change to utf8
    ['ALTER TABLE approval_route convert to charset utf8',
        'ALTER TABLE approvals convert to charset utf8',
        'ALTER TABLE job_info convert to charset utf8'
    ],
    // 10 Add triggers and project_trigger tables
    ['CREATE TABLE triggers ( id INT PRIMARY KEY NOT NULL AUTO_INCREMENT, ' +
    'name VARCHAR(256) NOT NULL, ' +
    'type VARCHAR(256) NOT NULL, ' +
    'status VARCHAR(32) NOT NULL, ' +
    'server_type VARCHAR(256) NOT NULL, ' +
    'data MEDIUMTEXT NOT NULL, ' +
    'data_type VARCHAR(256) NOT NULL ) ',
        'CREATE TABLE project_trigger ( id INT PRIMARY KEY NOT NULL AUTO_INCREMENT, ' +
        'project_id INT NOT NULL, ' +
        'trigger_id INT NOT NULL, ' +
        'CONSTRAINT fk_project_trigger_projects1 FOREIGN KEY (project_id) REFERENCES projects (id), ' +
        'CONSTRAINT fk_project_trigger_triggers1 FOREIGN KEY (trigger_id) REFERENCES triggers (id) )'
    ],
    // 11 Add servers, server_info tables
    ['CREATE TABLE servers ( id VARCHAR(125) PRIMARY KEY NOT NULL, ' +
    'type VARCHAR(256) NOT NULL, ' +
    'parent_id VARCHAR(256) NULL, ' +
    'os_type VARCHAR(32) NOT NULL, ' +
    'host VARCHAR(32) NOT NULL, ' +
    'port VARCHAR(32) NULL, ' +
    'domain_name VARCHAR(256) NULL, ' +
    'description VARCHAR(2048) NULL ) ',
        'CREATE TABLE server_info ( id INT PRIMARY KEY NOT NULL AUTO_INCREMENT, ' +
        'server_id VARCHAR(125) NOT NULL, ' +
        'property VARCHAR(256) NOT NULL, ' +
        'value MEDIUMTEXT NOT NULL, ' +
        'type VARCHAR(32) NOT NULL, ' +
        'CONSTRAINT fk_server_info_servers FOREIGN KEY (server_id) REFERENCES servers (id) )'
    ],
    // 12 Add toolchain information for tizen windows project
    ['drop procedure if exists add_toolchain',
        'CREATE PROCEDURE add_toolchain() BEGIN ' +
        'select count(t1.id) into @cnt from ' +
            '(select a.* from projects a ' +
            'where a.type = \'Tizen-Source\') t1 ' +
            'left join ' +
            '(select a.* from projects a, project_info b ' +
            'where a.type = \'Tizen-Source\' ' +
            'and a.id = b.project_id ' +
            'and b.property = \'PRE_BUILD\') t2 ' +
        'on t1.id = t2.id ' +
        'where t2.id is null; ' +
        'SET @iteration = 0; ' +
        'START TRANSACTION; ' +
            'WHILE @iteration < @cnt DO ' +
            'SET @iteration = @iteration + 1;' +
            'insert into project_info (project_id, property, value, type) ' +
                'VALUES ((select t1.id from ' +
                    '(select a.* from projects a where a.type = \'Tizen-Source\') t1 ' +
                    'left join ' +
                    '(select a.* from projects a, project_info b ' +
                        'where a.type = \'Tizen-Source\' ' +
                        'and a.id = b.project_id ' +
                        'and b.property = \'PRE_BUILD\') t2 ' +
                    'on t1.id = t2.id ' +
                    'where t2.id is null ' +
                    'limit 1) ' +
                ', \'PRE_BUILD\', \'[{"eventType":"installWinToolchain","property":{"ORDER":1,"PKG_NAME":"mingw32-msys-1.0-dev"}}]\', \'JSON\'); ' +
            'END WHILE; ' +
        'COMMIT; ' +
        'END',
        'CALL add_toolchain()'],
    // 13 add USE_GIT_SUBMODULES property
    ['drop procedure if exists add_git_submodules',
        'CREATE PROCEDURE add_git_submodules() BEGIN ' +
            'select count(t1.id) into @cnt from ' +
            '(select a.* from projects a ' +
                'where a.type = \'Tizen-Source\') t1 ' +
            'left join ' +
            '(select a.* from projects a, project_info b ' +
                'where a.type = \'Tizen-Source\' ' +
                'and a.id = b.project_id ' +
                'and b.property = \'USE_GIT_SUBMODULES\') t2 ' +
            'on t1.id = t2.id ' +
            'where t2.id is null; ' +
            'SET @iteration = 0; ' +
            'START TRANSACTION; ' +
            'WHILE @iteration < @cnt DO ' +
                'SET @iteration = @iteration + 1;' +
                'insert into project_info (project_id, property, value, type) ' +
                'VALUES ((select t1.id from ' +
                    '(select a.* from projects a where a.type = \'Tizen-Source\') t1 ' +
                    'left join ' +
                    '(select a.* from projects a, project_info b ' +
                        'where a.type = \'Tizen-Source\' ' +
                        'and a.id = b.project_id ' +
                        'and b.property = \'USE_GIT_SUBMODULES\') t2 ' +
                    'on t1.id = t2.id ' +
                    'where t2.id is null ' +
                    'limit 1) ' +
                ', \'USE_GIT_SUBMODULES\', \'false\', \'BOOLEAN\'); ' +
            'END WHILE; ' +
            'COMMIT; ' +
            'END ',
        'CALL add_git_submodules()'],
    // 14 add snapshots, snapshot_info, artifacts, artifact_info tables
    [
        'CREATE TABLE snapshots ( id INT PRIMARY KEY NOT NULL AUTO_INCREMENT, ' +
            'name VARCHAR(256) NOT NULL, ' +
            'distribution_name VARCHAR(256) NOT NULL, ' +
            'type VARCHAR(32) NOT NULL, ' +
            'time VARCHAR(64) NULL, ' +
            'status VARCHAR(10) NOT NULL, ' +
            'attribute VARCHAR(32) NOT NULL, ' +
            'path VARCHAR(1024) NULL, ' +
            'description VARCHAR(1024) NULL) ',
        'CREATE TABLE snapshot_info ( id INT PRIMARY KEY NOT NULL AUTO_INCREMENT, ' +
            'snapshot_id INT NOT NULL, ' +
            'property VARCHAR(256) NOT NULL, ' +
            'value MEDIUMTEXT NULL, ' +
            'type VARCHAR(16) NOT NULL, ' +
            'CONSTRAINT fk_snapshot_info_snapshots FOREIGN KEY (snapshot_id) REFERENCES snapshots (id) ) ',
        'CREATE TABLE artifacts ( id INT PRIMARY KEY NOT NULL AUTO_INCREMENT, ' +
            'name VARCHAR(256) NOT NULL, ' +
            'distribution_name VARCHAR(256) NOT NULL, ' +
            'type VARCHAR(256) NOT NULL, ' +
            'version VARCHAR(256) NULL, ' +
            'environment VARCHAR(32) NULL, ' +
            'attribute VARCHAR(32) NULL, ' +
            'status VARCHAR(10) NULL, ' +
            'file VARCHAR(256) NULL, ' +
            'path VARCHAR(1024) NULL, ' +
            'size BIGINT NULL, ' +
            'checksum VARCHAR(256) NULL, ' +
            'description MEDIUMTEXT NULL) ',
        'CREATE TABLE artifact_info ( id INT PRIMARY KEY NOT NULL AUTO_INCREMENT, ' +
            'artifact_id INT NOT NULL, ' +
            'property VARCHAR(256) NOT NULL, ' +
            'value MEDIUMTEXT NULL, ' +
            'type VARCHAR(16) NOT NULL, ' +
            'CONSTRAINT fk_artifact_info_artifacts FOREIGN KEY (artifact_id) REFERENCES artifacts (id) )',
        'CREATE TABLE snapshot_artifact ( id INT PRIMARY KEY NOT NULL AUTO_INCREMENT, ' +
            'snapshot_id INT NOT NULL, ' +
            'artifact_id INT NOT NULL, ' +
            'CONSTRAINT fk_snapshot_artifact_snapshots1 FOREIGN KEY (snapshot_id) REFERENCES snapshots (id), ' +
            'CONSTRAINT fk_snapshot_artifact_artifacts1 FOREIGN KEY (artifact_id) REFERENCES artifacts (id) )'
    ],
    // 15 Character set change to utf8
    [
        'ALTER TABLE artifact_info convert to charset utf8',
        'ALTER TABLE snapshot_info convert to charset utf8'
    ],
    // 16 Change data type (project_info.value)
    [
        'ALTER TABLE distribution_info CHANGE COLUMN value value MEDIUMTEXT NOT NULL '
    ]
];

module.exports = migrateSteps;
