/**
 * ts-push.js
 * Copyright (c)2000 - 2017 Samsung Electronics Co., Ltd. All rights reserved.
 *
 * Contact:
 * Sungmin Kim <sm.art.kim@samsung.com>
 * Jonghwan Park <iwin100.park@samsung.com>
 * Kitae Kim <kt920.kim@samsung.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Contributors:
 * - S-Core Co., Ltd
**/

'use strict';

var async = require('async');
var path = require('path');

var TsRepo = require('../../org.tizen.ts.cli.repository/ts-repo.js');
var Monitor = require('../../../lib/monitor.js');

module.exports.push = function (argv, log) {
    var localFsRepo = argv.lr;
    var isForce = argv.force;
    var pkgList = argv.package.split(',');

    if (!localFsRepo) {
        log.error('Required --local-repo option!');
        log.error('Usage: ts-cli push --package <package file path|list> --local-repo <repository path> [--force]');
        process.exit(-1);
    } else if (!pkgList) {
        log.error('Required --package option!');
        log.error('Usage: ts-cli push --package <package file path|list> --local-repo <repository path> [--force]');
        process.exit(-1);
    } else {
        // convert localFsRepo into absolute path in case of relative
        localFsRepo = path.resolve(localFsRepo);
    }

    var destDistName = path.basename(localFsRepo);

    var destDist = null;
    var destSnapshot = null;

    log.info('Tizen Studio Push Process - Start');

    var monitor = new Monitor({
        onProgress: function (info, cb) {
            if (info.logType) {
                log[info.logType](info.log);
            }
            cb(null);
        }
    });

    var destRepo = new TsRepo({
        location: localFsRepo
    });

    async.waterfall([
        function (cb) {
            log.info(' # - Open repository: \'' + localFsRepo + '\'');
            destRepo.load(function (err) {
                if (err) {
                    log.error(err);
                }
                cb(err);
            });
        },
        function (cb) {
            destDist = destRepo.distributions[destDistName];

            if (!destDist) {
                return cb(new Error(' # - Failed to get \'' + destDist + '\''));
            }

            destSnapshot = destDist.latestSnapshot;
            if (!destSnapshot) {
                log.warn(' # - Any snapshot does not exist in distribution \'' + destDist.name + '\'');
            }
            cb(null);
        },
        function (cb) {
            log.info(' # - Registering packages into ' + destDist.name);
            destDist.registerPackages(pkgList, { force: isForce }, monitor, cb);
        }
    ],
    function (err) {
        if (err) {
            log.error('Package-Push - Failure!');
            log.error(err);
            process.exit(-1);
        }
        log.info('Package-Push - Success');
        process.exit(0);
    });
};
